// -------------------------------- argParser.c -------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "argParser.h"

// Convert a CapyArgNbVal into a string and print it (used when printing help)
// Input:
//   nbVal: the CapyArgNbVal
static void PrintNbVal(CapyArgNbVal const nbVal) {

  // Switch on the CapyArgNbVal and print it
  switch(nbVal) {
    case capyArg_variableNbVal:
      printf("an undefined number of values");
      break;
    case capyArg_noVal:
      printf("no values");
      break;
    default:
      printf("%d value%s", nbVal, (nbVal > 1 ? "s" : ""));
  }
}

// Get the argument definition for a given label or shortcut
// Input:
//   arg: the label/shortcut
// Output:
//   Return the definition if the label/shortcut was found, else NULL
static CapyArg* GetDef(
  CapyArgParser const* const that,
           char const* const arg) {
#if BUILD_MODE == 0
  assert(that != NULL);
  assert(arg != NULL);
#endif

  // Loop on the arguments in the definition
  loop(i, that->nbArgs) {

    // If the string matches the label or shortcut,
    // return the definition
    if(
      (that->args[i].lbl != NULL && strcmp(arg, that->args[i].lbl) == 0) ||
      (
        that->args[i].shortLbl != NULL &&
        strcmp(arg, that->args[i].shortLbl) == 0
      )
    ) {
      return that->args + i;
    }
  }

  // If we reach here, the string wasn't found, return NULL
  return NULL;
}

// Print the help message describing the argument definition of the parser
static void Help(void) {
  methodOf(CapyArgParser);

  // Create the string decorator to print the help
  CapyStrDecorator decorator = CapyStrDecoratorCreate();
  $(&decorator, activate)(capyStrDecoratorBold);

  // Get the definition for the program (identified by argv[0])
  CapyArg* mainDef = GetDef(that, that->argv[0]);

  // If the definition for the program exists
  if(mainDef != NULL) {

    // Print the number of expected values and the help message for
    // the program
    $(&decorator, fprintf)(stdout,
      "%s\n",
      mainDef->lbl);
    size_t lenTitle = strlen(that->argv[0]);
    loop(j, lenTitle) $(&decorator, fprintf)(stdout, "=");
    printf("\n");
    if(mainDef->nbVal > 0) {
      printf("Expecting ");
      PrintNbVal(mainDef->nbVal);
      println(".");
    }
    if(mainDef->help != NULL) println("%s", mainDef->help);
  }

  // Print the header for optional arguments
  println("\nOptions:\n--------\n");

  // Loop on the arguments in the definition
  loop(i, that->nbArgs) {

    // If it's not the argument definition for the program
    if(that->args + i != mainDef) {

      // Print the label, shortcut, number of expected values and the help
      // message for the optional argument
      printf("  ");
      if(that->args[i].shortLbl) {
        $(&decorator, fprintf)(stdout, "%s", that->args[i].shortLbl);
      }
      if(that->args[i].lbl && that->args[i].shortLbl) {
        $(&decorator, fprintf)(stdout, ", ");
      }
      if(that->args[i].lbl) {
        $(&decorator, fprintf)(stdout, "%s", that->args[i].lbl);
      }
      printf(":\n");
      if(that->args[i].nbVal) {
        printf("Expecting ");
        PrintNbVal(that->args[i].nbVal);
        println(".");
      }
      if(that->args[i].help) println("%s\n", that->args[i].help);
    }
  }

  // Free the decorator
  $(&decorator, destruct)();
}

// Get the number of expected orphan values for the program
// Output:
//   Return the number of values
static CapyArgNbVal GetNbValProgram(CapyArgParser const* const that) {
#if BUILD_MODE == 0
  assert(that != NULL);
#endif

  // Get the definition for the program (identified by argv[0])
  CapyArg* mainDef = GetDef(that, that->argv[0]);

  // If there is a definition for the program, return the expected
  // number of argument
  if(mainDef != NULL) return mainDef->nbVal;

  // Else, consider it means there should be no values
  else return capyArg_noVal;
}

// Parse the arguments from the command line
// Input:
//   that: that parser
// Exceptions:
//   May raise CapyExc_InvalidCLIArg, CapyExc_MallocFailed
static void Parse(CapyArgParser* const that) {
#if BUILD_MODE == 0
  assert(that != NULL);
#endif

  // Loop on the arguments, if the help is requested, print it and exit
  loop(iArg, that->argc) if(strcmp("--help", that->argv[iArg]) == 0) {
    $(that, help)();
    exit(EXIT_SUCCESS);
  }

  // Decorator to print error and warning messages
  CapyStrDecorator decoratorError = CapyStrDecoratorCreate();
  $(&decoratorError, setFgColor)(&capyColorRGBARed);
  $(&decoratorError, activate)(capyStrDecoratorFgColor);
  CapyStrDecorator decoratorWarning = CapyStrDecoratorCreate();
  $(&decoratorWarning, setFgColor)(&capyColorRGBAYellow);
  $(&decoratorWarning, activate)(capyStrDecoratorFgColor);

  // Flag to memorise if there was invalid arguments
  bool valid = true;

  // Loop on the arguments (skipping the program name)
  loop(iArg, that->argc) if(iArg > 0) {

    // Get the definition for the argument
    CapyArg* argDef = GetDef(that, that->argv[iArg]);

    // If that's not an optional argument,
    // it must be a value of the program, add it to the list
    if(argDef == NULL) $(that->vals, add)(that->argv[iArg]);

    // Else, it's an optional argument
    else {

      // Loop on the previous arguments, if the current argument has
      // already been set, print a warning message
      loop(jArg, iArg - 1) if(strcmp(that->argv[jArg], that->argv[iArg]) == 0) {
        if(that->silent == false) {
          $(&decoratorWarning, fprintf)(stdout,
            "Option %s used several times. Only the last one will be used\n",
            (argDef->lbl ? argDef->lbl : argDef->shortLbl));
        }
        valid = false;
        jArg = iArg;
      }

      // Ensure the list of values for this argument is empty (for the
      // case it's present several times)
      $(argDef->vals, flush)();

      // If the argument expects a variable number of values
      if(argDef->nbVal < 0) {

        // Loop on the values until we reach the end of the argument list
        // or the next option
        __typeof__(iArg) iVal = 1;
        while(
          (iArg + iVal) < that->argc &&
          GetDef(that, that->argv[iArg + iVal]) == NULL
        ) {

          // Add the value to the argument's values list
          $(argDef->vals, add)(that->argv[iArg + iVal++]);
        }

        // Skip the values in the main loop on arguments
        iArg += iVal - 1;

      // Else, if the argument expects a given number of values
      } else if(argDef->nbVal > 0) {

        // Variable to memorise the number of values really found,
        // a priori the expected one
        int nbFoundVal = argDef->nbVal;

        // Loop on the expected number of values
        loop(iVal, argDef->nbVal) {

          // If we have reached the end of arguments or the next option
          if(
            iArg + iVal + 1 >= that->argc ||
            GetDef(that, that->argv[iArg + iVal + 1]) != NULL
          ) {

            // Print an error message
            if(that->silent == false) {
              $(&decoratorError, fprintf)(stdout,
                "Missing values for option %s (expected %d, got %ld).\n",
                (argDef->lbl ? argDef->lbl : argDef->shortLbl),
                argDef->nbVal,
                iVal);
            }
            valid = false;

            // Skip the remaining values and update the number of value
            // really found
            nbFoundVal = iVal + 1;
            iVal = argDef->nbVal;

          // Else, that's the next expected value, add it to the list of
          // values for this argument
          } else $(argDef->vals, add)(that->argv[iArg + iVal + 1]);
        }

        // Skip the values in the main loop on arguments
        iArg += nbFoundVal;
      }
    }
  }

  // Variable to memorise the number of argument expected by the program
  CapyArgNbVal nbValProgram = GetNbValProgram(that);

  // If the program expected a given number of values
  if(
    nbValProgram != capyArg_variableNbVal &&
    $(that->vals, getSize)() != (size_t)nbValProgram
  ) {
    size_t nbExpected = $(that->vals, getSize)();
    if(that->silent == false) {
      $(&decoratorError, fprintf)(stdout,
        "Missing values or values in excess for %s (expected %d, got %ld).\n",
        that->argv[0],
        nbValProgram,
        nbExpected);
    }
    valid = false;
  }

  // Free the decorator
  $(&decoratorError, destruct)();
  $(&decoratorWarning, destruct)();

  // If there was invalid arguments, raise an exception
  if(valid == false) raiseExc(CapyExc_InvalidCLIArg);
}

// Get the number of values for a given argument (number given via the
// command line, not the number in the definition)
// Input:
//   arg: the argument, must be in the argument definitions, may be
//        the label or the shortcut
// Output:
//   Return the number of values
static size_t GetNbVal(char const* const arg) {
  methodOf(CapyArgParser);
#if BUILD_MODE == 0
  assert(arg != NULL);
#endif

  // Declare the returned value
  size_t val = 0;

  // Variable to memorise the list of value of the argument
  CapyListPtrChar* vals = NULL;

  // If the argument is the program, set the list of values to the list of
  // orphans values
  if(strcmp(arg, that->argv[0]) == 0) vals = that->vals;

  // Else, it must be an optional argument
  else {

    // Get the definition for the argument
    CapyArg* argDef = GetDef(that, arg);

    // If the definition exists, set the list of values to the argument's one
    if(argDef != NULL) vals = argDef->vals;
  }

  // If the list of values exists, set the result value
  if(vals != NULL) val = $(vals, getSize)();

  // Return the value
  return val;
}

// Check if an optional argument is used
// Input:
//   arg: the argument to check, must be in the argument definitions, may be
//        the label or the shortcut
// Output:
//   Return true if the argument is used, else false
static bool IsSet(char const* const arg) {
  methodOf(CapyArgParser);
#if BUILD_MODE == 0
  assert(arg != NULL);
#endif

  // Get the definition for the argument
  CapyArg* argDef = GetDef(that, arg);

  // If the definition doesn't exist, return false
  if(argDef == NULL) return false;

  // Else, the definition exists
  else {

    // Loop on the arguments
    loop(j, that->argc) {

      // If it's the searched argument, return true
      if(
        (argDef->lbl != NULL && strcmp(argDef->lbl, that->argv[j]) == 0) ||
        (
          argDef->shortLbl != NULL &&
          strcmp(argDef->shortLbl, that->argv[j]) == 0
        )
      ) {
        return true;
      }
    }

    // If we reached here, the argument wasn't found, return false
    return false;
  }
}

// Get a value decoded as int for an optional argument
// Input:
//    arg: the argument to check, must be in the argument definitions, may be
//         the label or the shortcut
//   iVal: the index of the value
// Output:
//   Return the requested value, or 0 if the value was missing
static int64_t GetAsInt(
  char const* const arg,
       size_t const iVal) {
  methodOf(CapyArgParser);
#if BUILD_MODE == 0
  assert(arg != NULL);
#endif

  // Declare the returned value
  int64_t val = 0;

  // Variable to memorise the list of value of the argument
  CapyListPtrChar* vals = NULL;

  // If the argument is the program, set the list of values to the list of
  // orphans values
  if(strcmp(arg, that->argv[0]) == 0) vals = that->vals;

  // Else, it must be an optional argument
  else {

    // Get the definition for the argument
    CapyArg* argDef = GetDef(that, arg);

    // If the definition exists, set the list of values to the argument's one
    if(argDef != NULL) vals = argDef->vals;
  }

  // If the list of values exists and the value index is valid,
  // set the result value
  if(vals != NULL && $(vals, getSize)() > iVal) {
    sscanf($(vals, get)(iVal), "%" SCNd64, &val);
  }

  // Return the value
  return val;
}

// Get a value decoded as double for an optional argument
// Input:
//    arg: the argument to check, must be in the argument definitions, may be
//         the label or the shortcut
//   iVal: the index of the value
// Output:
//   Return the requested value, or 0.0 if the value was missing
static double GetAsDouble(
  char const* const arg,
       size_t const iVal) {
  methodOf(CapyArgParser);
#if BUILD_MODE == 0
  assert(arg != NULL);
#endif

  // Declare the returned value
  double val = 0.0;

  // Variable to memorise the list of value of the argument
  CapyListPtrChar* vals = NULL;

  // If the argument is the program, set the list of values to the list of
  // orphans values
  if(strcmp(arg, that->argv[0]) == 0) vals = that->vals;

  // Else, it must be an optional argument
  else {

    // Get the definition for the argument
    CapyArg* argDef = GetDef(that, arg);

    // If the definition exists, set the list of values to the argument's one
    if(argDef != NULL) vals = argDef->vals;
  }

  // If the list of values exists and the value index is valid,
  // set the result value
  if(vals != NULL && $(vals, getSize)() > iVal) {
    sscanf($(vals, get)(iVal), "%lf", &val);
  }

  // Return the value
  return val;
}

// Get a value as a string for an optional argument
// Input:
//    arg: the argument to check, must be in the argument definitions, may be
//         the label or the shortcut
//   iVal: the index of the value
// Output:
//   Return the requested value, or NULL if the value was missing
static char* GetAsStr(
  char const* const arg,
       size_t const iVal) {
  methodOf(CapyArgParser);
#if BUILD_MODE == 0
  assert(arg != NULL);
#endif

  // Declare the returned value
  char* val = NULL;

  // Variable to memorise the list of value of the argument
  CapyListPtrChar* vals = NULL;

  // If the argument is the program, set the list of values to the list of
  // orphans values
  if(strcmp(arg, that->argv[0]) == 0) vals = that->vals;

  // Else, it must be an optional argument
  else {

    // Get the definition for the argument
    CapyArg* argDef = GetDef(that, arg);

    // If the definition exists, set the list of values to the argument's one
    if(argDef != NULL) vals = argDef->vals;
  }

  // If the list of values exists and the value index is valid,
  // set the result value
  if(vals != NULL && $(vals, getSize)() > iVal) val = $(vals, get)(iVal);

  // Return the value
  return val;
}

// Free the memory used by a CapyArgParser
static void Destruct(void) {
  methodOf(CapyArgParser);

  // Free the lists of orphan and arguments values
  CapyListPtrCharFree(&(that->vals));
  loop(i, that->nbArgs) CapyListPtrCharFree(&(that->args[i].vals));
}

// Create a CapyArgParser, check the argument definitions and parse the
// arguments from the command line.
// Input:
//     argc: the argc argument of the main() function
//     argv: the argv argument of the main() function
//     args: the argument definition list
//   nbArgs: the number of arguments in the definition list
//   silent: silent flag
// Output:
//   Return a CapyArgParser
// Exceptions:
//   May raise CapyExc_InvalidCLIArg, CapyExc_MallocFailed
CapyArgParser CapyArgParserCreate_(
       int const argc,
    char** const argv,
  CapyArg* const args,
    size_t const nbArgs,
      bool const silent) {
#if BUILD_MODE == 0
  assert(argv != NULL);
  assert(args != NULL);
#endif

  // Create the parser
  CapyArgParser parser = {
    .args = args,
    .nbArgs = nbArgs,
    .argc = argc,
    .argv = argv,
    .silent = silent,
    .help = Help,
    .destruct = Destruct,
    .getNbVal = GetNbVal,
    .isSet = IsSet,
    .getAsInt = GetAsInt,
    .getAsDouble = GetAsDouble,
    .getAsStr = GetAsStr,
  };

  // Initialise the list of orphan values and arguments values
  parser.vals = CapyListPtrCharAlloc();
  loop(i, nbArgs) parser.args[i].vals = CapyListPtrCharAlloc();

  // Decorator to display error messages
  CapyStrDecorator decorator = CapyStrDecoratorCreate();
  $(&decorator, setFgColor)(&capyColorRGBARed);
  $(&decorator, activate)(capyStrDecoratorFgColor);

  // Flag to memorise if the definition is valid
  bool valid = true;

  // Loop on the argument definitions
  loop(i, nbArgs) {

    // If this argument has no label and no shortcut
    if(
      (args[i].lbl == NULL && args[i].shortLbl == NULL) ||
      (
        args[i].lbl != NULL && args[i].shortLbl != NULL &&
        strlen(args[i].lbl) == 0 && strlen(args[i].shortLbl) > 0
      )
    ) {

      // It is not a valid definition, print an error message
      if(parser.silent == false) {
        $(&decorator, fprintf)(stderr,
          "The %d%s argument definition has no label and no shortcut.\n",
          i + 1,
          (i == 0 ? "st" : i == 1 ? "nd" : "th"));
      }
      valid = false;
    }

    // If this argument has '--help' as label or shortcut
    if(
      (args[i].lbl != NULL && strcmp("--help", args[i].lbl) == 0) ||
      (args[i].shortLbl != NULL && strcmp("--help", args[i].shortLbl) == 0)
    ) {

      // It is not a valid definition, print an error message
      if(parser.silent == false) {
        $(&decorator, fprintf)(stderr,
          "The '--help' label is reserved to display help.\n");
      }
      valid = false;
    }
  }

  // Loop on pair of argument definitions
  loop(i, nbArgs) loop(j, nbArgs) if(i < j) {

    // If both arguments have the same label
    if(
      args[i].lbl != NULL && args[j].lbl != NULL &&
      strlen(args[i].lbl) > 0 && strlen(args[j].lbl) > 0
    ) {
      if(strcmp(args[i].lbl, args[j].lbl) == 0) {

        // This is not a valid definition, print an error message
        if(parser.silent == false) {
          $(&decorator, fprintf)(stderr,
            "%s is defined several times\n", args[j].lbl);
        }
        valid = false;
      }
    }

    // If both arguments have the same shortcut
    if(
      args[i].shortLbl != NULL && args[j].shortLbl != NULL &&
      strlen(args[i].shortLbl) > 0 && strlen(args[j].shortLbl) > 0
    ) {
      if(strcmp(args[i].shortLbl, args[j].shortLbl) == 0) {

        // This is not a valid definition, print an error message
        if(parser.silent == false) {
          $(&decorator, fprintf)(stderr,
            "%s is defined several times\n", args[j].shortLbl);
        }
        valid = false;
      }
    }
  }

  // Free the decorator
  $(&decorator, destruct)();

  // If the definition is invalid, raise an exception
  if(valid == false) raiseExc(CapyExc_InvalidCLIArg);

  // Else, parse the arguments from the command line
  else Parse(&parser);

  // Return the parser
  return parser;
}

// Allocate memory for a CapyArgParser, create it, check the argument
// definitions and parse the arguments from the command line.
// Input:
//     argc: the argc argument of the main() function
//     argv: the argv argument of the main() function
//     args: the argument definition list
//   nbArgs: the number of arguments in the definition list
//   silent: silent flag
// Output:
//   Return a CapyArgParser
// Exceptions:
//   May raise CapyExc_InvalidCLIArg, CapyExc_MallocFailed
CapyArgParser* CapyArgParserAlloc_(
       int const argc,
    char** const argv,
  CapyArg* const args,
    size_t const nbArgs,
      bool const silent) {
  CapyArgParser* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyArgParserCreate_(argc, argv, args, nbArgs, silent);
  return that;
}

// Free the memory used by a CapyArgParser and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyArgParser to free
void CapyArgParserFree(CapyArgParser** const that) {
  if(that == NULL || *that == NULL) return;

  // Free the memory
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
