// -------------------------------- argParser.c -------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_ARGPARSER_H
#define CAPY_ARGPARSER_H
#include "externalHeaders.h"
#include "list.h"
#include "strDecorator.h"

// Description:
// CLI arguments parser class.

// Enumeration of the possible number of values for one argument in the
// definition list
typedef enum CapyArgNbVal {
  capyArg_variableNbVal = -1,
  capyArg_noVal,
} CapyArgNbVal;

// Structure to memorise info about one argument in the definition list
typedef struct CapyArg {

  // The label of the argument (e.g. '--argument')
  char const* lbl;

  // The short label of the argument (e.g. '-a')
  char const* shortLbl;

  // The number or values following this argument
  CapyArgNbVal nbVal;
  CapyPad(CapyArgNbVal, 0);

  // The description of this argument
  char const* help;

  // The list of values of this argument
  CapyListPtrChar* vals;
} CapyArg;

// ArgParser object
typedef struct CapyArgParser {

  // The arguments definition
  CapyArg* args;

  // The number of arguments in the definition
  size_t nbArgs;

  // The argc and argv argument of the main() function
  char* const* argv;
  int argc;
  CapyPad(int, 1);

  // Silent flag (default: false)
  bool silent;
  CapyPad(bool, 2);

  // The orphan values (values in argv not associated to an argument in the
  // arguments definition)
  CapyListPtrChar* vals;

  // Destructor
  void (*destruct)(void);

  // Print the help message describing the argument definition of the parser
  void (*help)(void);

  // Get the number of values for a given argument (number given via the
  // command line, not the number in the definition)
  // Input:
  //   arg: the argument, must be in the argument definitions, may be
  //        the label or the shortcut
  // Output:
  //   Return the number of values
  size_t (*getNbVal)(char const* const arg);

  // Check if an optional argument is used
  // Input:
  //   arg: the argument to check, must be in the argument definitions, may be
  //        the label or the shortcut
  // Output:
  //   Return true if the argument is used, else false
  bool (*isSet)(char const* const arg);

  // Get a value decoded as int for an optional argument
  // Input:
  //    arg: the argument to check, must be in the argument definitions, may be
  //         the label or the shortcut
  //   iVal: the index of the value
  // Output:
  //   Return the requested value, or 0 if the value was missing
  int64_t (*getAsInt)(
    char const* const arg,
         size_t const iVal);

  // Get a value decoded as double for an optional argument
  // Input:
  //    arg: the argument to check, must be in the argument definitions, may be
  //         the label or the shortcut
  //   iVal: the index of the value
  // Output:
  //   Return the requested value, or 0.0 if the value was missing
  double (*getAsDouble)(
    char const* const arg,
         size_t const iVal);

  // Get a value as a string for an optional argument
  // Input:
  //    arg: the argument to check, must be in the argument definitions, may be
  //         the label or the shortcut
  //   iVal: the index of the value
  // Output:
  //   Return the requested value, or NULL if the value was missing
  char* (*getAsStr)(
    char const* const arg,
         size_t const iVal);
} CapyArgParser;

// Create a CapyArgParser, check the argument definitions and parse the
// arguments from the command line.
// Input:
//     argc: the argc argument of the main() function
//     argv: the argv argument of the main() function
//     args: the argument definition list
//   nbArgs: the number of arguments in the definition list
//   silent: silent flag
// Output:
//   Return a CapyArgParser
// Exceptions:
//   May raise CapyExc_InvalidCLIArg, CapyExc_MallocFailed
CapyArgParser CapyArgParserCreate_(
       int const argc,
    char** const argv,
  CapyArg* const args,
    size_t const nbArgs,
      bool const silent);

// Helper macro to automatically deduce the number of argument in the
// definition list
#define CapyArgParserCreate(argc, argv, args) \
  CapyArgParserCreate_(argc, argv, args, sizeof(args)/sizeof(CapyArg), false)

// Helper macro to automatically deduce the number of argument in the
// definition list (silent version for unit tests)
#define CapyArgParserCreateSilent(argc, argv, args) \
  CapyArgParserCreate_(argc, argv, args, sizeof(args)/sizeof(CapyArg), true)

// Allocate memory for a CapyArgParser, create it, check the argument
// definitions and parse the arguments from the command line.
// Input:
//     argc: the argc argument of the main() function
//     argv: the argv argument of the main() function
//     args: the argument definition list
//   nbArgs: the number of arguments in the definition list
//   silent: silent flag
// Output:
//   Return a CapyArgParser
// Exceptions:
//   May raise CapyExc_InvalidCLIArg, CapyExc_MallocFailed
CapyArgParser* CapyArgParserAlloc_(
       int const argc,
    char** const argv,
  CapyArg* const args,
    size_t const nbArgs,
      bool const silent);

// Helper macro to automatically deduce the number of argument in the
// definition list
#define CapyArgParserAlloc(argc, argv, args) \
  CapyArgParserAlloc_(argc, argv, args, sizeof(args)/sizeof(CapyArg), false)

// Helper macro to automatically deduce the number of argument in the
// definition list (silent version for unit tests)
#define CapyArgParserAllocSilent(argc, argv, args) \
  CapyArgParserAlloc_(argc, argv, args, sizeof(args)/sizeof(CapyArg), true)

// Free the memory used by a CapyArgParser and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyArgParser to free
void CapyArgParserFree(CapyArgParser** const that);
#endif
