// --------------------------------- bezier.c ----------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "bezier.h"

// Definition of the memory pool for the iteration windows
CapyDefMemPool(CapyMemPoolBezierIteratorWindow, CapyBezierIteratorWindow)

// Create a new look up table for binomial values for a given order
// Input:
//   order: the order n in the binomial[n][k]
// Output:
//   Return the look up table binomial_{order}[k] (k<=order)
static double* CreateBinomialLUT(CapyBezierOrder_t const order) {

  // Static look up table
  static double binomial[maxof(CapyBezierOrder_t)][maxof(CapyBezierOrder_t)];

  // If the LUT hasn't been calculated yet
  static bool flag = false;
  if(flag == false) {

    // Initialise the look up table
    loop(n, maxof(CapyBezierOrder_t)){
      binomial[n][0] = 1.0;
      for(__typeof__(n) j = 1; j < n; ++j) {
        binomial[n][j] = binomial[n - 1][j - 1] + binomial[n - 1][j];
      }
      binomial[n][n] = 1.0;
    }

    // Mark the LUT as calculated
    flag = true;
  }

  // Return the relevant row in the look up table
  return binomial[order];
}

// Get the number of control points (equals to (order + 1) ^ dimIn)
// Input:
//   that: the CapyBezier
// Output:
//   Return the number of control points
static size_t GetNbCtrls(CapyBezier const* const that) {

  // Return the number of control points
  int64_t nb = CapyPowi(that->order + 1, that->dimIn);
  if(nb >= 0) return (size_t)nb;
  raiseExc(CapyExc_NumericalOverflow);
  return 0;
}

// Initialise a new look up table for the index of a given control
// points in a given input dimension. Ordered as lut[iCtrl][iDimIn].
// Input:
//   that: the CapyBezier to initialise
static void CreateIdxCtrlLUT(CapyBezier* const that) {

  // Allocate memory for the look up table
  __typeof__(that->idxCtrlLUT) lut = NULL;
  safeMalloc(lut, that->nbCtrls);
  if(!(that->nbCtrls)) return;
  loop(iCtrl, that->nbCtrls) safeMalloc(lut[iCtrl], that->dimIn);

  // Initialise the look up table
  size_t steps = 1;
  loop(iDim, that->dimIn) {
    __typeof__(**lut) idxDim = 0;
    __typeof(that->nbCtrls + 0) iCtrl = 0;
    while(iCtrl < that->nbCtrls) {
      loop(iStep, steps) lut[iCtrl + iStep][iDim] = idxDim;
      iCtrl += steps;
      idxDim = (__typeof__(idxDim))((idxDim + 1) % (that->order + 1));
    }
    steps *= (that->order + 1);
  }

  // Set the look up table
  that->idxCtrlLUT = lut;
}

// Get a reference to the control point of a given ID.
// Input:
//   id: the ID of the control point
// Output:
//   Return a pointer to the control point (CapyVec of dimension
//   dimOut)
static CapyVec const* GetCtrlById(size_t const id) {
  methodOf(CapyBezier);

  // Return the control
  return that->ctrls + id;
}

// Set the values of the control point of given ID.
// Input:
//    id: the ID of the control point
//   vals: the value to set
static void SetCtrlById(
         size_t const id,
  double const* const vals) {
  methodOf(CapyBezier);

  // Set the values
  loop(iDim, that->dimOut) that->ctrls[id].vals[iDim] = vals[iDim];
}

// Get a reference to the control point at a given index in each
// input dimension.
// Input:
//   idx: the indices of the control point
// Output:
//   Return a pointer to the control point (CapyVec of dimension
//   dimOut)
static CapyVec const* GetCtrl(size_t const* const idx) {
  methodOf(CapyBezier);

  // Variable to memorise the index of the control
  size_t idxCtrl = 0;

  // Calculate the index in the array of control points corresponding
  // to the index in each input dimension
  loop(iDim, that->dimIn) {
    idxCtrl = idxCtrl * (that->order + 1) + idx[that->dimIn - iDim - 1];
  }

  // Return the control
  return that->ctrls + idxCtrl;
}

// Set the values of the control point at a given index in each
// input dimension.
// Input:
//    idx: the indices of the control point
//   vals: the value to set
static void SetCtrl(
  size_t const* const idx,
  double const* const vals) {
  methodOf(CapyBezier);

  // Get the control point
  CapyVec* ctrl = (CapyVec*)$(that, getCtrl)(idx);

  // Set the values
  loop(iDim, that->dimOut) ctrl->vals[iDim] = vals[iDim];
}

// Evaluate the CapyBezier for a given input and update the result
// and the control points' weight in that->ctrlWeights. This function
// is not thread safe due to that->ctrlWeights.
// Input:
//    in: the input vector
//   out: array of double updated with the result of evaluation
static void Eval(
  double const* const in,
        double* const out) {
  methodOf(CapyBezier);

  // Reset the result values
  loop(iOut, that->dimOut) out[iOut] = 0.0;

  // Loop on the control points
  loop(iCtrl, that->nbCtrls) {

    // Update the control point's weight according to the input value
    // in each dimension
    that->ctrlWeights.vals[iCtrl] = 1.0;
    loop(iIn, that->dimIn) {
      size_t jCtrl = that->idxCtrlLUT[iCtrl][iIn];
      double weight =
        that->binomialLUT[jCtrl] *
        CapyPowf(1.0 - in[iIn], that->order - jCtrl) *
        CapyPowf(in[iIn], jCtrl);
      that->ctrlWeights.vals[iCtrl] *= weight;
    }

    // Update the result values with the weighted contribution of the
    // control point
    loop(iOut, that->dimOut) {
      out[iOut] +=
        that->ctrlWeights.vals[iCtrl] * that->ctrls[iCtrl].vals[iOut];
    }
  }
}

// Save the CapyBezier to a binary stream
// Input:
//   stream: the binary stream to save onto
// Exception:
//   May raise CapyExc_StreamWriteError
static void Save(FILE* const stream) {
  methodOf(CapyBezier);
  safeFWrite(stream, sizeof(that->order), &(that->order));
  safeFWrite(stream, sizeof(that->dimIn), &(that->dimIn));
  safeFWrite(stream, sizeof(that->dimOut), &(that->dimOut));
  loop(iCtrl, that->nbCtrls) {
    CapyVec const* ctrl = $(that, getCtrlById)(iCtrl);
    safeFWrite(stream, sizeof(double) * ctrl->dim, ctrl->vals);
  }
}

// Export the CapyBezier as a C function to a stream
// Input:
//   stream: the stream to save onto
//     name: the name of the function
// Output:
//   The exported function's interface is "static void <name>(
//   double const* const in, double* const out)".
// Exception:
//   May raise CapyExc_StreamWriteError.
static void ExportToCFun(
        FILE* const stream,
  char const* const name) {
  methodOf(CapyBezier);
  safeFPrintf(stream, "%s", "#ifndef size_t\n#include <stddef.h>\n#endif\n");
  safeFPrintf(stream, "static void %s(\n", name);
  safeFPrintf(stream, "%s", "  double const* const in,\n");
  safeFPrintf(stream, "%s", "  double* const out) {\n");
  safeFPrintf(
    stream, "  size_t idxCtrlLUT[%lu][%lu];\n", that->nbCtrls, that->dimIn
  );
  loop(iCtrl, that->nbCtrls) loop(iIn, that->dimIn) {
    safeFPrintf(
      stream,
      "  idxCtrlLUT[%lu][%lu] = %lu;\n",
      iCtrl, iIn, that->idxCtrlLUT[iCtrl][iIn]
    );
  }
  safeFPrintf(stream, "  double binomialLUT[%lu];\n", that->nbCtrls);
  loop(iCtrl, that->nbCtrls) {
    safeFPrintf(
      stream, "  binomialLUT[%lu] = %.12lf;\n", iCtrl, that->binomialLUT[iCtrl]
    );
  }
  safeFPrintf(
    stream, "  double ctrls[%lu][%lu];\n", that->nbCtrls, that->dimOut
  );
  loop(iCtrl, that->nbCtrls) loop(iOut, that->dimOut) {
    safeFPrintf(
      stream,
      "  ctrls[%lu][%lu] = %.12lf;\n",
      iCtrl, iOut, that->ctrls[iCtrl].vals[iOut]
    );
  }
  safeFPrintf(stream, "  double ctrlWeights[%lu];\n", that->nbCtrls);
  safeFPrintf(stream, "%s", "  for(size_t iOut = 0; ");
  safeFPrintf(stream, "iOut < %lu; ", that->dimOut);
  safeFPrintf(stream, "%s", "++iOut)\n    out[iOut] = 0.0;\n");
  safeFPrintf(stream, "%s", "  for(size_t iCtrl = 0; ");
  safeFPrintf(stream, "iCtrl < %lu;", that->nbCtrls);
  safeFPrintf(stream, "%s", "++iCtrl) {\n");
  safeFPrintf(stream, "%s", "    ctrlWeights[iCtrl] = 1.0;\n");
  safeFPrintf(stream, "%s", "    for(size_t iIn = 0; ");
  safeFPrintf(stream, "iIn < %lu; ", that->dimIn);
  safeFPrintf(stream, "%s", "++iIn) {\n");
  safeFPrintf(
    stream, "%s", "      size_t jCtrl = idxCtrlLUT[iCtrl][iIn];\n"
  );
  safeFPrintf(stream, "%s", "      double weight =\n");
  safeFPrintf(stream, "%s", "        binomialLUT[jCtrl] *\n");
  safeFPrintf(
    stream, "        pow(1.0 - in[iIn], %u - jCtrl) *\n", that->order
  );
  safeFPrintf(stream, "%s", "        pow(in[iIn], jCtrl);\n");
  safeFPrintf(stream, "%s", "      ctrlWeights[iCtrl] *= weight;\n");
  safeFPrintf(stream, "%s", "    }\n");
  safeFPrintf(stream, "%s", "    for(size_t iOut = 0; ");
  safeFPrintf(stream, "iOut < %lu; ", that->dimOut);
  safeFPrintf(stream, "%s", "++iOut) {\n");
  safeFPrintf(
    stream,
    "%s",
    "      out[iOut] += ctrlWeights[iCtrl] * ctrls[iCtrl][iOut];\n"
  );
  safeFPrintf(stream, "%s", "    }\n");
  safeFPrintf(stream, "%s", "  }\n");
  safeFPrintf(stream, "%s", "}\n\n");
}

// Clone the Bezier curve
// Output:
//   Return a new CapyBezier.
static CapyBezier* BezierClone(void) {
  methodOf(CapyBezier);
  CapyBezier* clone = CapyBezierAlloc(that->order, that->dimIn, that->dimOut);
  loop(iOut, that->dimOut) loop(iCtrl, that->nbCtrls) {
    clone->ctrls[iCtrl].vals[iOut] = that->ctrls[iCtrl].vals[iOut];
  }
  return clone;
}

// Free the memory used by a CapyBezier
static void Destruct(void) {
  methodOf(CapyBezier);
  $(that, destructCapyMathFun)();
  CapyVecFreeArr(&(that->ctrls), that->nbCtrls);
  CapyVecDestruct(&(that->ctrlWeights));
  loop(iCtrl, that->nbCtrls) free(that->idxCtrlLUT[iCtrl]);
  free(that->idxCtrlLUT);
}

// Translate the bezier
// Input:
//   u: translation vector, must have same dimension as that->dimOut
// Output:
//   'u' is added to the output dimensions values.
static void BezierTranslate(CapyVec const* const u) {
  methodOf(CapyBezier);
  loop(i, that->nbCtrls) CapyVecAdd(that->ctrls + i, u, that->ctrls + i);
}

// Scale the bezier
// Input:
//   u: translation vector, must have same dimension as that->dimOut
// Output:
//   'u' is multiplied (component wise) to the output dimensions values.
static void BezierScale(CapyVec const* const u) {
  methodOf(CapyBezier);
  loop(iCtrl, that->nbCtrls) loop(iOut, that->dimOut) {
    that->ctrls[iCtrl].vals[iOut] *= u->vals[iOut];
  }
}

// Rotate the bezier (must have dimOut==2)
// Input:
//   theta: rotation angle in radians
// Output:
//   Output dimension values are rotated CW by 'theta' relative to the
//   origin.
static void BezierRotate2D(double const theta) {
  methodOf(CapyBezier);
  loop(iCtrl, that->nbCtrls) {
    double u[2];
    u[0] =
      cos(theta) * that->ctrls[iCtrl].vals[0] -
      sin(theta) * that->ctrls[iCtrl].vals[1];
    u[1] =
      sin(theta) * that->ctrls[iCtrl].vals[0] +
      cos(theta) * that->ctrls[iCtrl].vals[1];
    loop(i, 2) that->ctrls[iCtrl].vals[i] = u[i];
  }
}

// Create a CapyBezier
// Input:
//    order: the order of the Bezier object
//    dimIn: number of inputs
//   dimOut: number of outputs
// Output:
//   Return a CapyBezier
CapyBezier CapyBezierCreate(
  CapyBezierOrder_t const order,
             size_t const dimIn,
             size_t const dimOut) {

  // Create the Bezier
  CapyBezier that = {
    .order = order,
    .getCtrlById = GetCtrlById,
    .setCtrlById = SetCtrlById,
    .getCtrl = GetCtrl,
    .setCtrl = SetCtrl,
    .save = Save,
    .exportToCFun = ExportToCFun,
    .clone = BezierClone,
    .translate = BezierTranslate,
    .scale = BezierScale,
    .rotate2D = BezierRotate2D,
  };
  CapyInherits(that, CapyMathFun, (dimIn, dimOut));
  that.destruct = Destruct;
  that.eval = Eval;
  that.nbCtrls = GetNbCtrls(&that);
  that.ctrls = CapyVecAllocArr(dimOut, that.nbCtrls);
  that.ctrlWeights = CapyVecCreate(that.nbCtrls);
  that.binomialLUT = CreateBinomialLUT(that.order);
  CreateIdxCtrlLUT(&that);
  return that;
}

// Allocate memory for a new CapyBezier and create it
// Input:
//    order: the order of the Bezier object
//    dimIn: number of inputs
//   dimOut: number of outputs
// Output:
//   Return a CapyBezier
// Exception:
//   May raise CapyExc_MallocFailed.
CapyBezier* CapyBezierAlloc(
  CapyBezierOrder_t const order,
             size_t const dimIn,
             size_t const dimOut) {

  // Allocate memory and create the new CapyBezier
  CapyBezier* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyBezierCreate(order, dimIn, dimOut);

  // Return the new CapyBezier
  return that;
}

// Allocate memory for a new CapyBezier and load it from a binary stream
// Input:
//   stream: the binary stream to load the Bezier from
// Output:
//   Return a CapyBezier
// Exception:
//   May raise CapyExc_MallocFailed, CapyExc_StreamReadError.
CapyBezier* CapyBezierLoad(FILE* const stream) {
  CapyBezierOrder_t order = 0;
  safeFRead(stream, sizeof(order), &order);
  size_t dimIn = 0;
  safeFRead(stream, sizeof(dimIn), &dimIn);
  size_t dimOut = 0;
  safeFRead(stream, sizeof(dimOut), &dimOut);
  CapyBezier* that = CapyBezierAlloc(order, dimIn, dimOut);
  CapyVec vec = CapyVecCreate(that->dimOut);
  loop(iCtrl, that->nbCtrls) {
    safeFRead(stream, sizeof(double) * that->dimOut, vec.vals);
    $(that, setCtrlById)(iCtrl, vec.vals);
  }
  CapyVecDestruct(&vec);
  return that;
}

// Free the memory used by a CapyBezier* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyBezier to free
void CapyBezierFree(CapyBezier** const that) {
  if(*that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Check if the current CapyBezierIteratorWindow of an iterator is closed
// Input:
//   that: the iterator
// Output:
//   Return true if the window is closed, false else
static bool IsIteratorWindowClosed(CapyBezierIterator const* const that) {
  double dist = 0.0;
  loop(i, that->bezier->dimOut) {
    double x = that->windows->pos[0].out[i] - that->windows->pos[1].out[i];
    dist += x * x;
  }
  dist = sqrt(dist);
  return (
    that->windows->pos[1].in[0] - that->windows->pos[0].in[0] < 1.0 &&
    dist < that->epsilon
  );
}

// Split the current iteration window
// Input:
//   that: the iterator
// Output:
//   The current iteration window is split until it is closed
static void SplitIteratorWindow(CapyBezierIterator* const that) {
  while(IsIteratorWindowClosed(that) == false) {
    CapyBezierIteratorWindow* window = $(&(that->memPool), alloc)(NULL);
    window->pos[0].in[0] = that->windows->pos[0].in[0];
    window->pos[1].in[0] =
      0.5 * (that->windows->pos[0].in[0] + that->windows->pos[1].in[0]);
    $(that->bezier, eval)(window->pos[0].in, window->pos[0].out);
    $(that->bezier, eval)(window->pos[1].in, window->pos[1].out);
    that->windows->pos[0].in[0] = window->pos[1].in[0];
    loop(i, that->bezier->dimOut) {
      that->windows->pos[0].out[i] = window->pos[1].out[i];
    }
    window->next = that->windows;
    that->windows = window;
  }
}

// Reset the iterator
// Output:
//   Return the first pixel of the iteration
static CapyBezierPosition* IterReset(void) {
  methodOf(CapyBezierIterator);

  // Reset the index
  that->idx = 0;

  // Reset the iterator
  while(that->windows != NULL) {
    CapyBezierIteratorWindow* ptr = that->windows->next;
    $(&(that->memPool), free)(&(that->windows));
    that->windows = ptr;
  }
  that->windows = $(&(that->memPool), alloc)(NULL);
  that->windows->next = NULL;
  loop(i, 2) {
    that->windows->pos[i].in[0] = i;
    $(that->bezier, eval)(that->windows->pos[i].in, that->windows->pos[i].out);
  }
  SplitIteratorWindow(that);

  // Return the current position
  return that->windows->pos;
}

// Move the iterator to the next position
// Output:
//   Return the next position of the iteration
static CapyBezierPosition* IterNext(void) {
  methodOf(CapyBezierIterator);

  // Increment the index
  ++(that->idx);

  // Step the iterator
  if(that->windows->next) {
    CapyBezierIteratorWindow* ptr = that->windows;
    that->windows = ptr->next;
    $(&(that->memPool), free)(&ptr);
  }
  SplitIteratorWindow(that);

  // Return the current position
  return that->windows->pos;
}

// Check if the iterator is on a valid element
// Output:
//   Return true if the iterator is on a valid element, else false
static bool IterIsActive(void) {
  methodOf(CapyBezierIterator);
  return (IsIteratorWindowClosed(that) == false || that->windows->next != NULL);
}

// Get the current position of the iteration
// Output:
//   Return the current position
static CapyBezierPosition* IterGet(void) {
  methodOf(CapyBezierIterator);
  return that->windows->pos;
}

// Free the memory used by an iterator.
static void IteratorDestruct(void) {
  methodOf(CapyBezierIterator);
  $(&(that->memPool), destruct)();
}

// Create an iterator on a CapyBezier
// Input:
//   bezier: the bezier on which to iterate
//   type: the type of iterator
// Output:
//   Return the iterator
CapyBezierIterator CapyBezierIteratorCreate(CapyBezier const* const bezier) {

  // Ensure the input idmension of the bezier is equal to 1
  if(bezier->dimIn != 1) raiseExc(CapyExc_UndefinedExecution);

  // Create the new iterator
  CapyBezierIterator that = {
    .idx = 0,
    .bezier = bezier,
    .windows = NULL,
    .epsilon = 0.1,
    .memPool =
      CapyMemPoolBezierIteratorWindowCreate(sizeof(CapyBezierIteratorWindow)),
    .destruct = IteratorDestruct,
    .reset = IterReset,
    .next = IterNext,
    .isActive = IterIsActive,
    .get = IterGet,
  };
  (void)$(&that, reset)();

  // Return the new iterator
  return that;
}

// Allocate memory and create an iterator on a CapyBezier
// Input:
//   bezier: the bezier on which to iterate
//   type: the type of iterator
// Output:
//   Return the iterator
CapyBezierIterator* CapyBezierIteratorAlloc(CapyBezier const* const bezier) {
#if BUILD_MODE == 0
  assert(bezier != NULL);
#endif

  // Declare the new iterator
  CapyBezierIterator* that = NULL;

  // Allocate memory
  safeMalloc(that, 1);
  if(!that) return NULL;

  // Create the iterator
  *that = CapyBezierIteratorCreate(bezier);

  // Return the new iterator
  return that;
}

// Free the memory used by a pointer to an iterator and reset '*that' to NULL
// Input:
//   that: a pointer to the iterator to free
void CapyBezierIteratorFree(CapyBezierIterator** const that) {

  // If the iterator is already freed, nothing to do
  if(that == NULL || *that == NULL) return;

  // Destruct the iterator
  $(*that, destruct)();

  // Free memory
  free(*that);
  *that = NULL;
}

// Free the memory used by a CapyBezierSpline
static void DestructSpline(void) {
  methodOf(CapyBezierSpline);
  $(that, destructCapyMathFun)();
  loop(i, that->nbCurve) CapyBezierFree(that->curves + i);
  free(that->curves);
  free(that->nbSegment);
}

// Get the Bezier curve at a given location in the tensor
// Input:
//   idx: the indices of the segment
// Output:
//   Return a reference to the segment.
static CapyBezier* GetCurve(size_t const* const idx) {
  methodOf(CapyBezierSpline);

  // Variable to memorise the index of the curve
  size_t idxCurve = 0;

  // Calculate the index in the array of segments corresponding
  // to the index in each input dimension
  loop(iDim, that->dimIn) {
    __typeof__(that->dimIn) jDim = that->dimIn - iDim - 1;
    idxCurve += idx[jDim];
    if(jDim > 0) idxCurve *= that->nbSegment[jDim];
  }

  // Return the curve
  return that->curves[idxCurve];
}

// Set the Bezier curve at a given location in the tensor
// Input:
//     idx: the indices of the segment
//   curve: the Bezier curve
// Output:
//   The reference to the curve is updated. If there was a previous
//   refererence, it is destructed prior to update.
static void SetCurve(
  size_t const* const idx,
    CapyBezier* const curve) {
  methodOf(CapyBezierSpline);

  // Variable to memorise the index of the curve
  size_t idxCurve = 0;

  // Calculate the index in the array of segments corresponding
  // to the index in each input dimension
  loop(iDim, that->dimIn) {
    __typeof__(that->dimIn) jDim = that->dimIn - iDim - 1;
    idxCurve += idx[jDim];
    if(jDim > 0) idxCurve *= that->nbSegment[jDim];
  }

  // Destruct the current curve
  CapyBezierFree(that->curves + idxCurve);

  // Update the curve
  that->curves[idxCurve] = curve;
}

// Evaluation function for a CapyBezierSpline
// Input:
//    in: input values
//   out: output values
// Output:
//   'out' is updated
static void EvalSpline(
  double const* const in,
        double* const out) {
  methodOf(CapyBezierSpline);

  // Get the index of the appropriate curve for the input values
  // and its corrected input
  size_t idx[that->dimIn];
  double curveIn[that->dimIn];
  loop(i, that->dimIn) {
    double x = 0.0;
    curveIn[i] = modf(in[i], &x);
    idx[i] = (size_t)x;
    if(idx[i] == that->nbSegment[i]) {
      idx[i] -= 1;
      curveIn[i] = 1.0;
    }
  }

  // Get the appropriate curve
  CapyBezier* curve = $(that, getCurve)(idx);

  // Calculate the output
  $(curve, eval)(curveIn, out);
}

// Clone the Bezier spline
// Output:
//   Return a new CapyBezierSpline.
static CapyBezierSpline* SplineClone(void) {
  methodOf(CapyBezierSpline);
  CapyBezierSpline* clone =
    CapyBezierSplineAlloc(that->nbSegment, that->dimIn, that->dimOut);
  loop(i, that->nbCurve) {
    CapyBezierFree(clone->curves + i);
    clone->curves[i] = $(that->curves[i], clone)();
  }
  return clone;
}

// Translate the spline
// Input:
//   u: translation vector, must have same dimension as that->dimOut
// Output:
//   'u' is added to the output dimensions values.
static void SplineTranslate(CapyVec const* const u) {
  methodOf(CapyBezierSpline);
  loop(i, that->nbCurve) $(that->curves[i], translate)(u);
}

// Scale the spline
// Input:
//   u: translation vector, must have same dimension as that->dimOut
// Output:
//   'u' is multiplied (component wise) to the output dimensions values.
static void SplineScale(CapyVec const* const u) {
  methodOf(CapyBezierSpline);
  loop(i, that->nbCurve) $(that->curves[i], scale)(u);
}

// Rotate the spline (must have dimOut==2)
// Input:
//   theta: rotation angle in radians
// Output:
//   Output dimension values are rotated CW by 'theta' relative to the
//   origin.
static void SplineRotate2D(double const theta) {
  methodOf(CapyBezierSpline);
  loop(i, that->nbCurve) $(that->curves[i], rotate2D)(theta);
}

// Create a CapyBezierSpline
// Input:
//  nbSegment: array of dimIn integer, number of segments per input dimension
//      dimIn: number of inputs
//     dimOut: number of outputs
// Output:
//   Return a CapyBezierSpline
CapyBezierSpline CapyBezierSplineCreate(
  size_t const* const nbSegment,
         size_t const dimIn,
         size_t const dimOut) {
  CapyBezierSpline that = {
    .nbSegment = NULL,
    .curves = NULL,
    .nbCurve = 1,
    .getCurve = GetCurve,
    .setCurve = SetCurve,
    .clone = SplineClone,
    .translate = SplineTranslate,
    .scale = SplineScale,
    .rotate2D = SplineRotate2D,
  };
  CapyInherits(that, CapyMathFun, (dimIn, dimOut));
  that.destruct = DestructSpline;
  that.eval = EvalSpline;
  safeMalloc(that.nbSegment, dimIn);
  loop(i, dimIn) {
    that.nbSegment[i] = nbSegment[i];
    that.domains[i].min = 0.0;
    if((size_t)((double)(nbSegment[i])) != nbSegment[i]) {
      raiseExc(CapyExc_FloatingPointImprecision);
    }
    that.domains[i].max = (double)(nbSegment[i]);
    that.nbCurve *= nbSegment[i];
  }
  safeMalloc(that.curves, that.nbCurve);
  loop(i, that.nbCurve) that.curves[i] = CapyBezierAlloc(0, dimIn, dimOut);
  return that;
}

// Allocate memory for a new CapyBezierSpline and create it
// Input:
//  nbSegment: array of dimIn integer, number of segments per input dimension
//    dimIn: number of inputs
//   dimOut: number of outputs
// Output:
//   Return a CapyBezierSpline
// Exception:
//   May raise CapyExc_MallocFailed.
CapyBezierSpline* CapyBezierSplineAlloc(
  size_t const* const nbSegment,
         size_t const dimIn,
         size_t const dimOut) {

  // Allocate memory and create the new CapyBezierSpline
  CapyBezierSpline* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyBezierSplineCreate(nbSegment, dimIn, dimOut);

  // Return the new CapyBezierSpline
  return that;
}

// Free the memory used by a CapyBezierSpline* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyBezierSpline to free
void CapyBezierSplineFree(CapyBezierSpline** const that) {
  if(*that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Declare a list of Bezier spline
CapyDefList(CapyListBezierSpline, CapyBezierSpline*)
