// --------------------------------- bezier.h ----------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_BEZIER_H
#define CAPY_BEZIER_H
#include "externalHeaders.h"
#include "cext.h"
#include "capymath.h"
#include "mathfun.h"
#include "array.h"
#include "memoryPool.h"
#include "list.h"

// Maximum order and dimension for a CapyBezier
#define CAPY_BEZIER_MAX_DIMIN 254
#define CAPY_BEZIER_MAX_DIMOUT 254

// Description:
// Bezier surface class.

// Type of the order of a CapyBezier
typedef uint8_t CapyBezierOrder_t;

// Bezier curve object
typedef struct CapyBezier CapyBezier;
struct CapyBezier {

  // Inherits CapyMathFun
  struct CapyMathFunDef;

  // Order
  CapyBezierOrder_t order;
  CapyPad(CapyBezierOrder_t, order);

  // Number of control points
  size_t nbCtrls;

  // Look up table for the binomial values
  double* binomialLUT;

  // Look up table for the index in a given dimension of a given control
  // points
  size_t** idxCtrlLUT;

  // Control points. Ordered as follow, for example for a 2d->1d bilinear
  // ctrls[0] : in={0,0}
  // ctrls[1] : in={1,0}
  // ctrls[2] : in={0,1}
  // ctrls[3] : in={1,1}
  CapyVec* ctrls;

  // Vector to memorise the control points' weight during evaluation.
  // Ordered as that->ctrls
  CapyVec ctrlWeights;
  CapyPad(CapyVec, ctrlWeights);

  // Destructor
  void (*destructCapyMathFun)(void);

  // Get a reference to the control point of a given ID.
  // Input:
  //   id: the ID of the control point
  // Output:
  //   Return a pointer to the control point (CapyVec of dimension
  //   dimOut)
  CapyVec const* (*getCtrlById)(size_t const id);

  // Set the values of the control point of given ID.
  // Input:
  //    id: the ID of the control point
  //   vals: the value to set
  void (*setCtrlById)(
           size_t const id,
    double const* const vals);

  // Get a reference to the control point at a given index in each
  // input dimension.
  // Input:
  //   idx: the indices of the control point
  // Output:
  //   Return a pointer to the control point (CapyVec of dimension
  //   dimOut)
  CapyVec const* (*getCtrl)(size_t const* const idx);

  // Set the values of the control point at a given index in each
  // input dimension.
  // Input:
  //    idx: the indices of the control point
  //   vals: the value to set
  void (*setCtrl)(
    size_t const* const idx,
    double const* const vals);

  // Save the CapyBezier to a binary stream
  // Input:
  //   stream: the binary stream to save onto
  // Exception:
  //   May raise CapyExc_StreamWriteError
  void (*save)(FILE* const stream);

  // Export the CapyBezier as a C function to a stream
  // Input:
  //   stream: the stream to save onto
  //     name: the name of the function
  // Output:
  //   The exported function's interface is void <name>(double const* const in,
  //   double* const out).
  // Exception:
  //   May raise CapyExc_StreamWriteError.
  void (*exportToCFun)(
          FILE* const stream,
    char const* const name);

  // Clone the Bezier curve
  // Output:
  //   Return a new CapyBezier.
  CapyBezier* (*clone)(void);

  // Translate the bezier
  // Input:
  //   u: translation vector, must have same dimension as that->dimOut
  // Output:
  //   'u' is added to the output dimensions values.
  void (*translate)(CapyVec const* const u);

  // Scale the bezier
  // Input:
  //   u: translation vector, must have same dimension as that->dimOut
  // Output:
  //   'u' is multiplied (component wise) to the output dimensions values.
  void (*scale)(CapyVec const* const u);

  // Rotate the bezier (must have dimOut==2)
  // Input:
  //   theta: rotation angle in radians
  // Output:
  //   Output dimension values are rotated CW by 'theta' relative to the
  //   origin.
  void (*rotate2D)(double const theta);
};

// Create a CapyBezier
// Input:
//    order: the order of the Bezier object
//    dimIn: number of inputs
//   dimOut: number of outputs
// Output:
//   Return a CapyBezier
CapyBezier CapyBezierCreate(
  CapyBezierOrder_t const order,
             size_t const dimIn,
             size_t const dimOut);

// Allocate memory for a new CapyBezier and create it
// Input:
//    order: the order of the Bezier object
//    dimIn: number of inputs
//   dimOut: number of outputs
// Output:
//   Return a CapyBezier
// Exception:
//   May raise CapyExc_MallocFailed.
CapyBezier* CapyBezierAlloc(
  CapyBezierOrder_t const order,
             size_t const dimIn,
             size_t const dimOut);

// Allocate memory for a new CapyBezier and load it from a binary stream
// Input:
//   stream: the binary stream to load the Bezier from
// Output:
//   Return a CapyBezier
// Exception:
//   May raise CapyExc_MallocFailed, CapyExc_StreamReadError.
CapyBezier* CapyBezierLoad(FILE* const stream);

// Free the memory used by a CapyBezier* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyBezier to free
void CapyBezierFree(CapyBezier** const that);

// Structure to memorise the current position (inputs and outputs) of a
// CapyBezierIterator
typedef struct CapyBezierPosition {
  double in[CAPY_BEZIER_MAX_DIMIN];
  double out[CAPY_BEZIER_MAX_DIMOUT];
} CapyBezierPosition;

// Iteration windows for CapyBezierIterator
typedef struct CapyBezierIteratorWindow CapyBezierIteratorWindow;
struct CapyBezierIteratorWindow {
  CapyBezierPosition pos[2];
  CapyBezierIteratorWindow* next;
  void (*destruct)(void);
  CapyMemPoolFields(CapyBezierIteratorWindow);
};

// Declaration of the memory pool for the iteration windows
CapyDecMemPool(CapyMemPoolBezierIteratorWindow, CapyBezierIteratorWindow)

// Iterator on the positions of a CapyBezier structure.
typedef struct CapyBezierIterator {

  // Index of the current step in the iteration
  size_t idx;

  // Returned data type
  CapyBezierPosition datatype;

  // List of windows
  CapyBezierIteratorWindow* windows;

  // Bezier associated to the iteration
  CapyBezier const* bezier;

  // Epsilon for the step of iteration using euclidean distance along the
  // Bezier curve (default: 0.1)
  double epsilon;

  // Memory pool for the windows
  CapyMemPoolBezierIteratorWindow memPool;

  // Destructor
  void (*destruct)(void);

  // Reset the iterator
  // Output:
  //   Return the first position of the iteration
  CapyBezierPosition* (*reset)(void);

  // Move the iterator to the next position
  // Output:
  //   Return the next position of the iteration
  CapyBezierPosition* (*next)(void);

  // Check if the iterator is on a valid position
  // Output:
  //   Return true if the iterator is on a valid position, else false
  bool (*isActive)(void);

  // Get the current position of the iteration
  // Output:
  //   Return the current position
  CapyBezierPosition* (*get)(void);
} CapyBezierIterator;

// Create an iterator on a CapyBezier
// Input:
//   bezier: the bezier on which to iterate
// Output:
//   Return the iterator
CapyBezierIterator CapyBezierIteratorCreate(CapyBezier const* const bezier);

// Allocate memory and create an iterator on a CapyBezier
// Input:
//   bezier: the bezier on which to iterate
// Output:
//   Return the iterator
CapyBezierIterator* CapyBezierIteratorAlloc(CapyBezier const* const bezier);

// Free the memory used by a pointer to an iterator and reset '*that' to NULL
// Input:
//   that: a pointer to the iterator to free
void CapyBezierIteratorFree(CapyBezierIterator** const that);

// Bezier spline object (composite of Bezier curves)
typedef struct CapyBezierSpline CapyBezierSpline;
struct CapyBezierSpline {

  // Inherits CapyMathFun
  struct CapyMathFunDef;

  // Number of segments per input dimension
  size_t* nbSegment;

  // Tensor of CapyBezier curves. Ordered the same way as the contol points
  // in the Bezier curve. All curves must have the same dimIn and dimOut but
  // they can have various order. The continuity of the spline is not enforced.
  // Initialised to Bezier curve of order 0 and default values. The spline's
  // 't' parameters goes from 0 to nbSegment. For a given 't', the floor(t)-th
  // curve is used.
  CapyBezier** curves;

  // Total number of Bezier curves
  size_t nbCurve;

  // Destructor
  void (*destructCapyMathFun)(void);

  // Get the Bezier curve at a given location in the tensor
  // Input:
  //   idx: the indices of the segment
  // Output:
  //   Return a reference to the segment.
  CapyBezier* (*getCurve)(size_t const* const idx);

  // Set the Bezier curve at a given location in the tensor
  // Input:
  //     idx: the indices of the segment
  //   curve: the Bezier curve
  // Output:
  //   The reference to the curve is updated. If there was a previous
  //   refererence, it is destructed prior to update.
  void (*setCurve)(
    size_t const* const idx,
      CapyBezier* const curve);

  // Clone the Bezier spline
  // Output:
  //   Return a new CapyBezierSpline.
  CapyBezierSpline* (*clone)(void);

  // Translate the spline
  // Input:
  //   u: translation vector, must have same dimension as that->dimOut
  // Output:
  //   'u' is added to the output dimensions values.
  void (*translate)(CapyVec const* const u);

  // Scale the spline
  // Input:
  //   u: translation vector, must have same dimension as that->dimOut
  // Output:
  //   'u' is multiplied (component wise) to the output dimensions values.
  void (*scale)(CapyVec const* const u);

  // Rotate the spline (must have dimOut==2)
  // Input:
  //   theta: rotation angle in radians
  // Output:
  //   Output dimension values are rotated CW by 'theta' relative to the
  //   origin.
  void (*rotate2D)(double const theta);
};

// Create a CapyBezierSpline
// Input:
//  nbSegment: array of dimIn integer, number of segments per input dimension
//      dimIn: number of inputs
//     dimOut: number of outputs
// Output:
//   Return a CapyBezierSpline
CapyBezierSpline CapyBezierSplineCreate(
  size_t const* const nbSegment,
         size_t const dimIn,
         size_t const dimOut);

// Allocate memory for a new CapyBezierSpline and create it
// Input:
//  nbSegment: array of dimIn integer, number of segments per input dimension
//      dimIn: number of inputs
//     dimOut: number of outputs
// Output:
//   Return a CapyBezierSpline
// Exception:
//   May raise CapyExc_MallocFailed.
CapyBezierSpline* CapyBezierSplineAlloc(
  size_t const* const nbSegment,
         size_t const dimIn,
         size_t const dimOut);

// Free the memory used by a CapyBezierSpline* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyBezierSpline to free
void CapyBezierSplineFree(CapyBezierSpline** const that);

// Declare a list of Bezier spline
CapyDecList(CapyListBezierSpline, CapyBezierSpline*)
#endif
