// --------------------------------- bitarray.c ----------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "bitarray.h"

// Free the memory used by a CapyBitArray
static void Destruct(void) {
  methodOf(CapyBitArray);
  free(that->data);
  that->size = 0;
  that->data = NULL;
}

// Initialised the array with an array of bytes
// Input:
//   bytes: the bytes used to initialise
//   size: the number of bytes in 'bytes'
// Output:
//   'that->data' is freed if necessary and replaced with a copy of 'bytes',
//   and 'that->size' is updated
static void Set(
  uint8_t const* const bytes,
          size_t const size) {
  methodOf(CapyBitArray);
  if(that->data != NULL) free(that->data);
  safeMalloc(that->data, size);
  memcpy(that->data, bytes, size);
  that->size = 8 * size;
}

// Get one bit in the array
// Input:
//   iBit: the index of the bit
// Output:
//   Return the bit at the requested position
static bool GetBit(size_t const iBit) {
  methodOf(CapyBitArray);
  size_t iByte = iBit / 8;
  return ((that->data[iByte] >> (7 - (iBit - iByte * 8))) & 1);
}

// Resize the array
// Input:
//   size: the new size in bits
// Output:
//   Memory is allocated for 'that->data' and 'that->size' is updated. The
//   content of 'that->data' is undefined.
static void Resize(size_t const size) {
  methodOf(CapyBitArray);
  if(that->data != NULL) free(that->data);
  size_t sizeByte = size / 8;
  sizeByte += ((size % 8) != 0);
  safeMalloc(that->data, sizeByte);
  that->size = size;
}

// Set a bit in the array
// Input:
//   iBit: hte index of the bit
//   bit: the value of the bit
// Output:
//   The bit is updated.
static void SetBit(
  size_t const iBit,
    bool const bit) {
  methodOf(CapyBitArray);
  if(iBit >= that->size) {
    raiseExc(CapyExc_InvalidParameters);
    return;
  }
  size_t iByte = iBit / 8;
  size_t jBit = iBit - iByte * 8;
  if(bit) {
    that->data[iByte] |= (uint8_t)(1 << (7 - jBit));
  } else {
    that->data[iByte] &= (uint8_t)~(1 << (7 - jBit));
  }
}

// Create a CapyBitArray
// Output:
//   Return an empty CapyBitArray
CapyBitArray CapyBitArrayCreate(void) {

  // Create the array
  CapyBitArray that = {
    .size = 0,
    .data = NULL,
    .destruct = Destruct,
    .set = Set,
    .getBit = GetBit,
    .resize = Resize,
    .setBit = SetBit,
  };
  return that;
}


// Allocate memory for a new CapyBitArray and create it
// Output:
//   Return an empty CapyBitArray
// Exception:
//   May raise CapyExc_MallocFailed.
CapyBitArray* CapyBitArrayAlloc(void) {

  // Allocate memory and create the new CapyBNoise
  CapyBitArray* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyBitArrayCreate();

  // Return the new CapyBNoise
  return that;
}


// Free the memory used by a CapyBitArray* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyBitArray to free
void CapyBitArrayFree(CapyBitArray** const that) {
  if(*that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

