// --------------------------------- bitarray.h ----------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_BITARRAY_H
#define CAPY_BITARRAY_H
#include "externalHeaders.h"
#include "cext.h"

// Description:
// Bit array manipulation class.

// BitArray object
typedef struct CapyBitArray {

  // Size in bits of the array
  size_t size;

  // Data
  uint8_t* data;

  // Destructor
  void (*destruct)(void);

  // Initialised the array with an array of bytes
  // Input:
  //   bytes: the bytes used to initialise
  //   size: the number of bytes in 'bytes'
  // Output:
  //   'that->data' is freed if necessary and replaced with a copy of 'bytes',
  //   and 'that->size' is updated
  void (*set)(
    uint8_t const* const bytes,
            size_t const size);

  // Get one bit in the array
  // Input:
  //   iBit: the index of the bit
  // Output:
  //   Return the bit at the requested position
  bool (*getBit)(size_t const iBit);

  // Resize the array
  // Input:
  //   size: the new size in bits
  // Output:
  //   Memory is allocated for 'that->data' and 'that->size' is updated. The
  //   content of 'that->data' is undefined.
  void (*resize)(size_t const size);

  // Set a bit in the array
  // Input:
  //   iBit: hte index of the bit
  //   bit: the value of the bit
  // Output:
  //   The bit is updated.
  void (*setBit)(
    size_t const iBit,
      bool const bit);
} CapyBitArray;

// Create a CapyBitArray
// Output:
//   Return an empty CapyBitArray
CapyBitArray CapyBitArrayCreate(void);

// Allocate memory for a new CapyBitArray and create it
// Output:
//   Return an empty CapyBitArray
// Exception:
//   May raise CapyExc_MallocFailed.
CapyBitArray* CapyBitArrayAlloc(void);

// Free the memory used by a CapyBitArray* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyBitArray to free
void CapyBitArrayFree(CapyBitArray** const that);
#endif
