// --------------------------------- bnoise.c ----------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "bnoise.h"

// Evaluate the noise for a given input.
// Input:
//    in: the input vector
//   out: array of double updated with the result of evaluation
static void Eval(
  double const* const in,
        double* const out) {
  methodOf(CapyBNoise);

  // Warp and scale the input position from [0, 1] over the interval
  // [0, that->dim]
  double modPos[CAPY_BNOISE_NB_DIM_IN] = {0.0};
  double t[CAPY_BNOISE_NB_DIM_IN] = {0.0};
  size_t idxCurve[CAPY_BNOISE_NB_DIM_IN] = {0};
  loop(iCoord, CAPY_BNOISE_NB_DIM_IN) {
    if(in[iCoord] >= 0.0) {
      modPos[iCoord] = that->fDim * fmod(in[iCoord], 1.0);
    } else {
      modPos[iCoord] = that->fDim * (1.0 - fmod(fabs(in[iCoord]), 1.0));
    }
    t[iCoord] = fmod(modPos[iCoord], 1.0);
    idxCurve[iCoord] = (size_t)(floor(modPos[iCoord])) % that->dim;
  }

  // Map the warped position to a grid position using the two curve beziers
  double mappedPos[CAPY_BNOISE_NB_DIM_IN] = {0.0};
  loop(iCoord, CAPY_BNOISE_NB_DIM_IN) {
    double curvePos[CAPY_BNOISE_NB_DIM_IN] = {0.0};
    loop(iCurve, CAPY_BNOISE_NB_DIM_IN) {

      // Switch the coordinates
      int idxCoord = (iCoord == iCurve);

      // Calculate the position in the grid for the curve bezier given the
      // input calculated above
      curvePos[iCurve] =
        CapyCubicBezierEval(
          t[idxCoord], that->curve[iCurve][iCoord] + idxCurve[idxCoord] * 3);
    }

    // Get the mapped position as the center of the positions given by the
    // beziers curves
    mappedPos[iCoord] = 0.5 * (curvePos[0] + curvePos[1]);
  }

  // Get the indices of the cell and local position in that cell for the
  // mapped position
  size_t posIdx[CAPY_BNOISE_NB_DIM_IN] = {0};
  double posCell[CAPY_BNOISE_NB_DIM_IN] = {0.0};
  loop(iCoord, CAPY_BNOISE_NB_DIM_IN) {
    posIdx[iCoord] = (size_t)floor(mappedPos[iCoord]);
    posCell[iCoord] = fmod(mappedPos[iCoord], 1.0);

    // Apply smootherstep to the local position for derivability
    posCell[iCoord] = CapySmootherStep(posCell[iCoord]);
  }

  // Calculate the result value using bilinear interpolation on the cell
  // corners value using the local position
  double val[CAPY_BNOISE_NB_DIM_IN] = {0.0};
  loop(i, (size_t)CAPY_BNOISE_NB_DIM_IN) {
    val[i] =
      CapyLerpNorm2Arr(
        posCell[0], that->map + (posIdx[1] + i) * that->dim + posIdx[0]);
  }
  *out = CapyLerpNorm2Arr(posCell[1], val);
}

// Initialise the noise.
// Output:
//   The noise is initialise with new random values
static void Init(void) {
  methodOf(CapyBNoise);

  // Free the memory of an eventual previous initialisation
  loop(iCurve, CAPY_BNOISE_NB_DIM_IN) loop(iCoord, CAPY_BNOISE_NB_DIM_IN) {
    free(that->curve[iCurve][iCoord]);
  }
  free(that->map);

  // Allocate memory for the vertices value and initialise them to random
  // number in [0,1]
  size_t nbVertex = that->dim * that->dim;
  safeMalloc(that->map, nbVertex);
  loop(i, nbVertex) that->map[i] = $(&(that->rng), getDouble)();

  // Calculate the number of controls in a spline
  size_t nbCtrl = that->dim * 3 + 1;

  // Loop on the four bezier splines
  loop(iCurve, CAPY_BNOISE_NB_DIM_IN) loop(iCoord, CAPY_BNOISE_NB_DIM_IN) {

    // Allocate memory for the bezier spline controls (add 3 for an extra
    // segment, cf below)
    safeMalloc(that->curve[iCurve][iCoord], nbCtrl + 3);

    // The handles are set to random numbers in [0, order]
    double* ctrls = that->curve[iCurve][iCoord];
    loop(iCtrl, nbCtrl) if((iCtrl % 3) != 0) {
      ctrls[iCtrl] = $(&(that->rng), getDouble)() * (double)that->order;
    }

    // Calculate anchors as average of their neighbour handles and avoid the
    // degenerate case ctrl_i == ctrl_{i-2}
    loop(iCtrl, nbCtrl) if((iCtrl % 3) == 0) {
      size_t iCtrlPrev = 0;
      if(iCtrl > 0) iCtrlPrev = iCtrl - 1;
      else if(nbCtrl > 2) iCtrlPrev = nbCtrl - 2;
      size_t iCtrlNext = 0;
      if(iCtrl + 1 < nbCtrl) iCtrlNext = iCtrl + 1;
      if(fabs(ctrls[iCtrlPrev] - ctrls[iCtrlNext]) < 0.1) {
        ctrls[iCtrlPrev] = fmod(ctrls[iCtrlNext] + 0.1, that->order);
      }
      ctrls[iCtrl] = 0.5 * (ctrls[iCtrlPrev] + ctrls[iCtrlNext]);
    }

    // Duplicate the first segment at the end to avoid a modulo in BNoiseGet()
    loop(iCtrl, (size_t)3) ctrls[nbCtrl + iCtrl] = ctrls[1 + iCtrl];
  }
}

// Free the memory used by a CapyBNoise
static void Destruct(void) {
  methodOf(CapyBNoise);
  $(that, destructCapyNoise)();
  loop(iCurve, CAPY_BNOISE_NB_DIM_IN) loop(iCoord, CAPY_BNOISE_NB_DIM_IN) {
    free(that->curve[iCurve][iCoord]);
  }
  free(that->map);
}

// Create a CapyBNoise
// Input:
//   order: order of the noise
//    seed: seed of the noise
// Output:
//   Return a CapyBNoise
CapyBNoise CapyBNoiseCreate(
  CapyBNoiseOrder_t const order,
   CapyRandomSeed_t const seed) {

  // Create the Noise
  CapyBNoise that = {
    .order = order,
    .dim = order + 1,
    .fDim = order + 1,
    .map = NULL,
  };
  loop(iCurve, CAPY_BNOISE_NB_DIM_IN) loop(iCoord, CAPY_BNOISE_NB_DIM_IN) {
    that.curve[iCurve][iCoord] = NULL;
  }
  CapyInherits(
    that, CapyNoise, (seed, CAPY_BNOISE_NB_DIM_IN, CAPY_BNOISE_NB_DIM_OUT));
  that.destruct = Destruct;
  that.eval = Eval;
  that.init = Init;
  $(&that, init)();
  return that;
}

// Allocate memory for a new CapyBNoise and create it
// Input:
//   order: order of the noise
//    seed: seed of the noise
// Output:
//   Return a CapyBNoise
// Exception:
//   May raise CapyExc_MallocFailed.
CapyBNoise* CapyBNoiseAlloc(
  CapyBNoiseOrder_t const order,
   CapyRandomSeed_t const seed) {

  // Allocate memory and create the new CapyBNoise
  CapyBNoise* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyBNoiseCreate(order, seed);

  // Return the new CapyBNoise
  return that;
}

// Free the memory used by a CapyBNoise* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyBNoise to free
void CapyBNoiseFree(CapyBNoise** const that) {
  if(*that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
