// --------------------------------- bnoise.h ----------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_BNOISE_H
#define CAPY_BNOISE_H
#include "externalHeaders.h"
#include "cext.h"
#include "noise.h"

// Description:
// B-Noise generator class.

// Dimensions of noise: 2D->1D
#define CAPY_BNOISE_NB_DIM_IN 2
#define CAPY_BNOISE_NB_DIM_OUT 1

// Type of the order of a BNoise
typedef uint8_t CapyBNoiseOrder_t;

// BNoise object
typedef struct CapyBNoise {

  // Inherits CapyNoise
  CapyNoiseDef;

  // Order of the noise
  CapyBNoiseOrder_t order;
  CapyPad(CapyBNoiseOrder_t, order);

  // Dimension of the grid (square) in number of vertices
  size_t dim;

  // Cast to float of the dimension of the grid
  double fDim;

  // Noise data
  double* map;
  double* curve[CAPY_BNOISE_NB_DIM_IN][CAPY_BNOISE_NB_DIM_IN];

  // Destructor
  void (*destructCapyNoise)(void);
} CapyBNoise;

// Create a CapyBNoise
// Input:
//   order: order of the noise
//    seed: seed of the noise
// Output:
//   Return a CapyBNoise
CapyBNoise CapyBNoiseCreate(
  CapyBNoiseOrder_t const order,
   CapyRandomSeed_t const seed);

// Allocate memory for a new CapyBNoise and create it
// Input:
//   order: order of the noise
//    seed: seed of the noise
// Output:
//   Return a CapyBNoise
// Exception:
//   May raise CapyExc_MallocFailed.
CapyBNoise* CapyBNoiseAlloc(
  CapyBNoiseOrder_t const order,
   CapyRandomSeed_t const seed);

// Free the memory used by a CapyBNoise* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyBNoise to free
void CapyBNoiseFree(CapyBNoise** const that);
#endif
