// ---------------------------- bresenham.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "bresenham.h"

// Reset the iterator
// Output:
//   Return the first element of the iteration
static int64_t** Reset(void) {
  methodOf(CapyBresenham);
  loop(i, 2) that->pos[i] = that->startPos[i];
  that->idx = 0;
  loop(i, 2) {
    that->d[i] = labs(that->endPos[i] - that->startPos[i]);
    that->s[i] = that->startPos[i] < that->endPos[i] ? 1 : -1;
  }
  that->err[0] = (that->d[0] > that->d[1] ? that->d[0] : -that->d[1]) / 2;
  that->err[1] = that->err[0];
  return (int64_t**)&(that->pos);
}

// Check if the iterator is on a valid combination
// Output:
//   Return true if the iterator is on a valid combination, else false
static bool IsActive(void) {
  methodOf(CapyBresenham);
  return (that->pos[0] != that->endPos[0] || that->pos[1] != that->endPos[1]);
}

// Move the iterator to the next combination
// Output:
//   Return the next combination of the iteration
static int64_t** Next(void) {
  methodOf(CapyBresenham);
  ++(that->idx);
  that->err[1] = that->err[0];
  if(that->err[1] > -that->d[0]) {
    that->err[0] -= that->d[1];
    that->pos[0] += that->s[0];
  }
  if(that->err[1] < that->d[1]) {
    that->err[0] += that->d[0];
    that->pos[1] += that->s[1];
  }
  return (int64_t**)&(that->pos);
}

// Free the memory used by a CapyBresenham
static void Destruct(void) {
  methodOf(CapyBresenham);
  free(that->pos);
  free(that->startPos);
  free(that->endPos);
}

// Create a CapyBresenham
// Output:
//   Return a CapyBresenham
CapyBresenham CapyBresenhamCreate(void) {
  CapyBresenham that = {
    .idx = 0,
    .reset = Reset,
    .isActive = IsActive,
    .next = Next,
    .destruct = Destruct,
  };
  safeMalloc(that.pos, 2);
  that.pos[0] = 0;
  that.pos[1] = 0;
  safeMalloc(that.startPos, 2);
  that.startPos[0] = 0;
  that.startPos[1] = 0;
  safeMalloc(that.endPos, 2);
  that.endPos[0] = 1;
  that.endPos[1] = 1;
  return that;
}

// Allocate memory for a new CapyBresenham and create it
// Output:
//   Return a CapyBresenham
// Exception:
//   May raise CapyExc_MallocFailed.
CapyBresenham* CapyBresenhamAlloc(void) {
  CapyBresenham* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyBresenhamCreate();
  return that;
}

// Free the memory used by a CapyBresenham* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyBresenham to free
void CapyBresenhamFree(CapyBresenham** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
