// ------------------------ burrowswheelertransform.c -----------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "burrowswheelertransform.h"
#include "comparator.h"
#include "sort.h"

// Comparator class for the sorting of permutations
typedef struct CapyBWTComparator {

  // Inherits CapyComparator
  CapyComparatorDef;

  // Data transformed
  CapyBurrowsWheelerTransformData data;

  // Destructor
  void (*destructCapyComparator)(void);
} CapyBWTComparator;

// Compare the two given inputs.
// Input:
//   a: the first input
//   b: the second input
// Output:
//   Return an int, negative if a is 'before' b, 0 if a is 'same' as b,
//   positive if a is 'after' b.
static int EvalBWTComparator(
  void const* a,
  void const* b) {
  methodOf(CapyBWTComparator);
  uint8_t const* ptrs[2] = {*(uint8_t const**)a, *(uint8_t const**)b};
  do {
    if(*(ptrs[0]) < *(ptrs[1])) return -1;
    else if(*(ptrs[0]) > *(ptrs[1])) return 1;
    else {
      loop(i, 2) {
        (ptrs[i])++;
        if(ptrs[i] == that->data.bytes + that->data.size) {
          ptrs[i] = that->data.bytes;
        }
      }
    }
  } while(ptrs[0] != *(uint8_t const**)a);
  return 0;
}

// Free the memory used by a CapyBWTComparator
static void DestructBWTComparator(void) {
  methodOf(CapyBWTComparator);
  $(that, destructCapyComparator)();
}

// Create a CapyBWTComparator
// Output:
//   Return a CapyBWTComparator
static CapyBWTComparator CapyBWTComparatorCreate(
  CapyBurrowsWheelerTransformData const data) {
  CapyBWTComparator that;
  CapyInherits(that, CapyComparator, ());
  that.data = data;
  that.eval = EvalBWTComparator;
  that.destruct = DestructBWTComparator;
  return that;
}

// Transform the data
// Input:
//   data: the data to transform
// Output:
//   Return the transformed data
static CapyBurrowsWheelerTransformData Transform(
  CapyBurrowsWheelerTransformData const data) {

  // Variable to memorise the result
  CapyBurrowsWheelerTransformData res = {0};

  // Create the array of pointers for sorting the permutation without creating
  // the Burrows-Wheeler matrix
  uint8_t** ptrs = NULL;
  safeMalloc(ptrs, data.size);
  if(ptrs == NULL) return res;
  loop(i, data.size) ptrs[i] = data.bytes + i;

  // Allocate memory for the result
  safeMalloc(res.bytes, data.size);
  if(res.bytes == NULL) return res;
  res.size = data.size;

  // Create the comparator for sorting the permutation
  CapyBWTComparator cmp = CapyBWTComparatorCreate(data);

  // Sort the array of pointers, which is equivalent to sorting the permutations
  CapyQuickSort(ptrs, sizeof(*ptrs), data.size, (CapyComparator*)&cmp);

  // Update the result with the last byte of the sorted permutation
  loop(i, data.size) {
    uint8_t* lastPermutedByte = ptrs[i] + data.size - 1;
    if(lastPermutedByte >= data.bytes + data.size) {
      lastPermutedByte -= data.size;
    }
    res.bytes[i] = *lastPermutedByte;

    // Search the index of the first byte in transformed data
    if(lastPermutedByte == data.bytes) res.idxFirstByte = i;
  }

  // Free memory
  free(ptrs);
  $(&cmp, destruct)();

  // Return the result
  return res;
}

// Untransform the data
// Input:
//   data: the data to untransform
// Output:
//   Return the untransformed data
static CapyBurrowsWheelerTransformData Untransform(
  CapyBurrowsWheelerTransformData const data) {

  // Variable to memorise the result
  CapyBurrowsWheelerTransformData res = {0};

  // Create the arrays of number of occurence of each byte and rank
  size_t nb[256] = {0};
  size_t* rank = NULL;
  safeMalloc(rank, data.size);
  if(rank == NULL) return res;
  loop(iBytes, data.size) {
    rank[iBytes] = nb[data.bytes[iBytes]];
    ++(nb[data.bytes[iBytes]]);
  }

  // Allocate memory for the result
  safeMalloc(res.bytes, data.size);
  if(res.bytes == NULL) {
    free(rank);
    return res;
  }
  res.size = data.size;

  // Untransform the data from right to left
  size_t iRes = data.size;
  size_t iByte = data.idxFirstByte;
  while(iRes > 0) {
    iRes--;
    size_t iPrevByte = 0;
    loop(i, data.bytes[iByte]) iPrevByte += nb[i];
    iPrevByte += rank[iByte];
    iByte = iPrevByte;
    res.bytes[iRes] = data.bytes[iByte];
  }

  // Free memory
  free(rank);

  // Return the result
  return res;
}

// Free the memory used by a CapyBurrowsWheelerTransform
static void Destruct(void) {
  return;
}

// Create a CapyBurrowsWheelerTransform
// Output:
//   Return a CapyBurrowsWheelerTransform
CapyBurrowsWheelerTransform CapyBurrowsWheelerTransformCreate(void) {
  CapyBurrowsWheelerTransform that = {
    .destruct = Destruct,
    .transform = Transform,
    .untransform = Untransform,
  };
  return that;
}

// Allocate memory for a new CapyBurrowsWheelerTransform and create it
// Output:
//   Return a CapyBurrowsWheelerTransform
// Exception:
//   May raise CapyExc_MallocFailed.
CapyBurrowsWheelerTransform* CapyBurrowsWheelerTransformAlloc(void) {
  CapyBurrowsWheelerTransform* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyBurrowsWheelerTransformCreate();
  return that;
}

// Free the memory used by a CapyBurrowsWheelerTransform* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyBurrowsWheelerTransform to free
void CapyBurrowsWheelerTransformFree(CapyBurrowsWheelerTransform** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
