// ---------------------------- camera.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "camera.h"

// Project a 3D point in world coordinates to a 3D point in screen coordinates
// (pinhole version)
// Input:
//   in: the coordinates of the point to project
//   out: the result projected coordinates, can be same as 'in'
// Output:
//   'out' is updated.
static void ProjectPinhole(
    CapyCamera* const that,
  double const* const in,
        double* const out) {
  CapyVec p = CapyVecCreateLocal3D;
  loop(i, 3) p.vals[i] = in[i] - that->pos.vals[i];
  CapyVec u = {.dim = 3, .vals = out};
  CapyMatProdVec(&(that->pose), &p, &u);
  loop(i, 2) out[i] *= that->focalLength;
}

// Project a 3D point in screen coordinates to a 3D point in world coordinates
// (pinhole version)
// Input:
//   in: the coordinates of the point to inverse project
//   out: the result inverse projected coordinates, can be same as 'in'
// Output:
//   'out' is updated.
static void ProjectInvPinhole(
    CapyCamera* const that,
  double const* const in,
        double* const out) {
  CapyVec p = CapyVecCreateLocal3D;
  loop(i, 3) p.vals[i] = in[i];
  loop(i, 2) p.vals[i] /= that->focalLength;
  CapyVec u = {.dim = 3, .vals = out};
  CapyMatTransProdVec(&(that->pose), &p, &u);
  loop(i, 3) out[i] += that->pos.vals[i];
}

// Project a 3D point in world coordinates to a 3D point in screen coordinates
// Input:
//   in: the coordinates of the point to project
//   out: the result projected coordinates, can be same as 'in'
// Output:
//   'out' is updated. The projection is calculated according to the type
//   of camera
static void Project(
  double const* const in,
        double* const out) {
  methodOf(CapyCamera);
  if(that->type == capyCameraType_pinhole) ProjectPinhole(that, in, out);
  else raiseExc(CapyExc_UndefinedExecution);
}

// Project a 3D point in screen coordinates to a 3D point in world coordinates
// Input:
//   in: the coordinates of the point to inverse project
//   out: the result inverse projected coordinates, can be same as 'in'
// Output:
//   'out' is updated. The inverse projection is calculated according to the
//   type of camera
static void ProjectInv(
  double const* const in,
        double* const out) {
  methodOf(CapyCamera);
  if(that->type == capyCameraType_pinhole) ProjectInvPinhole(that, in, out);
  else raiseExc(CapyExc_UndefinedExecution);
}

// Convert screen coordinates into image coordinates (perspective version)
// Input:
//   posScreen: the screen coordinates
//   posOrig: the position in the image corresponding to the origin of the
//            screen
//   posImg: the result image coordinates
// Output:
//   'posImg' is updated
static void ToImgCoordPerspective(
    CapyCamera const* const that,
        double const* const posScreen,
  CapyImgPos_t const* const posOrig,
        CapyImgPos_t* const posImg) {
  loop(i, 2) {
    posImg[i] =
      (CapyImgPos_t)(posScreen[i] / posScreen[2] * that->sensorResolution);
  }
  posImg[0] = posOrig[0] + posImg[0];
  posImg[1] = posOrig[1] - posImg[1];
}

// Convert screen coordinates into image coordinates (orthographic version)
// Input:
//   posScreen: the screen coordinates
//   posOrig: the position in the image corresponding to the origin of the
//            screen
//   posImg: the result image coordinates
// Output:
//   'posImg' is updated
static void ToImgCoordOrthographic(
    CapyCamera const* const that,
        double const* const posScreen,
  CapyImgPos_t const* const posOrig,
        CapyImgPos_t* const posImg) {
  loop(i, 2) {
    posImg[i] =
      (CapyImgPos_t)(posScreen[i] * that->sensorResolution);
  }
  posImg[0] = posOrig[0] + posImg[0];
  posImg[1] = posOrig[1] - posImg[1];
}

// Convert screen coordinates into image coordinates
// Input:
//   posScreen: the screen coordinates
//   posOrig: the position in the image corresponding to the origin of the
//            screen
//   posImg: the result image coordinates
// Output:
//   'posImg' is updated
static void ToImgCoord(
        double const* const posScreen,
  CapyImgPos_t const* const posOrig,
        CapyImgPos_t* const posImg) {
  methodOf(CapyCamera);
  if(that->screenType == capyCameraScreenType_perspective) {
    ToImgCoordPerspective(that, posScreen, posOrig, posImg);
  } else if(that->screenType == capyCameraScreenType_orthographic) {
    ToImgCoordOrthographic(that, posScreen, posOrig, posImg);
  } else raiseExc(CapyExc_UndefinedExecution);
}

// Convert image coordinates to a direction vector from the camera position
// Input:
//   posImg: the image coordinates
//   posOrig: the position in the image corresponding to the origin of the
//            screen
//   out: the result direction vector
// Output:
//   'out' is updated and normalised
static void FromImgCoordPerspective(
    CapyCamera const* const that,
        CapyImgPos_t* const posImg,
  CapyImgPos_t const* const posOrig,
              double* const out) {

  // Calculate the screen position
  double posScreen[3];
  posScreen[0] =
    ((double)(posImg[0]) - (double)(posOrig[0])) / that->sensorResolution;
  posScreen[1] =
    -((double)(posImg[1]) - (double)(posOrig[1])) / that->sensorResolution;
  posScreen[2] = 1.0;

  // Calculate the world position
  double posWorld[3];
  $(that, projectInv)(posScreen, posWorld);

  // Calculate the normalised direction vector
  double l = 0.0;
  loop(i, 3) {
    out[i] = posWorld[i] - that->pos.vals[i];
    l += out[i] * out[i];
  }
  l = sqrt(l);
  loop(i, 3) out[i] /= l;
}

// Convert image coordinates to a direction vector from the camera position
// Input:
//   posImg: the image coordinates
//   posOrig: the position in the image corresponding to the origin of the
//            screen
//   out: the result direction vector
// Output:
//   'out' is updated and normalised
static void FromImgCoord(
        CapyImgPos_t* const posImg,
  CapyImgPos_t const* const posOrig,
              double* const out) {
  methodOf(CapyCamera);
  if(that->screenType == capyCameraScreenType_perspective) {
    FromImgCoordPerspective(that, posImg, posOrig, out);
  } else if(that->screenType == capyCameraScreenType_orthographic) {

    // Same as perspective
    FromImgCoordPerspective(that, posImg, posOrig, out);
  } else raiseExc(CapyExc_UndefinedExecution);
}

// Translate the camera
// Input:
//   u: the translation vector
// Output:
//   The camera position is updated
static void Translate(double const* const u) {
  methodOf(CapyCamera);
  CapyVec v = {.dim = 3, .vals = (double*)u};
  CapyVecAdd(&(that->pos), &v, &(that->pos));
}

// Rotate the camera
// Input:
//   q: the quaternion equivalent to the rotation
// Output:
//   The camera pose is updated
static void Rotate(CapyQuaternion const* const q) {
  methodOf(CapyCamera);
  loop(i, 3) $(q, apply)(that->pose.vals + i * 3, that->pose.vals + i * 3);
}

// Get the length in pixel of a projected segment
// Input:
//   posA: position of one end of the segment
//   posB: position of the other end of the segment
// Output:
//   Return the length
static double GetProjectedLength(
  double const* const posA,
  double const* const posB) {
  methodOf(CapyCamera);
  double u[3], v[3];
  $(that, project)(posA, u);
  $(that, project)(posB, v);
  loop(i, 2) {
    u[i] = u[i] / u[2] * that->sensorResolution;
    v[i] = v[i] / v[2] * that->sensorResolution;
  }
  double length =
    sqrt((u[0] - v[0]) * (u[0] - v[0]) + (u[1] - v[1]) * (u[1] - v[1]));
  return length;
}

// Set the camera parameters to render an isometric projection
// Output:
//   The camera parameters are modified.
static void SetToIsometric(void) {
  methodOf(CapyCamera);
  that->screenType = capyCameraScreenType_orthographic;
  that->focalLength = 1.0;
  that->sensorResolution = 1.0;
  loop(i, 3) that->pos.vals[i] = 1.0;
  loop(i, 3) {
    that->front.vals[i] = 0.0;
    that->up.vals[i] = 0.0;
    that->right.vals[i] = 0.0;
  }
  that->front.vals[2] = -1.0;
  that->up.vals[1] = 1.0;
  that->right.vals[0] = -1.0;
  CapyQuaternion q =
    CapyQuaternionCreateFromRotAxis(capyXAxis3D.vals, -M_PI_4);
  $(that, rotate)(&q);
  $(&q, destruct)();
  q = CapyQuaternionCreateFromRotAxis(capyYAxis3D.vals, M_PI_4);
  $(that, rotate)(&q);
  $(&q, destruct)();
}

// Set the camera parameters to render a dimetric projection
// Output:
//   The camera parameters are modified.
static void SetToDimetric(void) {
  methodOf(CapyCamera);
  that->screenType = capyCameraScreenType_orthographic;
  that->focalLength = 1.0;
  that->sensorResolution = 1.0;
  loop(i, 3) that->pos.vals[i] = 1.0;
  loop(i, 3) {
    that->front.vals[i] = 0.0;
    that->up.vals[i] = 0.0;
    that->right.vals[i] = 0.0;
  }
  that->front.vals[2] = -1.0;
  that->up.vals[1] = 1.0;
  that->right.vals[0] = -1.0;
  CapyQuaternion q =
    CapyQuaternionCreateFromRotAxis(capyXAxis3D.vals, -M_PI / 6.0);
  $(that, rotate)(&q);
  $(&q, destruct)();
  q = CapyQuaternionCreateFromRotAxis(capyYAxis3D.vals, M_PI_4);
  $(that, rotate)(&q);
  $(&q, destruct)();
}

// Free the memory used by a CapyCamera
static void Destruct(void) {
  methodOf(CapyCamera);
  CapyVecDestruct(&(that->pos));
  CapyMatDestruct(&(that->pose));
}

// Create a CapyCamera
// Output:
//   Return a CapyCamera
CapyCamera CapyCameraCreate(void) {
  CapyCamera that = {
    .type = capyCameraType_pinhole,
    .screenType = capyCameraScreenType_perspective,
    .pos = CapyVecCreate(3),
    .pose = CapyMatCreate(3, 3),
    .focalLength = 0.017,
    .sensorResolution = 100000.0,
    .destruct = Destruct,
    .project = Project,
    .projectInv = ProjectInv,
    .toImgCoord = ToImgCoord,
    .fromImgCoord = FromImgCoord,
    .translate = Translate,
    .rotate = Rotate,
    .getProjectedLength = GetProjectedLength,
    .setToIsometric = SetToIsometric,
    .setToDimetric = SetToDimetric,
  };
  CapyMatSetToIdentity(&(that.pose));
  that.right.dim = 3;
  that.right.vals = that.pose.vals;
  that.up.dim = 3;
  that.up.vals = that.pose.vals + 3;
  that.front.dim = 3;
  that.front.vals = that.pose.vals + 6;
  return that;
}

// Allocate memory for a new CapyCamera and create it
// Output:
//   Return a CapyCamera
// Exception:
//   May raise CapyExc_MallocFailed.
CapyCamera* CapyCameraAlloc(void) {
  CapyCamera* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyCameraCreate();
  return that;
}

// Free the memory used by a CapyCamera* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyCamera to free
void CapyCameraFree(CapyCamera** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
