// ---------------------------------- camera.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_CAMERA_H
#define CAPY_CAMERA_H
#include "externalHeaders.h"
#include "cext.h"
#include "quaternion.h"
#include "image.h"
#include "geometricShape.h"

// Description:
// Camera class.
// Assuming a left handed world coordinates system where +x/right, +y/up,
// +z/forward. All unit in meters.

// Types of camera
typedef enum CapyCameraType {
  capyCameraType_pinhole,
  capyCameraType_nb,
} CapyCameraType;

// Types of screen
typedef enum CapyCameraScreenType {
  capyCameraScreenType_perspective,
  capyCameraScreenType_orthographic,
  capyCameraScreenType_nb,
} CapyCameraScreenType;

// Camera object
typedef struct CapyCamera {

  // Type of camera (default: pinhole)
  CapyCameraType type;
  CapyPad(CapyCameraType, type);

  // Type of camera (default: perspective)
  CapyCameraScreenType screenType;
  CapyPad(CapyCameraScreenType, screenType);

  // Position (center of the lens, in world coordinates system, default: 0)
  CapyVec pos;

  // Pose matrix of the camera (in world coordinates system, default: identity)
  // Per row: right, up, front normalised vector
  CapyMat pose;

  // Right direction from the camera (in world coordinates system, reference
  // to the pose matrix values, default: x)
  CapyVec right;

  // Up direction from the camera (in world coordinates system, reference
  // to the pose matrix values, default: y)
  CapyVec up;

  // Forward direction from the camera (in world coordinates system, reference
  // to the pose matrix values, default: z)
  CapyVec front;

  // Focal length, the smaller the larger the field of view (default: 0.017)
  double focalLength;

  // Sensor resolution (pixels/meter, default: 100000)
  double sensorResolution;

  // Destructor
  void (*destruct)(void);

  // Project a 3D point in world coordinates to a 3D point in screen coordinates
  // Input:
  //   in: the coordinates of the point to project
  //   out: the result projected coordinates, can be same as 'in'
  // Output:
  //   'out' is updated. The projection is calculated according to the type
  //   of camera
  void (*project)(
    double const* const in,
          double* const out);

  // Project a 3D point in screen coordinates to a 3D point in world coordinates
  // Input:
  //   in: the coordinates of the point to inverse project
  //   out: the result inverse projected coordinates, can be same as 'in'
  // Output:
  //   'out' is updated. The inverse projection is calculated according to the
  //   type of camera
  void (*projectInv)(
    double const* const in,
          double* const out);

  // Convert screen coordinates into image coordinates
  // Input:
  //   posScreen: the screen coordinates
  //   posOrig: the position in the image corresponding to the origin of the
  //            screen
  //   posImg: the result image coordinates
  // Output:
  //   'posImg' is updated
  void (*toImgCoord)(
          double const* const posScreen,
    CapyImgPos_t const* const posOrig,
          CapyImgPos_t* const posImg);

  // Convert image coordinates to a direction vector from the camera position
  // Input:
  //   posImg: the image coordinates
  //   posOrig: the position in the image corresponding to the origin of the
  //            screen
  //   out: the result direction vector
  // Output:
  //   'out' is updated and normalised
  void (*fromImgCoord)(
          CapyImgPos_t* const posImg,
    CapyImgPos_t const* const posOrig,
                double* const out);

  // Translate the camera
  // Input:
  //   u: the translation vector
  // Output:
  //   The camera position is updated
  void (*translate)(double const* const u);

  // Rotate the camera
  // Input:
  //   q: the quaternion equivalent to the rotation
  // Output:
  //   The camera pose is updated
  void (*rotate)(CapyQuaternion const* const q);

  // Get the length in pixel of a projected segment
  // Input:
  //   posA: position of one end of the segment
  //   posB: position of the other end of the segment
  // Output:
  //   Return the length
  double (*getProjectedLength)(
    double const* const posA,
    double const* const posB);

  // Set the camera parameters to render an isometric projection
  // Output:
  //   The camera parameters are modified.
  void (*setToIsometric)(void);

  // Set the camera parameters to render a dimetric projection
  // Output:
  //   The camera parameters are modified.
  void (*setToDimetric)(void);
} CapyCamera;

// Create a CapyCamera
// Output:
//   Return a CapyCamera
CapyCamera CapyCameraCreate(void);

// Allocate memory for a new CapyCamera and create it
// Output:
//   Return a CapyCamera
// Exception:
//   May raise CapyExc_MallocFailed.
CapyCamera* CapyCameraAlloc(void);

// Free the memory used by a CapyCamera* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyCamera to free
void CapyCameraFree(CapyCamera** const that);
#endif
