// ---------------------------------- math.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_MATH_H
#define CAPY_MATH_H
#include "externalHeaders.h"
#include "trycatch.h"
#include "ratio.h"
#include "range.h"

// Description:
// Mathematical functions.

// Get the GCD of two positive integers
// Input:
//   a, b: the two integers
// Output:
//   Return the greatest common divisor using the Stein's algorithm
uint64_t CapyGcd(
  uint64_t a,
  uint64_t b);

// Get the GCD decomposition of two integers
// Input:
//   a, b: the two integers
//   x, y: the result two integers
// Output:
//   Update x and y with the result of ax+by=gcd(a,b) using the extended
//   Euclidian algorithm, and return gcd(a,b)
int64_t CapyGcdDecomposition(
   int64_t a,
   int64_t b,
  int64_t* x,
  int64_t* y);

// Get the GCD decomposition of two integers (version for positive inputs)
// Input:
//   a, b: the two integers
//   x, y: the result two integers
// Output:
//   Update x and y with the result of ax+by=gcd(a,b) using the extended
//   Euclidian algorithm, and return gcd(a,b)
uint64_t CapyGcdDecompositionUnsignedInput(
  uint64_t a,
  uint64_t b,
  int64_t* x,
  int64_t* y);

// Get the LCM of two positive integers
// Input:
//   a, b: the two integers
// Output:
//   Return the lowest common multpiple using the formula
//   lcm(a, b) = |ab|/gcd(a, b)
uint64_t CapyLcm(
  uint64_t a,
  uint64_t b);

// Peasant multiplication of two integers
// Input:
//   a, b: the two integers to multiply
// Output:
//   Return the multiplication of two integers using the Peasant method.
uint64_t CapyPeasantMul(
  uint64_t a,
  uint64_t b);

// Calculate ab/c for a,b,c positive integers while avoiding eventual
// intermediate overflow using the peasant multiplication
// Input:
//   a, b, c: the three integers
// Output:
//   Return the value of ab/c as an uint64_t for the integer part and
//   a CapyRatio for the remaining fractional part, or {0, capyRatioNaN}
//   if the calculation overflows.
typedef struct CapyPeasantMulDivRes {
  uint64_t base;
  CapyRatio frac;
} CapyPeasantMulDivRes;

CapyPeasantMulDivRes CapyPeasantMulDiv(
  uint64_t a,
  uint64_t b,
  uint64_t c);

// Generic smooth step function.
// Input:
//   x: input, in [0.0, 1.0]
//   a: smoothing coefficient in ]0.0, +inf]
// Output:
//   Return the smoothed value of x. If a equals 1.0, it's x itself. As
//   a gets lower than 1.0, the smoothed value varies following the
//   pattern fast-slow-fast. As a gets greater than 1.0, the smoothed
//   value varies following the pattern slow-fast-slow. Continuous but
//   not necessary derivable at x=0.0 or x=1.0.
double CapySmoothStep(
  double const x,
  double const a);

// Smoother step function
// Inputs:
//   x: the input value in [0,1]
// Output:
//   Return the smoothed value, in [0, 1]
double CapySmootherStep(double const x);

// Power function for an integer value and integer exponent
// Input:
//   x: the value
//   n: the power
// Output:
//   Return x^n
int64_t CapyPowi(
   int64_t const x,
  uint64_t const n);

// Power function for a real value and integer exponent
// Input:
//   x: the value
//   n: the power
// Output:
//   Return x^n
double CapyPowf(
    double const x,
  uint64_t const n);

// LERP function, map a double value linearly from a range to another
// Input:
//      x: the input
//   from: range of the input
//     to: range of the output
// Output:
//   Return the mapped intput
double CapyLerp(
                  double const x,
  CapyRangeDouble const* const from,
  CapyRangeDouble const* const to);

// LERP function, map a double value linearly from [0, 1] to an array
// Input:
//      x: the input
//     to: array of two values
// Output:
//   Return the mapped intput
double CapyLerpNorm2Arr(
  double const x,
  double const to[2]);

// Comparator functions for basic types (to be used with the qsort
// function)
typedef int (*CapyCmpFun)(
  void const*,
  void const*);
int CapyCmpCharInc(
  char const* a,
  char const* b);
int CapyCmpInt8Inc(
  int8_t const* a,
  int8_t const* b);
int CapyCmpUInt8Inc(
  uint8_t const* a,
  uint8_t const* b);
int CapyCmpInt16Inc(
  int16_t const* a,
  int16_t const* b);
int CapyCmpUInt16Inc(
  uint16_t const* a,
  uint16_t const* b);
int CapyCmpInt32Inc(
  int32_t const* a,
  int32_t const* b);
int CapyCmpUInt32Inc(
  uint32_t const* a,
  uint32_t const* b);
int CapyCmpInt64Inc(
  int64_t const* a,
  int64_t const* b);
int CapyCmpUInt64Inc(
  uint64_t const* a,
  uint64_t const* b);
int CapyCmpFloatInc(
  float const* a,
  float const* b);
int CapyCmpDoubleInc(
  double const* a,
  double const* b);
int CapyCmpSizeInc(
  size_t const* a,
  size_t const* b);
int CapyCmpCharDec(
  char const* a,
  char const* b);
int CapyCmpInt8Dec(
  int8_t const* a,
  int8_t const* b);
int CapyCmpUInt8Dec(
  uint8_t const* a,
  uint8_t const* b);
int CapyCmpInt16Dec(
  int16_t const* a,
  int16_t const* b);
int CapyCmpUInt16Dec(
  uint16_t const* a,
  uint16_t const* b);
int CapyCmpInt32Dec(
  int32_t const* a,
  int32_t const* b);
int CapyCmpUInt32Dec(
  uint32_t const* a,
  uint32_t const* b);
int CapyCmpInt64Dec(
  int64_t const* a,
  int64_t const* b);
int CapyCmpUInt64Dec(
  uint64_t const* a,
  uint64_t const* b);
int CapyCmpFloatDec(
  float const* a,
  float const* b);
int CapyCmpDoubleDec(
  double const* a,
  double const* b);

// Vector
typedef struct CapyVec {
  size_t dim;
  double* vals;
} CapyVec;

#define CapyVecDef(S, ...)                                 \
  struct {                                                 \
    size_t dim;                                            \
    union {double vals[S]; struct __attribute__((packed)){ \
      double __VA_ARGS__;};};                              \
  }

// Predefined vectors
extern CapyVec const capyXAxis2D;
extern CapyVec const capyYAxis2D;
extern CapyVec const capyXAxis3D;
extern CapyVec const capyYAxis3D;
extern CapyVec const capyZAxis3D;

// Macro to define a local CapyVec of dimension 3 and avoid memory allocation
#define CapyVecCreateLocal3D \
  (CapyVec){.dim = 3, .vals = (double[3]){0., 0., 0.}}

// Macro to define a local CapyVec of dimension 2 and avoid memory allocation
#define CapyVecCreateLocal2D \
  (CapyVec){.dim = 2, .vals = (double[2]){0., 0.}}

// Matrix (values stored by row)
typedef struct CapyMat {
  double* vals;
  union {
    size_t dims[2];
    struct __attribute__((packed)) {size_t nbCol; size_t nbRow;};
  };
} CapyMat;

// Macro to define a local CapyMat of dimension 3x3 and avoid memory allocation
#define CapyMatCreateLocal3x3                                \
  (CapyMat){                                                 \
    .dims = {3, 3},                                          \
    .vals = (double[9]){0., 0., 0., 0., 0., 0., 0., 0., 0.}, \
  }

// Macro to define a local CapyMat of dimension 2x2 and avoid memory allocation
#define CapyMatCreateLocal2x2                                \
  (CapyMat){                                                 \
    .dims = {2, 2},                                          \
    .vals = (double[4]){0., 0., 0., 0.},                     \
  }

// Allocate an array of CapyVec
// Input:
//   dim: the dimension of the vectors
//    nb: the size of the array
// Output:
//   Return a newly allocated array of CapyVec (values initialised to 0.0)
// Exceptions:
//   May raise CapyExc_MallocFailed
CapyVec* CapyVecAllocArr(
  size_t dim,
  size_t nb);

// Free an array of CapyVec
// Input:
//   that: the array of CapyVec
//     nb: the size of the array
void CapyVecFreeArr(
  CapyVec** const that,
           size_t nb);

// Create a CapyVec
// Input:
//   dim: the dimension of the vector
// Output:
//   Return a CapyVec (values initialised to 0.0)
// Exceptions:
//   May raise CapyExc_MallocFailed
CapyVec CapyVecCreate(size_t dim);

// Free a CapyVec
// Input:
//   that: the CapyVec to free
void CapyVecDestruct(CapyVec* const that);

// Get a 3D vector orthogonal to another 3D vector
// Input:
//   u: the input vector
//   v: the output vector orthonormal to u
// Output:
//   v is updated
void CapyVec3DGetOrtho(
  double const* const u,
        double* const v);

// Create a CapyMat
// Input:
//   nbCol: the number of columns of the matrix
//   nbRow: the number of rows of the matrix
// Output:
//   Return a CapyVec (values initialised to 0.0)
// Exceptions:
//   May raise CapyExc_MallocFailed
CapyMat CapyMatCreate(
  size_t const nbCol,
  size_t const nbRow);

// Create the 3x3 CapyMat for the rotation matrix around the i-th axis and
// given angle (same handness for the rotation as for the coordinates system)
// Input:
//   iAxis: the axis index (0: x, 1: y, 2: z)
//   theta: the angle in radians
// Output:
//   Return a CapyVec
// Exceptions:
//   May raise CapyExc_MallocFailed, CapyExc_UndefinedExecution
CapyMat CapyMatCreateRotMat(
  size_t const iAxis,
  double const theta);

// Free a CapyMat
// Input:
//   that: the CapyMat to free
void CapyMatDestruct(CapyMat* const that);

// Add two vectors
// Input:
//   a: first vector
//   b: second vector
//   c: result vector, c=a+b (c can be a or b)
void CapyVecAdd(
  CapyVec const* const a,
  CapyVec const* const b,
        CapyVec* const c);

// Substract two vectors
// Input:
//   a: first vector
//   b: second vector
//   c: result vector, c=a-b
void CapyVecSub(
  CapyVec const* const a,
  CapyVec const* const b,
        CapyVec* const c);

// Dot product of two vectors
// Input:
//   a: first vector
//   b: second vector
//   c: result vector, c=a.b
void CapyVecDot(
  CapyVec const* const a,
  CapyVec const* const b,
         double* const c);

// Cross product of two vectors of dimension 3
// Input:
//   a: first vector
//   b: second vector
//   c: result vector, c=a*b
void CapyVecCross(
  CapyVec const* const a,
  CapyVec const* const b,
        CapyVec* const c);

// Product vector scalar
// Input:
//   a: vector
//   b: scalar
//   c: result vector, c=a*b
void CapyVecMul(
  CapyVec const* const a,
          double const b,
        CapyVec* const c);

// Copy a vector to another
// Input:
//   a: from vector
//   b: to vector
void CapyVecCopy(
  CapyVec const* const a,
        CapyVec* const b);

// Normalise the vector
// Input:
//   a: the vector
// Output:
//   The vector is normalised.
void CapyVecNormalise(CapyVec* const a);

// Normalise the vector using the fast inverse square root
// Input:
//   a: the vector
// Output:
//   The vector is normalised.
void CapyVecNormaliseFast(CapyVec* const a);

// Get the norm of the vector
// Input:
//   a: the vector
// Output:
//   Retun the norm of the vector
double CapyVecGetNorm(CapyVec const* const a);

// Get the cosine-similarity of two vectors
// Input:
//   a: the first vector
//   b: the second vector
// Output:
//   Retun the dot product of normalised vectors
double CapyVecGetCosineSimilarity(
  CapyVec const* const a,
  CapyVec const* const b);

// Get the approximated norm of a 2D vector
// Input:
//   u: the 2D vector
// Output:
//   Retun the approx norm of the vector (equals to 0.96x+0.4y where x>=y>=0)
//   accurate within 4%
double CapyVec2DApproxNorm(CapyVec* const u);

// Get the angle between two 2D vectors
// Input:
//   u: the first 2D vector
//   v: the second 2D vector
// Output:
//   Return the angle between the 2 vectors in [-M_PI, M_PI]
double CapyVec2DGetAngle(
  CapyVec const* const u,
  CapyVec const* const v);

// Set the angle of a 2D vector
// Input:
//   a: vector
//   b: angle in radians
//   c: result vector, c=a*b
// Output:
//   Set the angle to 'b' (relative to x, ccw) while conserving the distance
//   of 'a' and store the result in 'c' (which can be 'a')
void CapyVec2DSetAngle(
  CapyVec const* const a,
          double const b,
        CapyVec* const c);

// Rotate a 2D vector by a given angle
// Input:
//   a: vector to rotate
//   theta: angle in radians
//   b: result vector
// Output:
//   Rotate CCW the vector 'a' by 'theta' and store the result in 'c'
//   (which can be 'a')
void CapyVec2DRotate(
  CapyVec const* const a,
          double const theta,
        CapyVec* const b);

// Get the squared norm of the vector
// Input:
//   a: the vector
// Output:
//   Retun the squared norm of the vector
double CapyVecGetSquaredNorm(CapyVec* const a);

// Get the moment of a vector's values
// Input:
//   u: the vector
//   c: the center of the moment
//   n: the order of the moment
// Output:
//   Return the moment. CapyVecGetMoment(u, 0, 0) is the sum of u's values, aka
//   total mass. CapyVecGetMoment(u, 0, 1) is the mean of u's values, aka
//   first raw moment. CapyVecGetMoment(u, mean(u), 2) is the variance of u's
//   values, aka second centered moment, or square of the standard deviation
//   sigma. CapyVecGetMoment(u, mean(u), 3) is the skewness, aka third centered
//   moment. CapyVecGetMoment(u, mean(u), 4) is the kurtosis, aka fourth
//   centered moment.
//   Standardized moment of order n (aka normalized n-th central moment) is
//   equal to:
//   CapyVecMoment(u, mean(u), n) / sqrt(CapyVecMoment(u, mean(u), 2))^n
double CapyVecGetMoment(
  CapyVec const* const u,
          double const c,
         uint8_t const n);

// Get the k-th largest number in a CapyVec (using quickselect)
// Input:
//   u: the vector
//   k: the index of the element to find
// Output:
//   Return the k-th largest number. Can also be used to get the median value
//   without sorting (CapyVecQuickSelect(u, u->dim/2))
double CapyVecQuickSelect(
  CapyVec const* const u,
          size_t const k);

// Get the median value of vector's values (using quickselect)
// Input:
//   u the vector
// Output:
//   Return the median value (threshold which split the vector values into
//   two sets ('smaller than threshold' and 'larger than threshold') of same
//   size)
double CapyVecGetMedian(CapyVec const* const u);

// Get the covariance of two vectors' values. The two vectors must have same
// dimension.
// Input:
//   u: the first vector
//   v: the second vector
// Output:
//   Return the covariance, equal to E[(u-E(u)(v-E(v))].
// Exception:
//   May raise CapyExc_InvalidParameters.
double CapyVecGetCovariance(
  CapyVec const* const u,
  CapyVec const* const v);

// Get the Pearson correlation of two vector's values. The two vectors must
// have same dimension.
// Input:
//   u: the first vector
//   v: the second vector
// Output:
//   Return the covariance (in [-1,1]), equal to cov(u, v)/(sigma(u)*sigma(v)).
// Exception:
//   May raise CapyExc_InvalidParameters.
double CapyVecGetPearsonCorrelation(
  CapyVec const* const u,
  CapyVec const* const v);

// Get the distance covariance of two vector's values (seen as univariate
// variables). The two vectors must have same dimension.
// Input:
//   u: the first vector
//   v: the second vector
// Output:
//   Return the distance covariance.
// Exception:
//   May raise CapyExc_InvalidParameters.
double CapyVecGetDistanceCovariance(
  CapyVec const* const u,
  CapyVec const* const v);

// Get the distance correlation of two vector's values (seen as univariate
// variables). The two vectors must have same dimension.
// Input:
//   u: the first vector
//   v: the second vector
// Output:
//   Return the covariance (in [0,1]).
// Exception:
//   May raise CapyExc_InvalidParameters.
double CapyVecGetDistanceCorrelation(
  CapyVec const* const u,
  CapyVec const* const v);

// Apply the softmax function to a vector
// Input:
//   u: the vector
//   t: 'temperature'
// Output:
//   The vector is updated. The temperature must be >0.0. A temperature of
//   1.0 gives the standard softmax function. The higher the temperature the
//   more uniformly distributed the result vector is. A temperature value
//   infinitely small produces a vector with value 1.0 for the max value of
//   the input, and 0.0 for all other values.
void CapyVecSoftmax(
  CapyVec* const u,
    double const t);

// Get the distance between two vectors. The two vectors must have same
// dimension.
// Input:
//   u: the first vector
//   v: the second vector
// Output:
//   Return the norm of the difference ofthe two vectors.
// Exception:
//   May raise CapyExc_InvalidParameters.
double CapyVecGetDistance(
  CapyVec const* const u,
  CapyVec const* const v);

// Apply lerp to a vector components
// Input:
//   from: start vector
//   to: end vector
//   res: result vector (can same as 'start' or 'end')
//   t: the coefficient (in [0,1])
// Output:
//   'res' is set to start+t*(end-start)
void CapyVecLerp(
  CapyVec const* const start,
  CapyVec const* const end,
        CapyVec* const res,
          double const t);

// Type for an easing function for CapyVecEasing
typedef double (*CapyVecEasingFun)(
  double const startVal,
  double const endVal,
  double const t);

// Apply easing to a vector components
// Input:
//   from: start vector
//   to: end vector
//   res: result vector (can same as 'start' or 'end')
//   t: the coefficient (in [0,1])
//   easing: the easing function
// Output:
//   'res' is updated
void CapyVecEasing(
  CapyVec const* const start,
  CapyVec const* const end,
        CapyVec* const res,
          double const t,
      CapyVecEasingFun easing);

// Product matrix vector
// Input:
//   a: matrix
//   b: vector
//   c: result vector, c=a*b
void CapyMatProdVec(
  CapyMat const* const a,
  CapyVec const* const b,
        CapyVec* const c);

// Product transposed matrix vector
// Input:
//   a: matrix
//   b: vector
//   c: result vector, c=a^t*b
void CapyMatTransProdVec(
  CapyMat const* const a,
  CapyVec const* const b,
        CapyVec* const c);

// Product matrix matrix
// Input:
//   a: matrix
//   b: matrix
//   c: result matrix, c=a*b
void CapyMatProdMat(
  CapyMat const* const a,
  CapyMat const* const b,
        CapyMat* const c);

// Product scalar matrix
// Input:
//   a: matrix
//   b: scalar
//   c: result matrix, c=a*b
void CapyMatProdScalar(
  CapyMat const* const a,
          double const b,
        CapyMat* const c);

// Add matrix matrix
// Input:
//   a: matrix
//   b: matrix
//   c: result matrix, c=a+b
void CapyMatAddMat(
  CapyMat const* const a,
  CapyMat const* const b,
        CapyMat* const c);

// Transpose matrix
// Input:
//   a: matrix
//   b: result transpose matrix
void CapyMatTransp(
  CapyMat const* const a,
        CapyMat* const b);

// Get the determinant of a matrix
//  a: matrix
//  b: result determinant
void CapyMatDet(
  CapyMat const* const a,
         double* const b);

// Pseudo inverse matrix (Moore-Penrose inverse)
// Input:
//   a: matrix
//   b: result pseudo inverse matrix
// Exceptions:
//   May raise CapyExc_MatrixInversionFailed, CapyExc_MallocFailed
void CapyMatPseudoInv(
  CapyMat const* const a,
        CapyMat* const b);

// Inverse matrix (if the matrix is not square the result is the pseudo
// inverse)
// Input:
//   a: matrix
//   b: result inverse matrix
// Exceptions:
//   May raise CapyExc_MatrixInversionFailed, CapyExc_MallocFailed
void CapyMatInv(
  CapyMat const* const a,
        CapyMat* const b);

// Get the QR decomposition of a matrix
// cf http://www.seas.ucla.edu/~vandenbe/133A/lectures/qr.pdf
// Input:
//   m: the matrix to decompose (nbRow>=nbCol)
//   q: the result Q matrix (same dimensions as m)
//   r: the result R matrix (dimensions: m.nbCol, m.nbCol)
// Exception:
//   May raise CapyExc_QRDecompositionFailed, CapyExc_MallocFailed.
void CapyMatGetQR(
  CapyMat const* const m,
        CapyMat* const q,
        CapyMat* const r);

// Get the Eigen values and vectors of a matrix
// cf http://madrury.github.io/
// jekyll/update/statistics/2017/10/04/qr-algorithm.html
// Input:
//   m: the matrix (nbRow==nbCol)
//   eigenVal: the Eigen values (from largest to smallest in
//             absolute value, dim==m.nbCol)
//   eigenVec: the Eigen vectors (same order as Eigen values, same dimension as
//             m, one per column)
// Output:
//   eigenVec and eigenVal are updated with the result.
// Exception:
//   May raise CapyExc_QRDecompositionFailed, CapyExc_MallocFailed.
void CapyMatGetEigen(
  CapyMat const* const m,
        CapyVec* const eigenVal,
        CapyMat* const eigenVec);

// Set a matrix to the identity.
// Input:
//   m: the matrix (may be rectangular)
void CapyMatSetToIdentity(CapyMat* const m);

// Copy a matrix to another.
// Input:
//    src: the matrix to be copied
//   dest: the matrix updated
void CapyMatCopy(
  CapyMat const* const src,
        CapyMat* const dest);

// Get the moment of a column of a matrix
// Input:
//   u: the vector
//   iCol: the column index
//   c: the center of the moment
//   n: the order of the moment
// Output:
//   Return the moment (cf CapyVecGetMoment)
double CapyMatGetMomentCol(
  CapyMat const* const m,
          size_t const iCol,
          double const c,
         uint8_t const n);

// Calculate (a*b)%c using the square and multiply algorithm
// Input:
//   a,b,c: the value of a, b and c in (a*b)%c
// Ouput:
//   Return (a*b)%c
uint64_t CapyMultMod(
  uint64_t a,
  uint64_t b,
  uint64_t c);

// Calculate (a^b)%c using the square and multiply algorithm
// Input:
//   a,b,c: the value of a, b and c in (a^b)%c
// Ouput:
//   Return (a^b)%c
uint64_t CapyPowMod(
  uint64_t a,
  uint64_t b,
  uint64_t c);

// Return the n-th Fibonacci number.
// Input:
//   n: the index of the element (starting from 0)
// Output:
//   Return the n-th Fibonnaci number.
uint64_t CapyGetFibonacciNumber(uint64_t const n);

// Return the Fibonacci sequence up to the 'n'-th element.
// Input:
//   n: the number of elements
// Output:
//   Return the Fibonnaci sequence in newly allocated array.
uint64_t* CapyFibonacciSeq(uint64_t const n);

// Return the index in the Fibonacci sequence of the smallest value greater
// or equal than a given value.
// Input:
//   val: the value
// Output:
//   Return the index in the Fibonnaci sequence. (i.e: CapyFibonacciIdx(10)=6)
uint64_t CapyFibonacciIdx(uint64_t const val);

// Structure to memorise a Fibonacci lattice (grid or polar)
typedef struct CapyFiboLattice {

  // Number of points in the lattice
  size_t nbPoints;

  // Array of points. For grid lattice, the points are in {[0,1],[0,1]}.
  // For polar lattice, the points are in {[0,1],[0,2pi]}
  // For lattice on sphere and demi-sphere the points are in {[0,pi],[0,2pi]}
  double* points;
} CapyFiboLattice;

// Free a CapyFiboLattice
// Input:
//   that: the CapyFiboLattice to free
void CapyFiboLatticeDestruct(CapyFiboLattice* const that);

// Return the Fibonacci grid lattice for the 'n'-th Fibonacci number
// Input:
//   n: the Fibonacci number
// Output:
//   Return the lattice.
CapyFiboLattice CapyFibonacciGridLattice(uint64_t const n);

// Return the Fibonacci polar lattice for the 'n'-th Fibonacci number
// Input:
//   n: the Fibonacci number
// Output:
//   Return the lattice.
CapyFiboLattice CapyFibonacciPolarLattice(uint64_t const n);

// Get the polar coordinates of n points uniformly distributed on a demi-sphere
// using the Fibonacci sequence
// Input:
//        n: the number of points
// Output:
//   Return the lattice.
//   points[2i] in [0,pi] and points[2i+1] in [0,2pi]
CapyFiboLattice CapyFibonacciDemiSphereLattice(size_t const n);

// Get the polar coordinates of n points uniformly distributed on a sphere
// using the Fibonacci sequence
// Input:
//        n: the number of points
// Output:
//   Return the lattice.
//   points[2i] in [0,pi] and points[2i+1] in [0,2pi]
CapyFiboLattice CapyFibonacciSphereLattice(size_t const n);

// Solve the quadratic equation a+bx+cx^2=0
// Input:
//   coeffs: the coefficients of the equation (in order a,b,...)
//    roots: array of size 2 to memorise the roots
// Output:
//   Return true and update 'roots' (sorted by increasing values) if there is a
//   solution, else return false and leave 'roots' unchanged. If there are less
//   roots than the maximum possible number, the smallest root is repeated to
//   fill in 'roots'.
bool CapySolveQuadratic(
  double const* const coeffs,
        double* const roots);

// Solve the cubic equation a+bx+cx^2+d^3=0
// Input:
//   coeffs: the coefficients of the equation (in order a,b,...)
//    roots: array of size 3 to memorise the roots
// Output:
//   Return true and update 'roots' (sorted by increasing values) if there is a
//   solution, else return false and leave 'roots' unchanged. If there are less
//   roots than the maximum possible number, the smallest root is repeated to
//   fill in 'roots'.
bool CapySolveCubic(
  double const* const coeffs,
        double* const roots);

// Solve the quartic equation a+bx+cx^2+dx^3+ex^4=0
// Input:
//   coeffs: the coefficients of the equation (in order a,b,...)
//    roots: array of size 4 to memorise the roots
// Output:
//   Return true and update 'roots' (sorted by increasing values) if there is a
//   solution, else return false and leave 'roots' unchanged. If there are less
//   roots than the maximum possible number, the smallest root is repeated to
//   fill in 'roots'.
bool CapySolveQuartic(
  double const* const coeffs,
        double* const roots);

// Get the approximated inverse square root of a number using the Quake
// algorithm (cf https://en.wikipedia.org/wiki/Fast_inverse_square_root)
// Input:
//   x: the number
// Output:
//   Return 1/sqrt(x).
float CapyFastInverseSquareRoot(float x);

// Get the approximated exponential of a number using the Quake
// algorithm (cf https://specbranch.com/posts/fast-exp/)
// Input:
//   x: the number
// Output:
//   Return exp(x).
float CapyFastExponential(float x);

// Convert from degree to radians
// Input:
//   theta: the angle in degree
// Output:
//   Return the ange in radians.
double CapyDegToRad(double const theta);

// Convert from radians to degree
// Input:
//   theta: the angle in radians
// Output:
//   Return the ange in degree.
double CapyRadToDeg(double const theta);

// ----- PIECEWISE GAUSSIAN -----

// Piecewise Gaussian
typedef struct CapyPiecewiseGaussian {
  double base, amp, mean, sig1, sig2;
} CapyPiecewiseGaussian;

// Piecewise Gaussian evaluation
// Input:
//   x: argument of the Gaussian
//   gauss: the Gaussian
// Output:
//   Return the value of the piecewise Gaussian at the requested argument
double CapyPiecewiseGaussianEval(
  CapyPiecewiseGaussian const* const gauss,
                        double const x);

// Ackley's function
// Input:
//   in: 2D input
//   out: 1D output
// Output:
//   'out' is updated. Cf https://en.wikipedia.org/wiki/Ackley_function
void CapyAckley(
  double const* const in,
        double* const out);

// Himmelblau's function
// Input:
//   in: 2D input
//   out: 1D output
// Output:
//   'out' is updated. Cf https://en.wikipedia.org/wiki/Himmelblau%27s_function
void CapyHimmelblau(
  double const* const in,
        double* const out);

// Check if a position is inside an ellipse (aligned with cooridnate system)
// Input:
//   pos: the position to check
//   center: the center of the ellipse
//   dims: the dimensions of the ellise
// Output:
//   Return true if pos is inside the ellipse, false else
bool CapyIsInsideEllipse(
  double const* const pos,
  double const* const center,
  double const* const dims);

// Calculate the value of a cubic Bezier curve
// Inputs:
//   t: argument of the function, in [0, 1]
//   params: the 4 control values of the Bezier
// Output:
//   Return the value of the Bezier
double CapyCubicBezierEval(
  double const t,
  double const params[4]);

// Quadratic easing acceleration
// Input:
//   from: start value
//   to: end value
//   t: control (in [0,1])
// Output:
//   Return the eased value
double CapyEaseQuadraticAccel(
  double const from,
  double const to,
  double const t);

// Quadratic easing deceleration
// Input:
//   from: start value
//   to: end value
//   t: control (in [0,1])
// Output:
//   Return the eased value
double CapyEaseQuadraticDecel(
  double const from,
  double const to,
  double const t);

// Quadratic easing acceleration first hald deceleration second half
// Input:
//   from: start value
//   to: end value
//   t: control (in [0,1])
// Output:
//   Return the eased value
double CapyEaseQuadraticAccelDecel(
  double const from,
  double const to,
  double const t);

// Quadratic easing deceleration first hald acceleration second half
// Input:
//   from: start value
//   to: end value
//   t: control (in [0,1])
// Output:
//   Return the eased value
double CapyEaseQuadraticDecelAccel(
  double const from,
  double const to,
  double const t);

// Conversion from acre to square meter
// Input:
//   areaAcre: the area in acre
// Output:
//   Return the area in square meter
double CapyConvAcreSquareMeter(double const areaAcre);

// Conversion from square meter to acre
// Input:
//   areaSquareMeter: the area in sqaure meter
// Output:
//   Return the area in acre
double CapyConvSquareMeterAcre(double const areaSquareMeter);

// Flory decomposition (aka Volumetric-deviatoric decomposition) of a matrix
// Input:
//   mat: the matrix to decompose
//   vol: the volumetric matrix
//   dev: the deviatoric matrix
// Output:
//   'vol' and 'dev' are updated. 'vol' represent the change in volume
//   (identity is no change) and 'dev' represent the change in shape
//   (identity is no change). All matrices must be square.
void CapyFloryDecomposition(
  CapyMat const* const mat,
        CapyMat* const vol,
        CapyMat* const dev);
#endif
