// ---------------------------- catmullromspline.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "catmullromspline.h"

// Definition of the memory pool for the iteration windows
CapyDefMemPool(
  CapyMemPoolCatmullRomIteratorWindow, CapyCatmullRomIteratorWindow)

// Evaluate the CapyCatmullRom for a given input
// Input:
//    in: the input vector (dimension 1, value in [0, 1], 0.0 gives the second
//        control points, 1.0 gives the third control points)
//   out: array of 'that->dimOut' double
// Output:
//   'out' is updated.
static void Eval(
  double const* const in,
        double* const out) {
  methodOf(CapyCatmullRom);
  CapyRangeDouble from = {.min = 0.0, .max = 1.0};
  CapyRangeDouble to = {.min = that->knots[1], .max = that->knots[2]};
  double const t = CapyLerp(*in, &from, &to);
  double const t2tt2t1 =
    (that->knots[2] - t) / (that->knots[2] - that->knots[1]);
  double const tt1t2t1 =
    (t - that->knots[1]) / (that->knots[2] - that->knots[1]);
  double const t2tt2t0 =
    (that->knots[2] - t) / (that->knots[2] - that->knots[0]);
  double const tt0t2t0 =
    (t - that->knots[0]) / (that->knots[2] - that->knots[0]);
  double const t3tt3t1 =
    (that->knots[3] - t) / (that->knots[3] - that->knots[1]);
  double const tt1t3t1 =
    (t - that->knots[1]) / (that->knots[3] - that->knots[1]);
  double const t1tt1t0 =
    (that->knots[1] - t) / (that->knots[1] - that->knots[0]);
  double const tt0t1t0 =
    (t - that->knots[0]) / (that->knots[1] - that->knots[0]);
  double const t3tt3t2 =
    (that->knots[3] - t) / (that->knots[3] - that->knots[2]);
  double const tt2t3t2 =
    (t - that->knots[2]) / (that->knots[3] - that->knots[2]);
  loop(iDim, that->dimOut) {
    double const a2 =
      t2tt2t1 * that->ctrls[1].vals[iDim] + tt1t2t1 * that->ctrls[2].vals[iDim];
    out[iDim] =
      t2tt2t1 * (
        t2tt2t0 * (
          t1tt1t0 * that->ctrls[0].vals[iDim] +
          tt0t1t0 * that->ctrls[1].vals[iDim]
        ) + tt0t2t0 * a2
      ) + tt1t2t1 * (
        t3tt3t1 * a2 +
        tt1t3t1 * (
          t3tt3t2 * that->ctrls[2].vals[iDim] +
          tt2t3t2 * that->ctrls[3].vals[iDim])
      );
  }
}

// Get a control point.
// Input:
//   id: the index of the control point, in [0,3]
// Output:
//   Return the control point
static CapyVec const* GetCtrl(size_t const id) {
  methodOf(CapyCatmullRom);
  return that->ctrls + id;
}

// Set the values of a control point
// Input:
//    id: the index of the control point
//   vals: the value to set, array of 'that->dimOut' double
// Output:
//   The control point values and the knot values are updated.
static void SetCtrl(
         size_t const id,
  double const* const vals) {
  methodOf(CapyCatmullRom);
  loop(iDim, that->dimOut) that->ctrls[id].vals[iDim] = vals[iDim];
  that->knots[0] = 0.0;
  loop(iKnot, 3) {
    double const dist =
      CapyVecGetDistance(that->ctrls + iKnot, that->ctrls + iKnot + 1);
    that->knots[iKnot + 1] = that->knots[iKnot] + pow(dist, that->alpha);
  }
}

// Free the memory used by a CapyCatmullRom
static void Destruct(void) {
  methodOf(CapyCatmullRom);
  $(that, destructCapyMathFun)();
  loop(i, 4) CapyVecDestruct(that->ctrls + i);
}

// Create a CapyCatmullRom
// Input:
//   dim: dimension of the control points
// Output:
//   Return a CapyCatmullRom
CapyCatmullRom CapyCatmullRomCreate(size_t const dim) {
  CapyCatmullRom that = {
    .alpha = 0.5,
    .setCtrl = SetCtrl,
    .getCtrl = GetCtrl,
  };
  CapyInherits(that, CapyMathFun, (1, dim));
  that.destruct = Destruct,
  that.eval = Eval;
  loop(i, 4) that.ctrls[i] = CapyVecCreate(dim);
  return that;
}

// Allocate memory for a new CapyCatmullRom and create it
// Input:
//   dim: dimension of the control points
// Output:
//   Return a CapyCatmullRom
// Exception:
//   May raise CapyExc_MallocFailed.
CapyCatmullRom* CapyCatmullRomAlloc(size_t const dim) {
  CapyCatmullRom* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyCatmullRomCreate(dim);
  return that;
}

// Free the memory used by a CapyCatmullRom* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyCatmullRom to free
void CapyCatmullRomFree(CapyCatmullRom** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Check if the current CapyCatmullRomIteratorWindow of an iterator is closed
// Input:
//   that: the iterator
// Output:
//   Return true if the window is closed, false else
static bool IsIteratorWindowClosed(CapyCatmullRomIterator const* const that) {
  double dist = 0.0;
  loop(i, that->spline->dimOut) {
    double x = that->windows->pos[0].out[i] - that->windows->pos[1].out[i];
    dist += x * x;
  }
  dist = sqrt(dist);
  return (
    that->windows->pos[1].in[0] - that->windows->pos[0].in[0] < 1.0 &&
    dist < that->epsilon
  );
}

// Split the current iteration window
// Input:
//   that: the iterator
// Output:
//   The current iteration window is split until it is closed
static void SplitIteratorWindow(CapyCatmullRomIterator* const that) {
  while(IsIteratorWindowClosed(that) == false) {
    CapyCatmullRomIteratorWindow* window = $(&(that->memPool), alloc)(NULL);
    window->pos[0].in[0] = that->windows->pos[0].in[0];
    window->pos[1].in[0] =
      0.5 * (that->windows->pos[0].in[0] + that->windows->pos[1].in[0]);
    $(that->spline, eval)(window->pos[0].in, window->pos[0].out);
    $(that->spline, eval)(window->pos[1].in, window->pos[1].out);
    that->windows->pos[0].in[0] = window->pos[1].in[0];
    loop(i, that->spline->dimOut) {
      that->windows->pos[0].out[i] = window->pos[1].out[i];
    }
    window->next = that->windows;
    that->windows = window;
  }
}

// Reset the iterator
// Output:
//   Return the first pixel of the iteration
static CapyCatmullRomPosition* IterReset(void) {
  methodOf(CapyCatmullRomIterator);

  // Reset the index
  that->idx = 0;

  // Reset the iterator
  while(that->windows != NULL) {
    CapyCatmullRomIteratorWindow* ptr = that->windows->next;
    $(&(that->memPool), free)(&(that->windows));
    that->windows = ptr;
  }
  that->windows = $(&(that->memPool), alloc)(NULL);
  that->windows->next = NULL;
  loop(i, 2) {
    that->windows->pos[i].in[0] = i;
    $(that->spline, eval)(that->windows->pos[i].in, that->windows->pos[i].out);
  }
  SplitIteratorWindow(that);

  // Return the current position
  return that->windows->pos;
}

// Move the iterator to the next position
// Output:
//   Return the next position of the iteration
static CapyCatmullRomPosition* IterNext(void) {
  methodOf(CapyCatmullRomIterator);

  // Increment the index
  ++(that->idx);

  // Step the iterator
  if(that->windows->next) {
    CapyCatmullRomIteratorWindow* ptr = that->windows;
    that->windows = ptr->next;
    $(&(that->memPool), free)(&ptr);
  }
  SplitIteratorWindow(that);

  // Return the current position
  return that->windows->pos;
}

// Check if the iterator is on a valid element
// Output:
//   Return true if the iterator is on a valid element, else false
static bool IterIsActive(void) {
  methodOf(CapyCatmullRomIterator);
  return (IsIteratorWindowClosed(that) == false || that->windows->next != NULL);
}

// Get the current position of the iteration
// Output:
//   Return the current position
static CapyCatmullRomPosition* IterGet(void) {
  methodOf(CapyCatmullRomIterator);
  return that->windows->pos;
}

// Free the memory used by an iterator.
static void IteratorDestruct(void) {
  methodOf(CapyCatmullRomIterator);
  $(&(that->memPool), destruct)();
}

// Create an iterator on a CapyCatmullRom
// Input:
//   spline: the spline on which to iterate
//   type: the type of iterator
// Output:
//   Return the iterator
CapyCatmullRomIterator CapyCatmullRomIteratorCreate(
  CapyCatmullRom const* const spline) {

  // Ensure the input idmension of the spline is equal to 1
  if(spline->dimIn != 1) raiseExc(CapyExc_UndefinedExecution);

  // Create the new iterator
  CapyCatmullRomIterator that = {
    .idx = 0,
    .spline = spline,
    .windows = NULL,
    .epsilon = 0.1,
    .memPool =
      CapyMemPoolCatmullRomIteratorWindowCreate(
        sizeof(CapyCatmullRomIteratorWindow)),
    .destruct = IteratorDestruct,
    .reset = IterReset,
    .next = IterNext,
    .isActive = IterIsActive,
    .get = IterGet,
  };
  (void)$(&that, reset)();

  // Return the new iterator
  return that;
}

// Allocate memory and create an iterator on a CapyCatmullRom
// Input:
//   spline: the spline on which to iterate
//   type: the type of iterator
// Output:
//   Return the iterator
CapyCatmullRomIterator* CapyCatmullRomIteratorAlloc(
  CapyCatmullRom const* const spline) {
#if BUILD_MODE == 0
  assert(spline != NULL);
#endif

  // Declare the new iterator
  CapyCatmullRomIterator* that = NULL;

  // Allocate memory
  safeMalloc(that, 1);
  if(!that) return NULL;

  // Create the iterator
  *that = CapyCatmullRomIteratorCreate(spline);

  // Return the new iterator
  return that;
}

// Free the memory used by a pointer to an iterator and reset '*that' to NULL
// Input:
//   that: a pointer to the iterator to free
void CapyCatmullRomIteratorFree(CapyCatmullRomIterator** const that) {

  // If the iterator is already freed, nothing to do
  if(that == NULL || *that == NULL) return;

  // Destruct the iterator
  $(*that, destruct)();

  // Free memory
  free(*that);
  *that = NULL;
}

