// ---------------------------- catmullromspline.h ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_CATMULLROMSPLINE_H
#define CAPY_CATMULLROMSPLINE_H
#include "externalHeaders.h"
#include "cext.h"
#include "capymath.h"
#include "mathfun.h"
#include "memoryPool.h"
#include "list.h"

// Description:
// CatmullRom class.

// Maximum dimension for a CapyCatmullRom
#define CAPY_CATMULLROM_MAX_DIMOUT 254

// CatmullRom object
typedef struct CapyCatmullRom {

  // Inherits CapyMathFun
  struct CapyMathFunDef;

  // Alpha parameter defining the type of spline (default: 0.5, in [0,1],
  // 0: uniform, 0.5: centripetal, 1: chordal) 
  double alpha;

  // Control points.
  CapyVec ctrls[4];

  // Knots value (automatically updated when setting the control points)
  double knots[4];

  // Destructor
  void (*destructCapyMathFun)(void);

  // Get a control point.
  // Input:
  //   id: the index of the control point, in [0,3]
  // Output:
  //   Return the control point
  CapyVec const* (*getCtrl)(size_t const id);

  // Set the values of a control point
  // Input:
  //    id: the index of the control point
  //   vals: the value to set, array of 'that->dimOut' double
  // Output:
  //   The control point values and the knot values are updated.
  void (*setCtrl)(
           size_t const id,
    double const* const vals);
} CapyCatmullRom;

// Create a CapyCatmullRom
// Input:
//   dim: dimension of the control points
// Output:
//   Return a CapyCatmullRom
CapyCatmullRom CapyCatmullRomCreate(size_t const dim);

// Allocate memory for a new CapyCatmullRom and create it
// Input:
//   dim: dimension of the control points
// Output:
//   Return a CapyCatmullRom
// Exception:
//   May raise CapyExc_MallocFailed.
CapyCatmullRom* CapyCatmullRomAlloc(size_t const dim);

// Free the memory used by a CapyCatmullRom* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyCatmullRom to free
void CapyCatmullRomFree(CapyCatmullRom** const that);

// Structure to memorise the current position (inputs and outputs) of a
// CapyCatmullRomIterator
typedef struct CapyCatmullRomPosition {
  double in[1];
  double out[CAPY_CATMULLROM_MAX_DIMOUT];
} CapyCatmullRomPosition;

// Iteration windows for CapyCatmullRomIterator
typedef struct CapyCatmullRomIteratorWindow CapyCatmullRomIteratorWindow;
struct CapyCatmullRomIteratorWindow {
  CapyCatmullRomPosition pos[2];
  CapyCatmullRomIteratorWindow* next;
  void (*destruct)(void);
  CapyMemPoolFields(CapyCatmullRomIteratorWindow);
};

// Declaration of the memory pool for the iteration windows
CapyDecMemPool(
  CapyMemPoolCatmullRomIteratorWindow, CapyCatmullRomIteratorWindow)

// Iterator on the positions of a CapyCatmullRom structure.
typedef struct CapyCatmullRomIterator {

  // Index of the current step in the iteration
  size_t idx;

  // Returned data type
  CapyCatmullRomPosition datatype;

  // List of windows
  CapyCatmullRomIteratorWindow* windows;

  // CatmullRom associated to the iteration
  CapyCatmullRom const* spline;

  // Epsilon for the step of iteration using euclidean distance along the
  // CatmullRom curve (default: 0.1)
  double epsilon;

  // Memory pool for the windows
  CapyMemPoolCatmullRomIteratorWindow memPool;

  // Destructor
  void (*destruct)(void);

  // Reset the iterator
  // Output:
  //   Return the first position of the iteration
  CapyCatmullRomPosition* (*reset)(void);

  // Move the iterator to the next position
  // Output:
  //   Return the next position of the iteration
  CapyCatmullRomPosition* (*next)(void);

  // Check if the iterator is on a valid position
  // Output:
  //   Return true if the iterator is on a valid position, else false
  bool (*isActive)(void);

  // Get the current position of the iteration
  // Output:
  //   Return the current position
  CapyCatmullRomPosition* (*get)(void);
} CapyCatmullRomIterator;

// Create an iterator on a CapyCatmullRom
// Input:
//   bezier: the bezier on which to iterate
// Output:
//   Return the iterator
CapyCatmullRomIterator CapyCatmullRomIteratorCreate(
  CapyCatmullRom const* const bezier);

// Allocate memory and create an iterator on a CapyCatmullRom
// Input:
//   bezier: the bezier on which to iterate
// Output:
//   Return the iterator
CapyCatmullRomIterator* CapyCatmullRomIteratorAlloc(
  CapyCatmullRom const* const bezier);

// Free the memory used by a pointer to an iterator and reset '*that' to NULL
// Input:
//   that: a pointer to the iterator to free
void CapyCatmullRomIteratorFree(
  CapyCatmullRomIterator** const that);
#endif
