// ---------------------------------- cext.c ----------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "cext.h"

// Variable to emulate the 'that' pointer in methods
_Thread_local void const* capyThat;

// Variable internally used by LibCapy to handle returned values
_Thread_local int capyRetInt[2];

// Safe fopen raising CapyExc_StreamOpenError if it fails
FILE* safeFOpen(
  char const* const pathname,
  char const* const mode) {
#if BUILD_MODE == 0
  assert(pathname != NULL);
  assert(mode != NULL);
#endif
  FILE* fp = fopen(pathname, mode);
  if(fp == NULL) raiseExc(CapyExc_StreamOpenError);
  return fp;
}

// Safe sprintf allocating memory as necessary for the result string and
// raising CapyExc_MallocFailed if the memory allocation failed
// Input:
//   str: pointer to the result string
//   fmt: format as in sprintf
//   ...: arguments as in sprintf
void safeSPrintf(
        char** const str,
   char const* const fmt,
                     ...) {
#if BUILD_MODE == 0
  assert(str != NULL);
  assert(fmt != NULL);
#endif

  // Get the length of the result string by creating in a one char buffer
  // vsnprintf will fails due to the insufficient space in the buffer but
  // return instead the necessary size of the buffer
  char c[1];
  va_list argp;
  va_start(argp, fmt);
  int len = vsnprintf(c, 1, fmt, argp) + 1;
  va_end(argp);

  // If an error occured 'len' is negative, stops here
  if(len < 0) {
    raiseExc(CapyExc_UndefinedExecution);
    return;
  }

  // Allocate memory for the result string
  safeMalloc(*str, (size_t)len);
  if(!*str) return;

  // Create the string
  va_start(argp, fmt);
  vsnprintf(*str, (size_t)len, fmt, argp);
  va_end(argp);
}

// Equality operator for floating point values
// Input:
//   a,b: the values to compare
// Output:
//   Return true if the values are considered equal using the ULP
//   method described here:
//   https://randomascii.wordpress.com/2012/02/25/
//   comparing-floating-point-numbers-2012-edition/
bool equalf(
  float const a,
  float const b) {

  // Check for equality near zero
  float absDiff = fabsf(a - b);
  float maxDiff = FLT_EPSILON * 10.0;
  if(absDiff <= maxDiff) return true;

  // Interpretation of float as int
  union {float f; int32_t i;} ua = {.f = a};
  union {float f; int32_t i;} ub = {.f = b};

  // Check for sign difference
  if((ua.i < 0) != (ub.i < 0)) return false;

  // Check for equality based on ULP for number away from zero
  int32_t ulpsDiff = abs(ua.i - ub.i);
  int32_t maxUlpsDiff = 10;
  if(ulpsDiff <= maxUlpsDiff) return true;

  // If we reach here numbers are not equals
  return false;
}

bool equald(
  double const a,
  double const b) {

  // Check for equality near zero
  double absDiff = fabs(a - b);
  double maxDiff = DBL_EPSILON * 10.0;
  if(absDiff <= maxDiff) return true;

  // Interpretation of double as int
  union {double f; int64_t i;} ua = {.f = a};
  union {double f; int64_t i;} ub = {.f = b};

  // Check for sign difference
  if((ua.i < 0) != (ub.i < 0)) return false;

  // Check for equality based on ULP for number away from zero
  int64_t ulpsDiff = llabs(ua.i - ub.i);
  int64_t maxUlpsDiff = 10;
  if(ulpsDiff <= maxUlpsDiff) return true;

  // If we reach here numbers are not equals
  return false;
}

// Get the minimum value in a array of base type values
// Input:
//    arr: the array
//   size: the size of the array
// Output:
//   Return the minimum value
#define CapyMinDef(type)                                \
type min_ ## type(type const* const arr, size_t size) { \
  type min = arr[size - 1];                             \
  loop(i, size - 1)                                     \
    if(min > arr[i]) min = arr[i];                      \
  return min;                                           \
}

CapyMinDef(char)
CapyMinDef(int8_t)
CapyMinDef(uint8_t)
CapyMinDef(int16_t)
CapyMinDef(uint16_t)
CapyMinDef(int32_t)
CapyMinDef(uint32_t)
CapyMinDef(int64_t)
CapyMinDef(uint64_t)
CapyMinDef(float)
CapyMinDef(double)

// Get the maximum value in a array of base type values
// Input:
//    arr: the array
//   size: the size of the array
// Output:
//   Return the maximum value
#define CapyMaxDef(type)                                \
type max_ ## type(type const* const arr, size_t size) { \
  type max = arr[size - 1];                             \
  loop(i, size - 1)                                     \
  if(max < arr[i]) max = arr[i];                        \
  return max;                                           \
}

CapyMaxDef(char)
CapyMaxDef(int8_t)
CapyMaxDef(uint8_t)
CapyMaxDef(int16_t)
CapyMaxDef(uint16_t)
CapyMaxDef(int32_t)
CapyMaxDef(uint32_t)
CapyMaxDef(int64_t)
CapyMaxDef(uint64_t)
CapyMaxDef(float)
CapyMaxDef(double)

// Get the minimum value in a array of base type values
// Input:
//    arr: the array
//   size: the size of the array
// Output:
//   Return the minimum value
#define CapyiMinDef(type)                                  \
size_t iMin_ ## type(type const* const arr, size_t size) { \
  type min = arr[size - 1];                                \
  size_t imin = size - 1;                                  \
  loop(i, size - 1)                                        \
    if(min > arr[i]) {min = arr[i]; imin = i;}             \
  return imin;                                             \
}

CapyiMinDef(char)
CapyiMinDef(int8_t)
CapyiMinDef(uint8_t)
CapyiMinDef(int16_t)
CapyiMinDef(uint16_t)
CapyiMinDef(int32_t)
CapyiMinDef(uint32_t)
CapyiMinDef(int64_t)
CapyiMinDef(uint64_t)
CapyiMinDef(float)
CapyiMinDef(double)

// Get the maximum value in a array of base type values
// Input:
//    arr: the array
//   size: the size of the array
// Output:
//   Return the maximum value
#define CapyiMaxDef(type)                                  \
size_t iMax_ ## type(type const* const arr, size_t size) { \
  type max = arr[size - 1];                                \
  size_t imax = size - 1;                                  \
  loop(i, size - 1)                                        \
  if(max < arr[i]) {max = arr[i]; imax = i; }              \
  return imax;                                             \
}

CapyiMaxDef(char)
CapyiMaxDef(int8_t)
CapyiMaxDef(uint8_t)
CapyiMaxDef(int16_t)
CapyiMaxDef(uint16_t)
CapyiMaxDef(int32_t)
CapyiMaxDef(uint32_t)
CapyiMaxDef(int64_t)
CapyiMaxDef(uint64_t)
CapyiMaxDef(float)
CapyiMaxDef(double)

// Clone of asprintf
// Input:
//   fmt: format as in sprintf
//   ...: arguments as in sprintf
// Output:
//   Return a newly allocated string
// Exception:
//   May raise CapyExc_MallocFailed
char* strCreate(
   char const* fmt,
               ...) {
#if BUILD_MODE == 0
  assert(fmt != NULL);
#endif

  // The result string
  char* str = NULL;

  // Get the length of the string
  char c[1];
  va_list argp;
  va_start(argp, fmt);
  int len = vsnprintf(c, 1, fmt, argp) + 1;
  va_end(argp);

  // If there was an error 'len' is negative
  if(len < 0) {
    raiseExc(CapyExc_UndefinedExecution);
    return NULL;
  }

  // Allocate memory
  safeMalloc(str, (size_t)len);
  if(!str) return NULL;

  // Create the string
  va_start(argp, fmt);
  vsnprintf(str, (size_t)len, fmt, argp);
  va_end(argp);

  // Return the string
  return str;
}

// Sleep for a given amount of time in milliseconds
// Input:
//   delayMs: delay in milliseconds
// Output:
//  Return -1 if the sleep has been interrupted by an interruption, else 0
int CapySleepMs(int32_t const delayMs) {

  // Convert the delay to a struct timespec
  Timespec ts = {
    .tv_sec = delayMs / 1000,
    .tv_nsec = (delayMs % 1000) * 1000000,
  };
  return nanosleep(&ts, &ts);
}

// Check if an address is inside the currently accessible address space
// Input:
//      ptr: the address to check
//   nbByte: the number of bytes checked from that adress, if equal to 0 uses
//           1 byte instead
// Output:
//   Return true if the 'nbByte' bytes from 'ptr' are in the accessible
//   adress space, else false
bool CapyIsAccessibleMem(
  void const* const ptr,
       size_t const nbByte) {
  if(ptr == NULL) return false;
  int fd[2];
  bool ret = true;
  int retPipe = pipe(fd);
  assert(retPipe == 0);
  if(write(fd[1], ptr, (nbByte == 0 ? 1 : nbByte)) < 0 && errno == EFAULT) {
    ret = false;
  }
  loop(i, 2) close(fd[i]);
  return ret;
}

// Get the quantity of memory currently used by the process calling this
// function.
// Output:
//   Return the quantity of memory in bytes. May return 0 if the quantity
//   of used memory couldn't be measured.
long CapyGetMemUsed(void) {
  struct rusage rsrc;
  int ret = getrusage(RUSAGE_SELF, &rsrc);
  if(ret != 0) return 0;
  return rsrc.ru_maxrss;
}

// Avoid child process to become zombies and wait until their parent's wait()
// call. This applies to *all* child processes.
void CapyChildProcEndsWithoutWait(void) {
  struct sigaction sa;
  memset(&sa, 0, sizeof(Sigaction));
  sa.sa_handler = SIG_DFL;
  sa.sa_flags = SA_NOCLDWAIT;
  sigaction(SIGCHLD, &sa, NULL);
}

// Check if the architecture is big endian
// Output:
//   Return true if the architecture is big endian, else false. If using
//   gcc one can also use:
//   __BYTE_ORDER__ == __ORDER_BIG_ENDIAN__
bool CapyIsBigEndian(void) {
  uint16_t const a = 0x0100;
  return (*((uint8_t*)&a) == 1);
}
