// ---------------------------------- cext.h ----------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_CEXT_H
#define CAPY_CEXT_H
#include "externalHeaders.h"
#include "trycatch.h"

// Description:
// Functions and macros extending the C language.

// Variable internally used by LibCapy to handle returned values
extern _Thread_local int capyRetInt[2];

// Conversion of a macro into a char* containing its value
#define CAPY_MACRO_TO_STR_(x) #x
#define CAPY_MACRO_TO_STR(x) CAPY_MACRO_TO_STR_(x)

// Safe malloc, allocates 'size*sizeof(*ptr)' bytes of memory and raises
// CapyExc_MallocFailed if the allocation fails.
#define safeMalloc(ptr, size)                        \
  do {                                               \
    ptr = malloc(sizeof(*(ptr)) * (size));           \
    if(ptr == NULL) {                                \
      raiseExc(CapyExc_MallocFailed);                \
      assert(false && "malloc failed uncatched");    \
    }                                                \
  } while(false)

// Safe realloc, raises CapyExc_MallocFailed if the reallocation fails
// and leaves 'ptr' unchanged if it fails
#define safeRealloc(ptr, size)                        \
  do {                                                \
    void* ptr_ = realloc(ptr, sizeof(*ptr) * (size)); \
    if(ptr_ == NULL) raiseExc(CapyExc_MallocFailed);  \
    else ptr = ptr_;                                  \
  } while(false)

// Shortcut equivalent to a for loop iterating from 0 to (nbIter-1)
// and storing the current iteration index in varName
#define loop(varName, nbIter)                 \
  for(__typeof__((nbIter) + 0) (varName) = 0; \
       (varName) < (nbIter); ++(varName))

// Macro used to create padding fields
// Input:
//   T: the type of the field to padd
//   I: index to differentiate several padding field in the same structure
#define CapyPad(T, I) char padding ## I \
  [sizeof(void*) * ((sizeof(void*) + sizeof(T)) / sizeof(void*)) - sizeof(T)]

// Shortcut equivalent to a for loop iterating on a CapyRange of indices
// and storing the current iteration index in varName (range's bound
// are inclusives)
#include "range.h"
#define loopRange(varName, range)                       \
  for(__typeof__((range).min) (varName) = (range).min;  \
    (varName) <= (range).max; ++(varName))

// Macros equivalent to printf and fprintf where a '\n' would have been
// added at the end of the formatting string
#define println(...)                                          \
  ((capyRetInt[0] = printf(__VA_ARGS__)) < 0 ? capyRetInt[0]: \
    ((capyRetInt[1] = printf("\n")) < 0 ? capyRetInt[1] :     \
      capyRetInt[0] + capyRetInt[1]))
#define fprintln(stream, ...)                                          \
  ((capyRetInt[0] = fprintf(stream, __VA_ARGS__)) < 0 ? capyRetInt[0]: \
    ((capyRetInt[1] = fprintf(stream, "\n")) < 0 ? capyRetInt[1] :     \
    capyRetInt[0] + capyRetInt[1]))

// Variable to emulate the 'that' pointer in methods
extern _Thread_local void const* capyThat;

// Macro to define a class method. To be added as the beginning of a function
// 'f' used as a method of class 'C'.
#define methodOf(C) C* that = (C*)capyThat

// Macro to use OOP methods. Given a structure 'S' with a method defined as
// pointer to function as field 'm', an instance 's' of 'S' can execute this
// method with the command $(s, m)() (where 's' must be a pointer to 'S')
// The function pointed to by 'm' can access the instance at the origin of
// the call by defining the macro
// #define CapyThatS struct S* that = (struct S*)capyThat
// and calling this macro at the head of the function, before any call to
// another $(..., ...). A pointer to the instance is then available in the
// body of function through the variable 'that'.
// The macro CapyThat... must be defined for each structure using OOP methods.
// LibCapy provide this macro for each of the structure it defines.
#define $(instance, method) \
  ((__typeof__(instance))(capyThat = (instance)))->method

// Same as $(instance, method) but the value of 'that' in the execution of
// the method is replaced with 'actor' instead of 'instance'.
// Used to implement genericity.
#define $$(instance, actor, method) \
  ((capyThat = (actor)) ? (instance)->method : NULL)

// Safe fopen raising CapyExc_StreamOpenError if it fails
FILE* safeFOpen(
  char const* const pathname,
  char const* const mode);

// Safe fscanf raising CapyExc_StreamReadError if it fails
#define safeFScanf(stream, format, ...)                 \
  do {                                                  \
    int capyRet = fscanf(stream, format, __VA_ARGS__);  \
    if(capyRet < 0) raiseExc(CapyExc_StreamReadError);  \
  } while(false)

// Safe fprintf raising CapyExc_StreamWriteError if it fails
#define safeFPrintf(stream, format, ...)                 \
  do {                                                   \
    int capyRet = fprintf(stream, format, __VA_ARGS__);  \
    if(capyRet < 0) raiseExc(CapyExc_StreamWriteError);  \
  } while(false)

// Safe fread raising CapyExc_StreamReadError if it fails
#define safeFRead(stream, nbByte, ptr)                        \
  do {                                                        \
    size_t capyRet = fread(ptr, 1, nbByte, stream);           \
    if(capyRet != nbByte) raiseExc(CapyExc_StreamReadError);  \
  } while(false)

// Safe fwrite raising CapyExc_StreamWriteError if it fails
#define safeFWrite(stream, nbByte, ptr)                        \
  do {                                                         \
    size_t capyRet = fwrite(ptr, 1, nbByte, stream);           \
    if(capyRet != nbByte) raiseExc(CapyExc_StreamWriteError);  \
  } while(false)

// Safe sprintf allocating memory as necessary for the result string and
// raising CapyExc_MallocFailed if the memory allocation failed
// Input:
//   str: pointer to the result string
//   fmt: format as in sprintf
//   ...: arguments as in sprintf
void safeSPrintf(
        char** const str,
   char const* const fmt,
                     ...);

// Return the number of arguments of a variadic macro given
// the type 'type' of these arguments
#define CAPY_VA_NB_ARGS(type, ...) \
  (size_t)(sizeof((type[]){__VA_ARGS__})/sizeof(type))

// Equality operator for float and double types.
// Input:
//   a,b: the values to compare
// Output:
//   Return true if the values are considered
//   equal using the ULP method described here:
//   https://randomascii.wordpress.com/2012/02/25/
//   comparing-floating-point-numbers-2012-edition/
#define equal(a, b) _Generic(a, \
  float: equalf,                \
  float const: equalf,          \
  double: equald,               \
  double const: equald)(a, b)
bool equalf(
  float const a,
  float const b);
bool equald(
  double const a,
  double const b);

// Get the minimum value in a array of base type values
// Input:
//    arr: the array
//   size: the size of the array
// Output:
//   Return the minimum value
#define min(arr, size) _Generic(arr, \
  char*: min_char,                   \
  char const*: min_char,             \
  int8_t*: min_int8_t,               \
  int8_t const*: min_int8_t,         \
  uint8_t*: min_int8_t,              \
  uint8_t const*: min_int8_t,        \
  int32_t*: min_int32_t,             \
  int32_t const*: min_int32_t,       \
  uint32_t*: min_int32_t,            \
  uint32_t const*: min_int32_t,      \
  int64_t*: min_int64_t,             \
  int64_t const*: min_int64_t,       \
  uint64_t*: min_int64_t,            \
  uint64_t const*: min_int64_t,      \
  float*: min_float,                 \
  float const*: min_float,           \
  double*: min_double,               \
  double const*: min_double)(arr, size)
#define CapyMinDec(type) type min_ ## type(type const* const arr, size_t size);
CapyMinDec(char)
CapyMinDec(int8_t)
CapyMinDec(uint8_t)
CapyMinDec(int16_t)
CapyMinDec(uint16_t)
CapyMinDec(int32_t)
CapyMinDec(uint32_t)
CapyMinDec(int64_t)
CapyMinDec(uint64_t)
CapyMinDec(float)
CapyMinDec(double)

// Get the maximum value in a array of base type values
// Input:
//    arr: the array
//   size: the size of the array
// Output:
//   Return the maximum value
#define max(arr, size) _Generic(arr, \
  char*: max_char,                   \
  char const*: max_char,             \
  int8_t*: max_int8_t,               \
  int8_t const*: max_int8_t,         \
  uint8_t*: max_int8_t,              \
  uint8_t const*: max_int8_t,        \
  int32_t*: max_int32_t,             \
  int32_t const*: max_int32_t,       \
  uint32_t*: max_int32_t,            \
  uint32_t const*: max_int32_t,      \
  int64_t*: max_int64_t,             \
  int64_t const*: max_int64_t,       \
  uint64_t*: max_int64_t,            \
  uint64_t const*: max_int64_t,      \
  float*: max_float,                 \
  float const*: max_float,           \
  double*: max_double,               \
  double const*: max_double)(arr, size)
#define CapyMaxDec(type) type max_ ## type(type const* const arr, size_t size);
CapyMaxDec(char)
CapyMaxDec(int8_t)
CapyMaxDec(uint8_t)
CapyMaxDec(int16_t)
CapyMaxDec(uint16_t)
CapyMaxDec(int32_t)
CapyMaxDec(uint32_t)
CapyMaxDec(int64_t)
CapyMaxDec(uint64_t)
CapyMaxDec(float)
CapyMaxDec(double)

// Get the index of the minimum value in a array of base type values
// Input:
//    arr: the array
//   size: the size of the array
// Output:
//   Return the minimum value
#define iMin(arr, size) _Generic(arr, \
  char*: iMin_char,                   \
  char const*: iMin_char,             \
  int8_t*: iMin_int8_t,               \
  int8_t const*: iMin_int8_t,         \
  uint8_t*: iMin_int8_t,              \
  uint8_t const*: iMin_int8_t,        \
  int32_t*: iMin_int32_t,             \
  int32_t const*: iMin_int32_t,       \
  uint32_t*: iMin_int32_t,            \
  uint32_t const*: iMin_int32_t,      \
  int64_t*: iMin_int64_t,             \
  int64_t const*: iMin_int64_t,       \
  uint64_t*: iMin_int64_t,            \
  uint64_t const*: iMin_int64_t,      \
  float*: iMin_float,                 \
  float const*: iMin_float,           \
  double*: iMin_double,               \
  double const*: iMin_double)(arr, size)
#define CapyIMinDec(type) \
  size_t iMin_ ## type(type const* const arr, size_t size);
CapyIMinDec(char)
CapyIMinDec(int8_t)
CapyIMinDec(uint8_t)
CapyIMinDec(int16_t)
CapyIMinDec(uint16_t)
CapyIMinDec(int32_t)
CapyIMinDec(uint32_t)
CapyIMinDec(int64_t)
CapyIMinDec(uint64_t)
CapyIMinDec(float)
CapyIMinDec(double)

// Get the index of the maximum value in a array of base type values
// Input:
//    arr: the array
//   size: the size of the array
// Output:
//   Return the maximum value
#define iMax(arr, size) _Generic(arr, \
  char*: iMax_char,                   \
  char const*: iMax_char,             \
  int8_t*: iMax_int8_t,               \
  int8_t const*: iMax_int8_t,         \
  uint8_t*: iMax_int8_t,              \
  uint8_t const*: iMax_int8_t,        \
  int32_t*: iMax_int32_t,             \
  int32_t const*: iMax_int32_t,       \
  uint32_t*: iMax_int32_t,            \
  uint32_t const*: iMax_int32_t,      \
  int64_t*: iMax_int64_t,             \
  int64_t const*: iMax_int64_t,       \
  uint64_t*: iMax_int64_t,            \
  uint64_t const*: iMax_int64_t,      \
  float*: iMax_float,                 \
  float const*: iMax_float,           \
  double*: iMax_double,               \
  double const*: iMax_double)(arr, size)
#define CapyiMaxDec(type) \
  size_t iMax_ ## type(type const* const arr, size_t size);
CapyiMaxDec(char)
CapyiMaxDec(int8_t)
CapyiMaxDec(uint8_t)
CapyiMaxDec(int16_t)
CapyiMaxDec(uint16_t)
CapyiMaxDec(int32_t)
CapyiMaxDec(uint32_t)
CapyiMaxDec(int64_t)
CapyiMaxDec(uint64_t)
CapyiMaxDec(float)
CapyiMaxDec(double)

// for loop using the 'iterator' to iterates on each 'elem' of the iterator's
// associated container. A copy of the value in the container can be
// accessed with 'elem', and the adress of the value with 'elem'Ptr.
#define forEach(elem, iterator)                                             \
  for(                                                                      \
    __typeof__((iterator).datatype) *(elem ## Ptr) =                        \
      $(&(iterator), reset)(), elem =                                       \
      (elem ## Ptr ? *elem ## Ptr : (&(iterator))->datatype);               \
    $(&(iterator), isActive)();                                             \
    elem ## Ptr = $(&(iterator), next)(),                                   \
    elem = (elem ## Ptr ? *elem ## Ptr : elem))

// Inheritance operator. Use it in the 'Create' function of the inheriting
// class to initialise the 'Parent' properties and methods in 'Instance'.
// 'Args' are arguments given to the parent 'Create' function (inside
// parenthesis)
#define CapyInherits(Instance, Parent, Args)          \
  Parent capyParent = Parent ## Create Args;          \
  memcpy(&(Instance), &capyParent, sizeof(Parent));   \
  (Instance).destruct ## Parent = (Instance).destruct

// Clone of asprintf
// Input:
//   fmt: format as in sprintf
//   ...: arguments as in sprintf
// Output:
//   Return a newly allocated string
// Exception:
//   May raise CapyExc_MallocFailed
char* strCreate(
   char const* fmt,
               ...);

// Sleep for a given amount of time in milliseconds
// Input:
//   delayMs: delay in milliseconds
// Output:
//  Return -1 if the sleep has been interrupted by an interruption, else 0
int CapySleepMs(int32_t const delayMs);

// Check if an address is inside the currently accessible address space
// Input:
//      ptr: the address to check
//   nbByte: the number of bytes checked from that adress, if equal to 0 uses
//           1 byte instead
// Output:
//   Return true if the 'nbByte' bytes from 'ptr' are in the accessible
//   adress space, else false
bool CapyIsAccessibleMem(
  void const* const ptr,
       size_t const nbByte);

// Get the quantity of memory currently used by the process calling this
// function.
// Output:
//   Return the quantity of memory in bytes. May return 0 if the quantity
//   of used memory couldn't be measured.
long CapyGetMemUsed(void);

// Avoid child process to become zombies and wait until their parent's wait()
// call. This applies to *all* child processes.
void CapyChildProcEndsWithoutWait(void);

// Typedef for struct sigaction to comply with CBo
typedef struct sigaction Sigaction;

// Get teh maximum possible value for an integer type
#define issigned(t) (((t)(-1)) < ((t) 0))
#define umaxof(t) (((0x1ULL << ((sizeof(t) * 8ULL) - 1ULL)) - 1ULL) | \
                    (0xFULL << ((sizeof(t) * 8ULL) - 4ULL)))
#define smaxof(t) (((0x1ULL << ((sizeof(t) * 8ULL) - 1ULL)) - 1ULL) | \
                    (0x7ULL << ((sizeof(t) * 8ULL) - 4ULL)))
#define maxof(t) ((unsigned long long) (issigned(t) ? smaxof(t) : umaxof(t)))

// Get a pointer to the instance of the structure containing 'ptr' given its
// a structure of type 'containerType' and 'ptr' points to the 'fieldName' field
// in that structure.
#define containerOf(ptr, containerType, fieldName) \
    (containerType*)((char*)(ptr) - offsetof(containerType, fieldName))

// Check if the architecture is big endian
// Output:
//   Return true if the architecture is big endian, else false. If using
//   gcc one can also use:
//   __BYTE_ORDER__ == __ORDER_BIG_ENDIAN__
bool CapyIsBigEndian(void);
#endif
