// ------------------------------ chrono.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "chrono.h"

// Labels for units
char const* capyChronoUnitLbl[capyChrono_nbUnit] = {"d", "h", "m", "s", "ms"};

// Get the current date and time in format yyyymmddhhiiss
// Input:
//   buffer: buffer updated with the result
// Output:
//   buffer is updated.
static void GetTimeStamp(char* const buffer) {
  time_t now;
  struct tm * timeinfo;
  time(&now);
  timeinfo = localtime(&now);
  sprintf(
    buffer, "%d%02d%02d%02d%02d%02d",
    timeinfo->tm_year + 1900, timeinfo->tm_mon + 1, timeinfo->tm_mday,
    timeinfo->tm_hour, timeinfo->tm_min, timeinfo->tm_sec);
}

// Start the timer
static void Start(void) {
  methodOf(CapyChrono);

  // Update the start time
  if(clock_gettime(CLOCK_MONOTONIC, &(that->startTime)) == -1) {
    raiseExc(CapyExc_NoTimeAvailable);
  }
}

// Stop the timer
static void Stop(void) {
  methodOf(CapyChrono);

  // Update the stop time
  if(clock_gettime(CLOCK_MONOTONIC, &(that->stopTime)) == -1) {
    raiseExc(CapyExc_NoTimeAvailable);
  }

  // Update the elapsed time
  CapyChronoTime_t timeMs =
    (that->stopTime.tv_sec * 1000 + that->stopTime.tv_nsec / 1000000) -
    (that->startTime.tv_sec * 1000 + that->startTime.tv_nsec / 1000000);
  CapyChronoTime_t coeff[capyChrono_nbUnit] =
    {86400000, 3600000, 60000, 1000, 1};
  loop(i, capyChrono_nbUnit) {
    that->elapsedTime[i] = timeMs / coeff[i];
    timeMs -= that->elapsedTime[i] * coeff[i];
  }
}

// Get the elapsed time in a given unit
// Input:
//   unit: the unit of the returned value
// Output:
//   Return the elapsed time between the last call to start() and the last
//   call to stop()
static double GetElapsedTime(CapyChronoUnit unit) {
  methodOf(CapyChrono);

  // Variable to memorise the result
  double time = 0.0;

  // Calculate the elapsed time in the requested unit
  size_t coeff[capyChrono_nbUnit] = {86400000, 3600000, 60000, 1000, 1};
  loop(i, capyChrono_nbUnit) {
    time +=
      (double)(that->elapsedTime[i]) * ((double)coeff[i] / (double)coeff[unit]);
  }

  // Return the result
  return time;
}

// Get the elapsed time in the most convenient unit
// Input:
//   unit: a pointer to the choosen unit
// Output:
//   Return the elapsed time between the last call to start() and the last
//   call to stop() into the largest unit such as the returned time is greater
//   than 1.0
static double GetElapsedTimeBestUnit(CapyChronoUnit* const unit) {
  methodOf(CapyChrono);

  // Variable to memorise the result
  double time = 0.0;

  // Calculate the elapsed time in the requested unit
  size_t coeff[capyChrono_nbUnit] = {86400000, 3600000, 60000, 1000, 1};
  *unit = capyChrono_day;
  bool flag = false;
  loop(i, capyChrono_nbUnit) {
    if(that->elapsedTime[i] > 0) {
      time +=
        (double)(that->elapsedTime[i]) *
        ((double)coeff[i] / (double)coeff[*unit]);
      flag = true;
    } else if(!flag && i < capyChrono_nbUnit - 1) ++(*unit);
  }

  // Return the result
  return time;
}

// Free the memory used by a CapyChrono
static void Destruct(void) {
  return;
}

// Create a CapyChrono
// Output:
//   Return a CapyChrono. It is automatically started upon creation.
CapyChrono CapyChronoCreate(void) {
  CapyChrono chrono = {
    .destruct = Destruct,
    .start = Start,
    .stop = Stop,
    .getElapsedTime = GetElapsedTime,
    .getElapsedTimeBestUnit = GetElapsedTimeBestUnit,
    .getTimeStamp = GetTimeStamp,
  };
  loop(i, capyChrono_nbUnit) chrono.elapsedTime[i] = 0;
  if(clock_gettime(CLOCK_MONOTONIC, &(chrono.startTime)) == -1) {
    raiseExc(CapyExc_NoTimeAvailable);
  }
  chrono.stopTime = chrono.startTime;
  return chrono;
}

// Allocate memory for a new CapyChrono and create it
// Output:
//   Return a CapyChrono. It is automatically started upon creation.
// Exception:
//   May raise CapyExc_MallocFailed.
CapyChrono* CapyChronoAlloc(void) {
  CapyChrono* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyChronoCreate();
  return that;
}

// Free the memory used by a CapyChrono* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyChrono to free
void CapyChronoFree(CapyChrono** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
