// ---------------------------------- chrono.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_CHRONO_H
#define CAPY_CHRONO_H
#include "externalHeaders.h"
#include "cext.h"

// Description:
// Class to make time measurement.

// Enumeration for the units of the elapsed time
typedef enum CapyChronoUnit {
  capyChrono_day,
  capyChrono_hour,
  capyChrono_minute,
  capyChrono_second,
  capyChrono_millisecond,
  capyChrono_nbUnit,
} CapyChronoUnit;

// Labels for units
extern char const* capyChronoUnitLbl[capyChrono_nbUnit];

// Type for the time values of a CapyChrono
typedef int64_t CapyChronoTime_t;

// Chrono object
typedef struct CapyChrono {

  // Elapsed time divided into each unit component.
  // For example 1.2s is [0,0,0,1,200]
  CapyChronoTime_t elapsedTime[capyChrono_nbUnit];
  CapyPad(CapyChronoTime_t, elapsedTime);

  // Start and stop times
  struct timespec startTime;
  struct timespec stopTime;

  // Destructor
  void (*destruct)(void);

  // Start the timer
  void (*start)(void);

  // Stop the timer
  void (*stop)(void);

  // Get the elapsed time in a given unit between
  // Input:
  //   unit: the unit of the returned value
  // Output:
  //   Return the elapsed time between the last call to start() and the last
  //   call to stop()
  double (*getElapsedTime)(CapyChronoUnit unit);

  // Get the elapsed time in the most convenient unit
  // Input:
  //   unit: a pointer to the choosen unit
  // Output:
  //   Return the elapsed time between the last call to start() and the last
  //   call to stop() into the largest unit such as the returned time is greater
  //   than 1.0
  double (*getElapsedTimeBestUnit)(CapyChronoUnit* const unit);

  // Get the current date and time in format yyyymmddhhiiss
  // Input:
  //   buffer: buffer updated with the result
  // Output:
  //   buffer is updated.
  void (*getTimeStamp)(char* const buffer);
} CapyChrono;

// Create a CapyChrono
// Output:
//   Return a CapyChrono
CapyChrono CapyChronoCreate(void);

// Allocate memory for a new CapyChrono and create it
// Output:
//   Return a CapyChrono
// Exception:
//   May raise CapyExc_MallocFailed.
CapyChrono* CapyChronoAlloc(void);

// Free the memory used by a CapyChrono* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyChrono to free
void CapyChronoFree(CapyChrono** const that);
#endif
