// --------------------------- collisionDetection.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "collisionDetection.h"
#include "capymath.h"

// Check if a point collides with a circle.
// Input:
//   point: the point
//   circle: the circle
// Output:
//   Return true if there is collision, else false.
static bool IsPointCollidingCircle(
  CapyPoint2D const* const point,
   CapyCircle const* const circle) {
  CapyVec u = {
    .dim=2,
    .vals=(double[2]){point->x - circle->center.x, point->y - circle->center.y},
  };
  double l = CapyVecGetNorm(&u);
  return (l <= circle->radius);
}

// Check if a circle collides with a circle.
// Input:
//   circleA: the first circle
//   circleB: the second circle
// Output:
//   Return true if there is collision, else false.
static bool IsCircleCollidingCircle(
  CapyCircle const* const circleA,
  CapyCircle const* const circleB) {
  CapyVec u = {
    .dim=2,
    .vals=(double[2]){
      circleA->center.x - circleB->center.x,
      circleA->center.y - circleB->center.y
    },
  };
  double l = CapyVecGetNorm(&u);
  return (l <= circleA->radius + circleB->radius);
}

// Check if a point collides with a rectangle.
// Input:
//   point: the point
//   rect: the rectangle
// Output:
//   Return true if there is collision, else false.
static bool IsPointCollidingRectangle(
    CapyPoint2D const* const point,
  CapyRectangle const* const rect) {
  return (
    point->x >= rect->corners[0].x &&
    point->x <= rect->corners[1].x &&
    point->y >= rect->corners[0].y &&
    point->y <= rect->corners[1].y);
}

// Check if a rectangle collides with a rectangle.
// Input:
//   rectA: the first rectangle
//   rectB: the second rectangle
// Output:
//   Return true if there is collision, else false.
static bool IsRectangleCollidingRectangle(
  CapyRectangle const* const rectA,
  CapyRectangle const* const rectB) {
  return !(
    rectA->corners[1].x < rectB->corners[0].x ||
    rectA->corners[0].x > rectB->corners[1].x ||
    rectA->corners[1].y < rectB->corners[0].y ||
    rectA->corners[0].y > rectB->corners[1].y
  );
}

// Free the memory used by a CapyCollisionDetection
static void Destruct(void) {
  return;
}

// Create a CapyCollisionDetection
// Output:
//   Return a CapyCollisionDetection
CapyCollisionDetection CapyCollisionDetectionCreate(void) {
  CapyCollisionDetection that = {
    .destruct = Destruct,
    .isPointCollidingCircle = IsPointCollidingCircle,
    .isCircleCollidingCircle = IsCircleCollidingCircle,
    .isPointCollidingRectangle = IsPointCollidingRectangle,
    .isRectangleCollidingRectangle = IsRectangleCollidingRectangle,
  };
  return that;
}

// Allocate memory for a new CapyCollisionDetection and create it
// Output:
//   Return a CapyCollisionDetection
// Exception:
//   May raise CapyExc_MallocFailed.
CapyCollisionDetection* CapyCollisionDetectionAlloc(void) {
  CapyCollisionDetection* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyCollisionDetectionCreate();
  return that;
}

// Free the memory used by a CapyCollisionDetection* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyCollisionDetection to free
void CapyCollisionDetectionFree(CapyCollisionDetection** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
