// --------------------------- collisionDetection.h ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_COLLISIONDETECTION_H
#define CAPY_COLLISIONDETECTION_H
#include "externalHeaders.h"
#include "cext.h"
#include "geometricShape.h"

// Description:
// Collision detection between geometries.

// CollisionDetection object
typedef struct CapyCollisionDetection {

  // Destructor
  void (*destruct)(void);

  // Check if a point collides with a circle.
  // Input:
  //   point: the point
  //   circle: the circle
  // Output:
  //   Return true if there is collision, else false.
  bool (*isPointCollidingCircle)(
    CapyPoint2D const* const point,
     CapyCircle const* const circle);

  // Check if a circle collides with a circle.
  // Input:
  //   circleA: the first circle
  //   circleB: the second circle
  // Output:
  //   Return true if there is collision, else false.
  bool (*isCircleCollidingCircle)(
    CapyCircle const* const circleA,
    CapyCircle const* const circleB);

  // Check if a point collides with a rectangle.
  // Input:
  //   point: the point
  //   rect: the rectangle
  // Output:
  //   Return true if there is collision, else false.
  bool (*isPointCollidingRectangle)(
      CapyPoint2D const* const point,
    CapyRectangle const* const rect);

  // Check if a rectangle collides with a rectangle.
  // Input:
  //   rectA: the first rectangle
  //   rectB: the second rectangle
  // Output:
  //   Return true if there is collision, else false.
  bool (*isRectangleCollidingRectangle)(
    CapyRectangle const* const rectA,
    CapyRectangle const* const rectB);
} CapyCollisionDetection;

// Create a CapyCollisionDetection
// Output:
//   Return a CapyCollisionDetection
CapyCollisionDetection CapyCollisionDetectionCreate(void);

// Allocate memory for a new CapyCollisionDetection and create it
// Output:
//   Return a CapyCollisionDetection
// Exception:
//   May raise CapyExc_MallocFailed.
CapyCollisionDetection* CapyCollisionDetectionAlloc(void);

// Free the memory used by a CapyCollisionDetection* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyCollisionDetection to free
void CapyCollisionDetectionFree(CapyCollisionDetection** const that);
#endif
