// ------------------------------ strDecorator.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "color.h"

// Constants to speed up HSV2RGB
static CapyColorValue_t const hsv_bound_a = 60.0 / 360.0;
static CapyColorValue_t const hsv_bound_b = 120.0 / 360.0;
static CapyColorValue_t const hsv_bound_c = 180.0 / 360.0;
static CapyColorValue_t const hsv_bound_d = 240.0 / 360.0;
static CapyColorValue_t const hsv_bound_e = 300.0 / 360.0;

// Predefined colors
CapyColorData const capyColorRGBAWhite = {.RGBA = {1.0, 1.0, 1.0, 1.0}};
CapyColorData const capyColorRGBABlack = {.RGBA = {0.0, 0.0, 0.0, 1.0}};
CapyColorData const capyColorRGBARed = {.RGBA = {1.0, 0.0, 0.0, 1.0}};
CapyColorData const capyColorRGBAGreen = {.RGBA = {0.0, 1.0, 0.0, 1.0}};
CapyColorData const capyColorRGBABlue = {.RGBA = {0.0, 0.0, 1.0, 1.0}};
CapyColorData const capyColorRGBAYellow = {.RGBA = {1.0, 1.0, 0.0, 1.0}};
CapyColorData const capyColorRGBWhite = {.RGB = {1.0, 1.0, 1.0}};
CapyColorData const capyColorRGBBlack = {.RGB = {0.0, 0.0, 0.0}};
CapyColorData const capyColorRGBRed = {.RGB = {1.0, 0.0, 0.0}};
CapyColorData const capyColorRGBGreen = {.RGB = {0.0, 1.0, 0.0}};
CapyColorData const capyColorRGBBlue = {.RGB = {0.0, 0.0, 1.0}};
CapyColorData const capyColorRGBLightBlue = {.RGB = {0.5, 0.5, 1.0}};
CapyColorData const capyColorRGBYellow = {.RGB = {1.0, 1.0, 0.0}};
CapyColorData const capyColorHSVWhite = {.HSV = {0.0, 0.0, 1.0}};
CapyColorData const capyColorHSVBlack = {.HSV = {0.0, 0.0, 0.0}};
CapyColorData const capyColorHSVRed = {.HSV = {0.0, 1.0, 1.0}};
CapyColorData const capyColorHSVGreen = {.HSV = {hsv_bound_b, 1.0, 1.0}};
CapyColorData const capyColorHSVBlue = {.HSV = {hsv_bound_d, 1.0, 1.0}};
CapyColorData const capyColorHSVYellow = {.HSV = {hsv_bound_a, 1.0, 1.0}};

// Color spaces name
char const* capyColorSpaceLbl[capyColorSpace_nb] = {
  "sRGB",
  "HSV",
  "XYZ",
  "LAB",
  "rgb",
  "l1l2l3",
  "c1c2c3",
  "Oklab"
};

// Equality operator for CapyColorValue_t
#define equalColorValue(a, b) _Generic(a, \
  float: equalf,                          \
  double: equald)(a, b)

// Module operator for CapyColorValue_t
#define moduloColorValue(a, b) _Generic(a, \
  float: fmodf,                            \
  double: fmod)(a, b)

// Absolute operator for CapyColorValue_t
#define absColorValue(a) _Generic(a, \
  float: fabsf,                      \
  double: fabs)(a)

// Convert the color from RGB to HSV
// Output:
//   Return the converted color, values in [0,1]
// Can be used with $$(..., CapyColorData*, ...)
static CapyColorData RGB2HSV(void) {
  methodOf(CapyColor);

  // Declare the result variable
  CapyColorData color = capyColorRGBABlack;

  // Calculate HSV
  CapyColorValue_t* RGB = that->vals.RGB;
  size_t iMax = iMax(RGB, 3);
  size_t iMin = iMin(RGB, 3);
  CapyColorValue_t range = RGB[iMax] - RGB[iMin];
  if(equalColorValue(range, 0.0)) color.HSV[0] = 0.0;
  else if(iMax == 0) {
    color.HSV[0] = moduloColorValue(((RGB[1] - RGB[2]) / range), 6.0);
  } else if(iMax == 1) color.HSV[0] = (RGB[2] - RGB[0]) / range + 2.0;
  else if(iMax == 2) color.HSV[0] = (RGB[0] - RGB[1]) / range + 4.0;
  color.HSV[0] /= 6.0;
  if(color.HSV[0] < 0.0) color.HSV[0] = 1.0 + color.HSV[0];
  color.HSV[2] = RGB[iMax];
  if(equalColorValue(color.HSV[2], 0.0)) color.HSV[1] = 0.0;
  else color.HSV[1] = range / color.HSV[2];

  // Return the result
  return color;
}

// Convert the color from HSV to RGB
// Output:
//   Return the converted color, values in [0,1]
// Can be used with $$(..., CapyColorData*, ...)
static CapyColorData HSV2RGB(void) {
  methodOf(CapyColor);

  // Declare the result variable
  CapyColorData color = capyColorRGBAWhite;
  CapyColorValue_t* RGB = color.RGB;

  // Calculate RGB
  CapyColorValue_t* HSV = that->vals.HSV;
  CapyColorValue_t c = HSV[1] * HSV[2];
  CapyColorValue_t x =
    c * (1.0 - absColorValue(moduloColorValue(HSV[0] * 6.0, 2.0) - 1.0));
  CapyColorValue_t m = HSV[2] - c;
  if(HSV[0] < hsv_bound_a) {
    RGB[0] = c; RGB[1] = x; RGB[2] = 0;
  } else if(hsv_bound_a <= HSV[0] && HSV[0] < hsv_bound_b) {
    RGB[0] = x; RGB[1] = c; RGB[2] = 0;
  } else if(hsv_bound_b <= HSV[0] && HSV[0] < hsv_bound_c) {
    RGB[0] = 0; RGB[1] = c; RGB[2] = x;
  } else if(hsv_bound_c <= HSV[0] && HSV[0] < hsv_bound_d) {
    RGB[0] = 0; RGB[1] = x; RGB[2] = c;
  } else if(hsv_bound_d <= HSV[0] && HSV[0] < hsv_bound_e) {
    RGB[0] = x; RGB[1] = 0; RGB[2] = c;
  } else if(hsv_bound_e <= HSV[0]) {
    RGB[0] = c; RGB[1] = 0; RGB[2] = x;
  }
  loop(i, 3) RGB[i] += m;

  // Return the result
  return color;
}

// Convert the color from RGB (in [0,1]) to XYZ (in [0,1])
// Output:
//   Return the converted color
// Can be used with $$(..., CapyColorData*, ...)
static CapyColorData RGB2XYZ(void) {
  methodOf(CapyColor);

  // Declare the result variable
  CapyColorData color = capyColorRGBAWhite;

  // Set variables according to illuminant
  double m[9];
  switch(that->illuminant) {
    case capyColorIlluminant_D50:
      m[0] = 0.436027535573195;
      m[1] = 0.385097932872408;
      m[2] = 0.143074531554397;
      m[3] = 0.222478677613186;
      m[4] = 0.716902127457834;
      m[5] = 0.0606191949289806;
      m[6] = 0.0139242392790820;
      m[7] = 0.0970836931437703;
      m[8] = 0.714092067577148;
      break;
    case capyColorIlluminant_D65:
      m[0] = 0.4124108464885388;
      m[1] = 0.3575845678529519;
      m[2] = 0.18045380393360833;
      m[3] = 0.21264934272065283;
      m[4] = 0.7151691357059038;
      m[5] = 0.07218152157344333;
      m[6] = 0.019331758429150258;
      m[7] = 0.11919485595098397;
      m[8] = 0.9503900340503373;
      break;
    default:
      raiseExc(CapyExc_UndefinedExecution);
      m[0] = 1.0; m[1] = 0.0; m[2] = 0.0;
      m[3] = 0.0; m[4] = 1.0; m[5] = 0.0;
      m[6] = 0.0; m[7] = 0.0; m[8] = 1.0;
  }

  // Linearisation
  CapyColorValue_t RGB[3] = {0., 0., 0.};
  loop(i, 3) {
    RGB[i] = (
      that->vals.RGB[i] <= 0.0392156862745 ?
      that->vals.RGB[i] / 12.92 :
      pow((that->vals.RGB[i] + 0.055) / 1.055, 2.4));
  }

  // Calculate XYZ
  color.XYZ[0] = m[0] * RGB[0] + m[1] * RGB[1] + m[2] * RGB[2];
  color.XYZ[1] = m[3] * RGB[0] + m[4] * RGB[1] + m[5] * RGB[2];
  color.XYZ[2] = m[6] * RGB[0] + m[7] * RGB[1] + m[8] * RGB[2];

  // Return the result
  return color;
}

// Convert the color from XYZ (in [0,1]) to RGB (in [0,1])
// Output:
//   Return the converted color
// Can be used with $$(..., CapyColorData*, ...)
static CapyColorData XYZ2RGB(void) {
  methodOf(CapyColor);

  // Declare the result variable
  CapyColorData color = capyColorRGBAWhite;

  // Set variables according to illuminant
  double m[9];
  switch(that->illuminant) {
    case capyColorIlluminant_D50:
      m[0] = 3.13424933163426;
      m[1] = -1.61717292521282;
      m[2] = -0.490692377104512;
      m[3] = -0.978746070339639;
      m[4] = 1.91611436125945;
      m[5] = 0.0334415219513205;
      m[6] = 0.0719490494816283;
      m[7] = -0.228969853236611;
      m[8] = 1.40540126012171;
      break;
    case capyColorIlluminant_D65:
      m[0] = 3.240812398895283;
      m[1] = -1.5373084456298136;
      m[2] = -0.4985865229069666;
      m[3] = -0.9692430170086407;
      m[4] = 1.8759663029085742;
      m[5] = 0.04155503085668564;
      m[6] = 0.055638398436112804;
      m[7] = -0.20400746093241362;
      m[8] = 1.0571295702861434;
      break;
    default:
      raiseExc(CapyExc_UndefinedExecution);
      m[0] = 1.0; m[1] = 1.0; m[2] = 1.0;
      m[3] = 0.0; m[4] = 1.0; m[5] = 1.0;
      m[6] = 0.0; m[7] = 1.0; m[8] = 1.0;
  }

  // Calculate sRGB
  CapyColorValue_t RGB[3];
  RGB[0] =
    m[0] * that->vals.XYZ[0] +
    m[1] * that->vals.XYZ[1] +
    m[2] * that->vals.XYZ[2];
  RGB[1] =
    m[3] * that->vals.XYZ[0] +
    m[4] * that->vals.XYZ[1] +
    m[5] * that->vals.XYZ[2];
  RGB[2] =
    m[6] * that->vals.XYZ[0] +
    m[7] * that->vals.XYZ[1] +
    m[8] * that->vals.XYZ[2];

  // Delinearisation
  loop(i, 3) {
    color.RGB[i] =
      RGB[i] <= 0.00313066844250060782371 ?
      RGB[i] * 12.92 : pow(RGB[i], 1.0 / 2.4) * 1.055 - 0.055;
  }

  // Return the result
  return color;
}

// Functions used to convert from/to LAB
static double ConversionLABf(double t) {
  if(t > pow(6.0 / 29.0, 3.0)) return pow(t, 1.0 / 3.0);
  else return t / (3.0 * pow(6.0 / 29.0, 2.0)) + 4.0 / 29.0;
}

static double ConversionLABInvf(double t) {
  if(t > (6.0 / 29.0)) return pow(t, 3.0);
  else return 3.0 * pow(6.0 / 29.0, 2.0) * (t - 4.0 / 29.0);
}

// Convert the color from LAB (in [0,100],[-128,128],[-128,128]) to XYZ
// (in [0,1])
// Output:
//   Return the converted color
// Can be used with $$(..., CapyColorData*, ...)
static CapyColorData LAB2XYZ(void) {
  methodOf(CapyColor);

  // Declare the result variable
  CapyColorData color = capyColorRGBAWhite;

  // Set variables according to illuminant
  double Xn, Yn, Zn;
  switch(that->illuminant) {
    case capyColorIlluminant_D50:
      Xn = 0.964212;
      Yn = 1.0;
      Zn = 0.825188;
      break;
    case capyColorIlluminant_D65:
      Xn = 0.950489;
      Yn = 1.0;
      Zn = 1.08884;
      break;
    default:
      raiseExc(CapyExc_UndefinedExecution);
      Xn = 1.0;
      Yn = 1.0;
      Zn = 1.0;
  }

  // Calculate XYZ
  color.XYZ[0] =
    Xn * ConversionLABInvf(
      (that->vals.LAB[0] + 16.0) / 116.0 + that->vals.LAB[1] / 500.0);
  color.XYZ[1] =
    Yn * ConversionLABInvf((that->vals.LAB[0] + 16.0) / 116.0);
  color.XYZ[2] =
    Zn * ConversionLABInvf(
      (that->vals.LAB[0] + 16.0) / 116.0 - that->vals.LAB[2] / 200.0);

  // Return the result
  return color;
}

// Convert the color from Oklab to XYZ (D65 white point)
// (in [0,1])
// Output:
//   Return the converted color
// Can be used with $$(..., CapyColorData*, ...)
static CapyColorData Oklab2XYZ(void) {
  methodOf(CapyColor);

  // Declare the result variable
  CapyColorData color = capyColorRGBAWhite;

  // Calculate Oklab
  CapyColorValue_t lms[3];
  lms[0] =
    +1.00000000 * that->vals.Oklab[0] +
    +0.39633779 * that->vals.Oklab[1] +
    +0.21580376 * that->vals.Oklab[2];
  lms[1] =
    +1.00000001 * that->vals.Oklab[0] +
    -0.10556134 * that->vals.Oklab[1] +
    -0.06385417 * that->vals.Oklab[2];
  lms[2] =
    1.00000005 * that->vals.Oklab[0] +
    -0.08948418 * that->vals.Oklab[1] +
    -1.29148554 * that->vals.Oklab[2];
  lms[0] = lms[0] * lms[0] * lms[0];
  lms[1] = lms[1] * lms[1] * lms[1];
  lms[2] = lms[2] * lms[2] * lms[2];
  color.XYZ[0] =
    +1.22701385 * lms[0] +
    -0.55779998 * lms[1] +
    +0.28125615 * lms[2];
  color.XYZ[1] =
    -0.04058018 * lms[0] +
    +1.11225687 * lms[1] +
    -0.07167668 * lms[2];
  color.XYZ[2] =
    -0.07638128 * lms[0] +
    -0.42148198 * lms[1] +
    +1.58616322 * lms[2];

  // Return the result
  return color;
}

// Convert the color from XYZ (in [0,1]) to LAB (in
// [0,100],[-128,128],[-128,128])
// Output:
//   Return the converted color
// Can be used with $$(..., CapyColorData*, ...)
static CapyColorData XYZ2LAB(void) {
  methodOf(CapyColor);

  // Declare the result variable
  CapyColorData color = capyColorRGBAWhite;

  // Set variables according to illuminant
  double Xn, Yn, Zn;
  switch(that->illuminant) {
    case capyColorIlluminant_D50:
      Xn = 0.964212;
      Yn = 1.0;
      Zn = 0.825188;
      break;
    case capyColorIlluminant_D65:
      Xn = 0.950489;
      Yn = 1.0;
      Zn = 1.08884;
      break;
    default:
      raiseExc(CapyExc_UndefinedExecution);
      Xn = 1.0;
      Yn = 1.0;
      Zn = 1.0;
  }

  // Calculate LAB
  double yyn = ConversionLABf(that->vals.XYZ[1] / Yn);
  color.LAB[0] = 116.0 * yyn - 16.0;
  color.LAB[1] =
    500.0 * (ConversionLABf(that->vals.XYZ[0] / Xn) - yyn);
  color.LAB[2] =
    200.0 * (yyn - ConversionLABf(that->vals.XYZ[2] / Zn));

  // Return the result
  return color;
}

// Convert the color from XYZ to Oklab
// Output:
//   Return the converted color
// Can be used with $$(..., CapyColorData*, ...)
static CapyColorData XYZ2Oklab(void) {
  methodOf(CapyColor);

  // Declare the result variable
  CapyColorData color = capyColorRGBAWhite;

  // Calculate Oklab
  CapyColorValue_t lms[3];
  lms[0] =
    +0.8189330101 * that->vals.XYZ[0] +
    +0.3618667424 * that->vals.XYZ[1] +
    -0.1288597137 * that->vals.XYZ[2];
  lms[1] =
    +0.0329845436 * that->vals.XYZ[0] +
    +0.9293118715 * that->vals.XYZ[1] +
    +0.0361456387 * that->vals.XYZ[2];
  lms[2] =
    +0.0482003018 * that->vals.XYZ[0] +
    +0.2643662691 * that->vals.XYZ[1] +
    +0.6338517070 * that->vals.XYZ[2];
  lms[0] = cbrt(lms[0]);
  lms[1] = cbrt(lms[1]);
  lms[2] = cbrt(lms[2]);
  color.Oklab[0] =
    +0.2104542553 * lms[0] +
    +0.7936177850 * lms[1] +
    -0.0040720468 * lms[2];
  color.Oklab[1] =
    +1.9779984951 * lms[0] +
    -2.4285922050 * lms[1] +
    +0.4505937099 * lms[2];
  color.Oklab[2] =
    +0.0259040371 * lms[0] +
    +0.7827717662 * lms[1] +
    -0.8086757660 * lms[2];

  // Return the result
  return color;
}

// Convert the color from RGB to rgb
// Output:
//   Return the converted color
// Can be used with $$(..., CapyColorData*, ...)
static CapyColorData RGB2rgb(void) {
  methodOf(CapyColor);

  // Declare the result variable
  CapyColorData color = capyColorRGBAWhite;

  // Calculate rgb
  double sum = that->vals.RGB[0] + that->vals.RGB[1] + that->vals.RGB[2];
  loop(i, 3) color.rgb[i] = that->vals.RGB[i] / sum;

  // Return the result
  return color;
}

// Convert the color from RGB to l1l2l3
// Output:
//   Return the converted color
// Can be used with $$(..., CapyColorData*, ...)
static CapyColorData RGB2l1l2l3(void) {
  methodOf(CapyColor);

  // Declare the result variable
  CapyColorData color = capyColorRGBABlack;

  // Calculate l1l2l3
  CapyColorValue_t* RGB = that->vals.RGB;
  double a = (RGB[0] - RGB[1]) * (RGB[0] - RGB[1]);
  double b = (RGB[0] - RGB[2]) * (RGB[0] - RGB[2]);
  double c = (RGB[1] - RGB[2]) * (RGB[1] - RGB[2]);
  double sum = a + b + c;
  if(sum > 0.0) {
    color.rgb[0] = a / sum;
    color.rgb[1] = b / sum;
    color.rgb[2] = c / sum;
  }

  // Return the result
  return color;
}

// Convert the color from RGB to c1c2c3
// Output:
//   Return the converted color
// Can be used with $$(..., CapyColorData*, ...)
static CapyColorData RGB2c1c2c3(void) {
  methodOf(CapyColor);

  // Declare the result variable
  CapyColorData color = capyColorRGBABlack;

  // Calculate c1c2c3
  CapyColorValue_t* RGB = that->vals.RGB;
  if(RGB[1] > 0.0 || RGB[2] > 0.0) {
    color.rgb[0] =
      atan(RGB[0] / (RGB[1] > RGB[2] ? RGB[1] : RGB[2])) / 1.57079632679;
  }
  if(RGB[0] > 0.0 || RGB[2] > 0.0) {
    color.rgb[1] =
      atan(RGB[1] / (RGB[0] > RGB[2] ? RGB[0] : RGB[2])) / 1.57079632679;
  }
  if(RGB[0] > 0.0 || RGB[1] > 0.0) {
    color.rgb[2] =
      atan(RGB[2] / (RGB[0] > RGB[1] ? RGB[0] : RGB[1])) / 1.57079632679;
  }

  // Return the result
  return color;
}

// Check if that is equal to 'color'
// Input:
//   color: the color to be compared with
// Output:
//   Return true if the colors are identical, else false
static bool equalsRGB(CapyColorData const* const color) {
  methodOf(CapyColor);
#if BUILD_MODE == 0
  assert(color != NULL);
#endif
  CapyColorValue_t* RGB = that->vals.RGB;
  return (
    equalColorValue(RGB[0], color->RGB[0]) &&
    equalColorValue(RGB[1], color->RGB[1]) &&
    equalColorValue(RGB[2], color->RGB[2]));
}

static bool equalsRGBA(CapyColorData const* const color) {
  methodOf(CapyColor);
#if BUILD_MODE == 0
  assert(color != NULL);
#endif
  CapyColorValue_t* RGBA = that->vals.RGBA;
  return (
    equalColorValue(RGBA[0], color->RGBA[0]) &&
    equalColorValue(RGBA[1], color->RGBA[1]) &&
    equalColorValue(RGBA[2], color->RGBA[2]) &&
    equalColorValue(RGBA[3], color->RGBA[3]));
}

// Free the memory used by a CapyColor
static void Destruct(void) {
  return;
}

// Create a CapyColor
// Input:
//   color: the color values
// Output:
//   Return a CapyColor
CapyColor CapyColorCreate(CapyColorData const color) {
  return (CapyColor){
    .vals = color,
    .illuminant = capyColorIlluminant_D65,
    .destruct = Destruct,
    .RGB2HSV = RGB2HSV,
    .HSV2RGB = HSV2RGB,
    .RGB2XYZ = RGB2XYZ,
    .XYZ2RGB = XYZ2RGB,
    .LAB2XYZ = LAB2XYZ,
    .Oklab2XYZ = Oklab2XYZ,
    .XYZ2LAB = XYZ2LAB,
    .XYZ2Oklab = XYZ2Oklab,
    .RGB2rgb = RGB2rgb,
    .RGB2l1l2l3 = RGB2l1l2l3,
    .RGB2c1c2c3 = RGB2c1c2c3,
    .equalsRGB = equalsRGB,
    .equalsRGBA = equalsRGBA,
    .equalsHSV = equalsRGB,
    .equalsXYZ = equalsRGB,
    .equalsLAB = equalsRGB,
    .equalsrgb = equalsRGB,
    .equalsl1l2l3 = equalsRGB,
    .equalsc1c2c3 = equalsRGB,
  };
}

// Allocate memory for a new CapyColor and create it
// Input:
//   color: the color values
// Output:
//   Return a CapyColor
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColor* CapyColorAlloc(CapyColorData const color) {
  CapyColor* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyColorCreate(color);
  return that;
}

// Free the memory used by a CapyColor* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyColor to free
void CapyColorFree(CapyColor** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Set a color in the palette
// Input:
//   iColor: the index of the color
//   color: the color to set
// Output:
//   The color is set.
static void SetColor(
         size_t const iColor,
  CapyColorData const color) {
  methodOf(CapyColorPalette);
  that->colors[iColor] = color;
}

// Get a color in the palette
// Input:
//   iColor: the index of the color
// Output:
//   Return the color.
static CapyColorData GetColor(size_t const iColor) {
  methodOf(CapyColorPalette);
  return that->colors[iColor];
}

// Get a color interpolated from the colors of the palette
// Input:
//   x: interpolation value in [0,that->size-1]
// Output:
//   Return the color.
static CapyColorData GetColorInterpolated(double const x) {
  methodOf(CapyColorPalette);
  CapyColorData color = that->colors[0];
  if(that->size < 2) return color;
  size_t idx = (size_t)floor(x);
  if(idx >= that->size - 1) idx = that->size - 2;
  double t = x - (double)idx;
  loop(i, 4) {
    color.vals[i] =
      (1.0 - t) * that->colors[idx].vals[i] + t * that->colors[idx + 1].vals[i];
  }
  return color;
}

// Set random colors in the palette
// Input:
//   rng: the RNG to be used
// Output:
//   The RGB colors are set.
static void SetRandomRGB(CapyRandom* const rng) {
  methodOf(CapyColorPalette);
  loop(iColor, that->size) {
    loop(i, 3) that->colors[iColor].RGB[i] = $(rng, getDouble)();
  }
}

// Set random pastel colors in the palette
// Input:
//   rng: the RNG to be used
// Output:
//   The RGB colors are set.
static void SetRandomPastelRGB(CapyRandom* const rng) {
  methodOf(CapyColorPalette);
  loop(iColor, that->size) {
    loop(i, 3) that->colors[iColor].RGB[i] = 0.5 + 0.5 * $(rng, getDouble)();
  }
}

// Free the memory used by a CapyColorPalette
static void DestructPalette(void) {
  methodOf(CapyColorPalette);
  free(that->colors);
}

// Create a CapyColorPalette
// Input:
//   size: the number of colors in the palette
// Output:
//   Return a CapyColorPalette, all colors initialised to capyColorRGBABlack
CapyColorPalette CapyColorPaletteCreate(size_t const size) {
  CapyColorPalette that = {
    .size = size,
    .colors = NULL,
    .destruct = DestructPalette,
    .set = SetColor,
    .get = GetColor,
    .getInterpolated = GetColorInterpolated,
    .setRandomRGB = SetRandomRGB,
    .setRandomPastelRGB = SetRandomPastelRGB,
  };
  safeMalloc(that.colors, that.size);
  loop(i, that.size) that.colors[i] = capyColorRGBABlack;
  return that;
}

// Allocate memory for a new CapyColorPalette and create it
// Input:
//   size: the number of colors in the palette
// Output:
//   Return a CapyColorPalette, all colors initialised to capyColorRGBABlack
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColorPalette* CapyColorPaletteAlloc(size_t const size) {
  CapyColorPalette* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyColorPaletteCreate(size);
  return that;
}

// Free the memory used by a CapyColorPalette* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyColorPalette to free
void CapyColorPaletteFree(CapyColorPalette** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
