// ---------------------------------- color.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_COLOR_H
#define CAPY_COLOR_H
#include "externalHeaders.h"
#include "cext.h"
#include "random.h"

// Description:
// Color class.

// Type for the color values
typedef double CapyColorValue_t;

// Color data
// Values in [0.0, 1.0] for RGB, RGBA, HSV
// Alpha channel: 0.0 -> fully transparent, 1.0 -> fully opaque
typedef union CapyColorData {
  CapyColorValue_t vals[4];
  CapyColorValue_t intensity;
  CapyColorValue_t RGB[3];
  CapyColorValue_t RGBA[4];
  CapyColorValue_t HSV[3];
  CapyColorValue_t XYZ[3];
  CapyColorValue_t LAB[3];
  CapyColorValue_t rgb[3];
  CapyColorValue_t l1l2l3[3];
  CapyColorValue_t c1c2c3[3];
  CapyColorValue_t Oklab[3];
} CapyColorData;

// Predefined colors
extern CapyColorData const capyColorRGBAWhite;
extern CapyColorData const capyColorRGBABlack;
extern CapyColorData const capyColorRGBARed;
extern CapyColorData const capyColorRGBAGreen;
extern CapyColorData const capyColorRGBABlue;
extern CapyColorData const capyColorRGBAYellow;
extern CapyColorData const capyColorRGBWhite;
extern CapyColorData const capyColorRGBBlack;
extern CapyColorData const capyColorRGBRed;
extern CapyColorData const capyColorRGBGreen;
extern CapyColorData const capyColorRGBBlue;
extern CapyColorData const capyColorRGBLightBlue;
extern CapyColorData const capyColorRGBYellow;
extern CapyColorData const capyColorHSVWhite;
extern CapyColorData const capyColorHSVBlack;
extern CapyColorData const capyColorHSVRed;
extern CapyColorData const capyColorHSVGreen;
extern CapyColorData const capyColorHSVBlue;
extern CapyColorData const capyColorHSVYellow;

// Color spaces
typedef enum {
  capyColorSpace_sRGB,
  capyColorSpace_HSV,
  capyColorSpace_XYZ,
  capyColorSpace_LAB,
  capyColorSpace_rgb,
  capyColorSpace_l1l2l3,
  capyColorSpace_c1c2c3,
  capyColorSpace_Oklab,
  capyColorSpace_nb,
} CapyColorSpace;

// Color spaces name
extern char const* capyColorSpaceLbl[capyColorSpace_nb];

// Color illuminant
typedef enum {
  capyColorIlluminant_D50, // 2 degrees
  capyColorIlluminant_D65, // 2 degrees
} CapyColorIlluminant;

// Color object
typedef struct CapyColor {

  // Color values
  CapyColorData vals;

  // Color illuminant for conversion (by default D65)
  CapyColorIlluminant illuminant;
  CapyPad(CapyColorIlluminant, 0);

  // Destructor
  void (*destruct)(void);

  // Convert the color from RGB to HSV
  // Output:
  //   Return the converted color, values in [0,1]
  // Can be used with $$(..., CapyColorData*, ...)
  CapyColorData (*RGB2HSV)(void);

  // Convert the color from HSV to RGB
  // Output:
  //   Return the converted color, values in [0,1]
  // Can be used with $$(..., CapyColorData*, ...)
  CapyColorData (*HSV2RGB)(void);

  // Convert the color from RGB (in [0,1]) to XYZ (in [0,1])
  // Output:
  //   Return the converted color
  // Can be used with $$(..., CapyColorData*, ...)
  CapyColorData (*RGB2XYZ)(void);

  // Convert the color from XYZ (in [0,1]) to RGB (in [0,1])
  // Output:
  //   Return the converted color, values in [0,1]
  // Can be used with $$(..., CapyColorData*, ...)
  CapyColorData (*XYZ2RGB)(void);

  // Convert the color from LAB to XYZ
  // Output:
  //   Return the converted color
  // Can be used with $$(..., CapyColorData*, ...)
  CapyColorData (*LAB2XYZ)(void);

  // Convert the color from Oklab to XYZ (D65 white point)
  // Output:
  //   Return the converted color
  // Can be used with $$(..., CapyColorData*, ...)
  CapyColorData (*Oklab2XYZ)(void);

  // Convert the color from XYZ to LAB
  // Output:
  //   Return the converted color, values in [0,1]
  // Can be used with $$(..., CapyColorData*, ...)
  CapyColorData (*XYZ2LAB)(void);

  // Convert the color from XYZ (D65 white point) to Oklab
  // Output:
  //   Return the converted color
  // Can be used with $$(..., CapyColorData*, ...)
  CapyColorData (*XYZ2Oklab)(void);

  // Convert the color from RGB to rgb
  // Output:
  //   Return the converted color
  // Can be used with $$(..., CapyColorData*, ...)
  CapyColorData (*RGB2rgb)(void);

  // Convert the color from RGB to l1l2l3
  // Output:
  //   Return the converted color
  // Can be used with $$(..., CapyColorData*, ...)
  CapyColorData (*RGB2l1l2l3)(void);

  // Convert the color from RGB to c1c2c3
  // Output:
  //   Return the converted color
  // Can be used with $$(..., CapyColorData*, ...)
  CapyColorData (*RGB2c1c2c3)(void);

  // Check if that is equal to 'color'
  // Input:
  //   color: the color to be compared with
  // Output:
  //   Return true if the colors are identical, else false
  // Can be used with $$(..., CapyColorData*, ...)
  bool (*equalsRGB)(CapyColorData const* const color);
  bool (*equalsRGBA)(CapyColorData const* const color);
  bool (*equalsHSV)(CapyColorData const* const color);
  bool (*equalsXYZ)(CapyColorData const* const color);
  bool (*equalsLAB)(CapyColorData const* const color);
  bool (*equalsrgb)(CapyColorData const* const color);
  bool (*equalsl1l2l3)(CapyColorData const* const color);
  bool (*equalsc1c2c3)(CapyColorData const* const color);
} CapyColor;

// Create a CapyColor
// Input:
//   color: the color values
// Output:
//   Return a CapyColor
CapyColor CapyColorCreate(CapyColorData const color);

// Allocate memory for a new CapyColor and create it
// Input:
//   color: the color values
// Output:
//   Return a CapyColor
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColor* CapyColorAlloc(CapyColorData const color);

// Free the memory used by a CapyColor* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyColor to free
void CapyColorFree(CapyColor** const that);

// ColorPalette object
typedef struct CapyColorPalette {

  // Number of colors in the palette
  size_t size;

  // Colors in the palette
  CapyColorData* colors;

  // Destructor
  void (*destruct)(void);

  // Set a color in the palette
  // Input:
  //   iColor: the index of the color
  //   color: the color to set
  // Output:
  //   The color is set.
  void (*set)(
           size_t const iColor,
    CapyColorData const color);

  // Get a color in the palette
  // Input:
  //   iColor: the index of the color
  // Output:
  //   Return the color.
  CapyColorData (*get)(size_t const iColor);

  // Get a color interpolated from the colors of the palette
  // Input:
  //   x: interpolation value in [0,that->size-1]
  // Output:
  //   Return the color.
  CapyColorData (*getInterpolated)(double const x);

  // Set random colors in the palette
  // Input:
  //   rng: the RNG to be used
  // Output:
  //   The RGB colors are set.
  void (*setRandomRGB)(CapyRandom* const rng);

  // Set random pastel colors in the palette
  // Input:
  //   rng: the RNG to be used
  // Output:
  //   The RGB colors are set.
  void (*setRandomPastelRGB)(CapyRandom* const rng);
} CapyColorPalette;

// Create a CapyColorPalette
// Input:
//   size: the number of colors in the palette
// Output:
//   Return a CapyColorPalette, all colors initialised to capyColorRGBABlack
CapyColorPalette CapyColorPaletteCreate(size_t const size);

// Allocate memory for a new CapyColorPalette and create it
// Input:
//   size: the number of colors in the palette
// Output:
//   Return a CapyColorPalette, all colors initialised to capyColorRGBABlack
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColorPalette* CapyColorPaletteAlloc(size_t const size);

// Free the memory used by a CapyColorPalette* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyColorPalette to free
void CapyColorPaletteFree(CapyColorPalette** const that);
#endif
