// ------------------- colorChart.c --------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "colorChart.h"
#include "diffevo.h"
#include "floodFill.h"
#include "graph.h"
#include "pen.h"
#include "pointCloud.h"
#include "kmeans.h"

// QP203 color chart data (stored per rows, from top left to bottom right)
// In the patent, based on the grey patches ordering, the diagram in
// figure 2 puts the top-left (in LibCapy) to the bottom-right and the
// bottom-right (in LibCapy) to the top-left. The values in the patent
// are given for (D50, 2 degrees)
// QP203 has 9 color patches with value out of the sRGB gamut
CapyColorChartDim const qp203Dims = { .nbCol = 5, .nbRow = 7 };
CapyColorData const qp203LAB[35] = {
  {.vals = {90.1, 4.4, 39.2, 1.0}},
  {.vals = {84.3, 20.8, 31.4, 1.0}},
  {.vals = {82.4, 24.6, 14.9, 1.0}},
  {.vals = {81.9, 17.3, -8.8, 1.0}},
  {.vals = {30.1, 0.6, 0.4, 1.0}},
  {.vals = {84.9, -10.5, 31.8, 1.0}},
  {.vals = {53.4, -31.3, -24.4, 1.0}},
  {.vals = {56.3, -42.3, -13.3, 1.0}},
  {.vals = {54.8, -41.0, -2.6, 1.0}},
  {.vals = {34.7, 0.4, 1.0, 1.0}},
  {.vals = {80.1, -22.4, 13.1, 1.0}},
  {.vals = {70.9, -1.5, 56.6, 1.0}},
  {.vals = {69.4, -11.4, 53.1, 1.0}},
  {.vals = {64.6, -23.5, 41.3, 1.0}},
  {.vals = {43.8, 0.2, 0.7, 1.0}},
  {.vals = {85.1, -22.9, -6.1, 1.0}},
  {.vals = {42.8, -17.1, -35.6, 1.0}},
  {.vals = {41.2, 3.4, -46.4, 1.0}},
  {.vals = {39.0, 13.9, -42.5, 1.0}},
  {.vals = {58.0, 0.4, 0.2, 1.0}},
  {.vals = {73.6, 36.1, 57.2, 1.0}},
  {.vals = {51.0, -50.5, 13.9, 1.0}},
  {.vals = {53.5, -47.5, 24.0, 1.0}},
  {.vals = {51.2, -39.1, 29.7, 1.0}},
  {.vals = {68.8, 0.3, 1.2, 1.0}},
  {.vals = {81.9, 1.7, 76.7, 1.0}},
  {.vals = {43.6, 50.7, 21.9, 1.0}},
  {.vals = {44.8, 55.5, 15.8, 1.0}},
  {.vals = {43.9, 50.6, 2.9, 1.0}},
  {.vals = {84.8, 0.0, 1.5, 1.0}},
  {.vals = {86.8, 1.7, 76.7, 1.0}},
  {.vals = {56.1, 27.9, -23.9, 1.0}},
  {.vals = {55.5, 34.5, -16.9, 1.0}},
  {.vals = {56.7, 37.5, -5.7, 1.0}},
  {.vals = {94.2, 0.0, 2.8, 1.0}},
};

// Values converted from the sRGB values
CapyColorData const qp203sRGB[35] = {
  {.vals = {0.996533, 0.872070, 0.592504, 1.0}},
  {.vals = {1.026120, 0.764890, 0.596604, 1.0}},
  {.vals = {1.004343, 0.735471, 0.699773, 1.0}},
  {.vals = {0.903203, 0.755539, 0.865774, 1.0}},
  {.vals = {0.282501, 0.276643, 0.275571, 1.0}},
  {.vals = {0.820156, 0.850921, 0.591111, 1.0}},
  {.vals = {-0.427461, 0.561622, 0.661414, 1.0}},
  {.vals = {-0.672750, 0.605908, 0.614731, 1.0}},
  {.vals = {-0.259003, 0.586625, 0.526165, 1.0}},
  {.vals = {0.324622, 0.318961, 0.313783, 1.0}},
  {.vals = {0.630269, 0.825285, 0.678686, 1.0}},
  {.vals = {0.757600, 0.677314, 0.247839, 1.0}},
  {.vals = {0.672456, 0.683465, 0.259528, 1.0}},
  {.vals = {0.518957, 0.657171, 0.307682, 1.0}},
  {.vals = {0.408757, 0.405315, 0.401366, 1.0}},
  {.vals = {0.612858, 0.884673, 0.874928, 1.0}},
  {.vals = {-0.278066, 0.435026, 0.626285, 1.0}},
  {.vals = {0.187266, 0.384650, 0.682410, 1.0}},
  {.vals = {0.310824, 0.338604, 0.634096, 1.0}},
  {.vals = {0.549793, 0.545551, 0.545208, 1.0}},
  {.vals = {1.012650, 0.599175, 0.292847, 1.0}},
  {.vals = {-0.485520, 0.557026, 0.374207, 1.0}},
  {.vals = {-0.066259, 0.578671, 0.326788, 1.0}},
  {.vals = {0.201267, 0.542958, 0.264656, 1.0}},
  {.vals = {0.663264, 0.657245, 0.649743, 1.0}},
  {.vals = {0.919747, 0.785991, 0.141873, 1.0}},
  {.vals = {0.706655, 0.226549, 0.275601, 1.0}},
  {.vals = {0.739117, 0.210719, 0.326579, 1.0}},
  {.vals = {0.692929, 0.234876, 0.397926, 1.0}},
  {.vals = {0.833947, 0.829863, 0.819098, 1.0}},
  {.vals = {0.977994, 0.839357, 0.211533, 1.0}},
  {.vals = {0.654634, 0.459017, 0.693318, 1.0}},
  {.vals = {0.704800, 0.429353, 0.640594, 1.0}},
  {.vals = {0.756215, 0.428011, 0.578329, 1.0}},
  {.vals = {0.941713, 0.933980, 0.913389, 1.0}},
};

// Flags for the QP203's swatches inside the sRGB gamut
bool const qp203IsInsideSRGBGamut[35] = {
  true, false, false, true, true,
  true, false, false, false, true,
  true, true, true, true, true,
  true, false, true, true, true,
  false, false, false, true, true,
  true, true, true, true, true,
  true, true, true, true, true,
};

// X-Rite ColorChecker Classic color chart data (stored per rows, from top
// left to bottom right)
CapyColorChartDim const xriteClassicDims = { .nbCol = 6, .nbRow = 4 };
CapyColorData const xriteClassicLAB[24] = {
  {.vals = {37.986, 13.555, 14.059, 1.0}},
  {.vals = {65.711, 18.13, 17.81, 1.0}},
  {.vals = {49.927, -4.88, -21.905, 1.0}},
  {.vals = {43.139, -13.095, 21.905, 1.0}},
  {.vals = {55.112, 8.844, -25.399, 1.0}},
  {.vals = {70.719, -33.397, -0.199, 1.0}},
  {.vals = {62.661, 36.067, 57.096, 1.0}},
  {.vals = {40.02, 10.41, -45.964, 1.0}},
  {.vals = {51.124, 48.239, 16.248, 1.0}},
  {.vals = {30.325, 22.976, -21.587, 1.0}},
  {.vals = {72.532, -23.709, 57.255, 1.0}},
  {.vals = {71.941, 19.363, 67.857, 1.0}},
  {.vals = {28.778, 14.179, -50.297, 1.0}},
  {.vals = {55.261, -38.342, 31.37, 1.0}},
  {.vals = {42.101, 53.378, 28.19, 1.0}},
  {.vals = {81.733, 4.039, 79.819, 1.0}},
  {.vals = {51.935, 49.986, -14.574, 1.0}},
  {.vals = {51.038, -28.631, -28.638, 1.0}},
  {.vals = {96.539, -0.425, 1.186, 1.0}},
  {.vals = {81.257, -0.638, -0.335, 1.0}},
  {.vals = {66.766, -0.734, -0.504, 1.0}},
  {.vals = {50.867, -0.153, -0.27, 1.0}},
  {.vals = {35.656, -0.421, -1.231, 1.0}},
  {.vals = {20.461, -0.079, -0.973, 1.0}},
};

CapyColorData const xriteClassicsRGB[24] = {
  {.vals = {0.460810, 0.316802, 0.264199}},
  {.vals = {0.790465, 0.576589, 0.504671}},
  {.vals = {0.323720, 0.482184, 0.609952}},
  {.vals = {0.365435, 0.419940, 0.254701}},
  {.vals = {0.483064, 0.505729, 0.689195}},
  {.vals = {0.352379, 0.743261, 0.675789}},
  {.vals = {0.900591, 0.483027, 0.184184}},
  {.vals = {0.168241, 0.365415, 0.665606}},
  {.vals = {0.787728, 0.323752, 0.378670}},
  {.vals = {0.354565, 0.232198, 0.412697}},
  {.vals = {0.655619, 0.736030, 0.256634}},
  {.vals = {0.930347, 0.629922, 0.161836}},
  {.vals = {-0.128022, 0.257862, 0.575096}},
  {.vals = {0.294038, 0.580342, 0.295686}},
  {.vals = {0.716994, 0.193636, 0.220621}},
  {.vals = {0.968123, 0.772911, 0.105505}},
  {.vals = {0.754112, 0.336847, 0.588036}},
  {.vals = {-0.846665, 0.536141, 0.664954}},
  {.vals = {0.961635, 0.961458, 0.951774}},
  {.vals = {0.785337, 0.792971, 0.793731}},
  {.vals = {0.629686, 0.638582, 0.640223}},
  {.vals = {0.473007, 0.475286, 0.476651}},
  {.vals = {0.322290, 0.329908, 0.336352}},
  {.vals = {0.190154, 0.193653, 0.198807}},
};

// Flags for the X-Rite's swatches inside the sRGB gamut
bool const xriteClassicIsInsideSRGBGamut[24] = {
  true, true, true, true, true, true,
  true, true, true, true, true, true,
  false, true, true, true, true, false,
  true, true, true, true, true, true,
};

// Default options for the color chart detection
CapyColorChartDetectOpt const capyColorChartDetectDefaultOpt = {
  .stdDev = 1.1,
  .gaussKernelSize = 4.0,
  .flagScoreGradient = true,
  .thresholdAspectRatio = 0.5,
  .thresholdCoverage = 0.4,
  .thresholdParallelism = 0.15,
  .thresholdPairing = 0.5,
  .nbStepDiffusion = 5,
  .hysteresis = {0.1, 0.25},
  .flagKeepPairing = true,
  .flagKmeansClustering = false,
};

// Get the color at a given position in the chart
static CapyColorData Get(
  size_t const iRow,
  size_t const iCol) {
  methodOf(CapyColorChart);
  return that->colors[iRow * that->nbCol + iCol];
}

// Flooding condition function
// Input:
//    img: the image to flood
//   from: the source pixel
//     to: the destination pixel
// Output:
//   Return true if the flooding propagate from 'from' to 'to'
static bool FloodConditionFun(
       CapyImg const* const img,
  CapyImgPixel const* const from,
  CapyImgPixel const* const to) {

  // Flood if the difference in intensity is less than 0.5
  double diffIntensity =
    fabs(img->pixels[from->idx].intensity - img->pixels[to->idx].intensity);
  return (diffIntensity < 0.5);
}

// Check if a quadrilateral is approximately a parallelogram
// Input:
//   quad: the quadrilateral
//    opt: the detection options
// Output:
//   Return true if the condition is fulfilled
static bool CheckQuadParallelism(
        CapyQuadrilateral const* const quad,
  CapyColorChartDetectOpt const* const opt) {
  double length[4] = {0., 0., 0., 0.};
  loop(i, 4) {
    length[i] =
      sqrt(
        pow(quad->corners[i].x - quad->corners[(i + 1) % 4].x, 2.0) +
        pow(quad->corners[i].y - quad->corners[(i + 1) % 4].y, 2.0));
  }
  double avg[2] = {0., 0.};
  loop(i, 2) avg[i] = 0.5 * (length[i] + length[i + 2]);
  loop(i, 4) {
    if(fabs(length[i] - avg[i % 2]) / avg[i % 2] > opt->thresholdParallelism) {
      return false;
    }
  }
  return true;
}

// Check if a quadrilateral has a valid aspect ratio
// Input:
//   quad: the quadrilateral
//    opt: the detection options
// Output:
//   Return true if the condition is fulfilled
static bool CheckQuadAspectRatio(
        CapyQuadrilateral const* const quad,
  CapyColorChartDetectOpt const* const opt) {
  double length[4] = {0., 0., 0., 0.};
  loop(i, 4) {
    length[i] =
      sqrt(
        pow(quad->corners[i].x - quad->corners[(i + 1) % 4].x, 2.0) +
        pow(quad->corners[i].y - quad->corners[(i + 1) % 4].y, 2.0));
  }
  double sum[2] = {0., 0.};
  loop(i, 2) sum[i] = length[i] + length[i + 2];
  double ratio = sum[0] / sum[1];
  if(
    ratio < opt->thresholdAspectRatio ||
    ratio > 1.0 / opt->thresholdAspectRatio
  ) {
    return false;
  }
  return true;
}

// Check if the approximating quadrilateral covers its patch correctly
// Input:
//   patch: the patch
//     opt: the detection options
// Output:
//   Return true if the condition is fulfilled
static bool CheckQuadCoverage(
           CapyColorPatch const* const patch,
  CapyColorChartDetectOpt const* const opt) {

  // Variable to count the number of pixels in the segment which are
  // in the quadrialteral
  size_t nbPixelIn = 0;

  // Loop on the pixels in the segment
  forEach(pixel, (((CapyColorPatch*)patch)->pixels.iter)) {

    // Get the bilinear coordinates of the pixel in the quadrilateral
    double pos[2] = {(double)pixel.pos.x, (double)pixel.pos.y};
    double coords[2];
    $(&(patch->approxQuad), getBilinearCoords)(pos, coords);

    // If the pixel is in the quadrilateral, increment the counter
    if(
      coords[0] >= 0.0 && coords[0] <= 1.0 &&
      coords[1] >= 0.0 && coords[1] <= 1.0
    ) {
      ++nbPixelIn;
    }
  }

  // Check that most of the pixels in the segment are in the
  // quadrilateral
  double ratio = (double)nbPixelIn / (double)$(&(patch->pixels), getSize)();
  if(ratio < 1.0 - opt->thresholdCoverage) return false;

  // Check that most of the quadrilateral is filled by the segment's
  // pixels
  double area = $(&(patch->approxQuad), getArea)();
  ratio = (double)nbPixelIn / area;
  if(ratio < 1.0 - opt->thresholdCoverage) return false;
  return true;
}

// Get the homogeneous patches in an image
// Input:
//      that: the ColorChart
//       img: the edge map
//   patches: the homogeneous patches
//       opt: the detection options
// Output:
//   Return the set of homogeneous patches
static CapyColorPatches* FilterPatches(
                 CapyColorChart* const that,
                  CapyImg const* const img,
               CapyColorPatches* const patches,
  CapyColorChartDetectOpt const* const opt) {

  // Variable to memorise the valid patches
  CapyColorPatches* validPatches = CapyColorPatchesAlloc();

  // Get the minimum and maximum area for one color patch in the image
  size_t minAreaPatch = that->kernelSize * that->kernelSize;
  size_t nbPixels = $(img, getNbPixels)();
  size_t maxAreaPatch = nbPixels / (that->nbRow * that->nbCol);

  // If the patch is too small, the approximating quadrilateral will be
  // too inaccurate, then ensure a minimum number of pixels in the patches
  if(minAreaPatch < 50) minAreaPatch = 50;

  // Loop on the homogeneous patches
  while($(patches, getNbPatch)() > 0) {
    CapyColorPatch patch = $(patches->list, pop)();
    $(&(patch.pixels), initIterator)();

    // Variable to memorise if the segment is valid
    bool isValid = true;

    // Check that the patch is not an edge
    size_t idx = patch.pixels.data[0].idx;
    isValid &= (img->pixels[idx].intensity < 0.5);

    // Check the size of the patch
    if(isValid) {
      size_t areaPatch = $(&(patch.pixels), getSize)();
      isValid &= (areaPatch >= minAreaPatch);
      isValid &= (areaPatch <= maxAreaPatch);
    }

    // Check the validity of the patch based on the quadrilateral properties
    if(isValid) isValid = CheckQuadParallelism(&(patch.approxQuad), opt);
    if(isValid) isValid = CheckQuadAspectRatio(&(patch.approxQuad), opt);
    if(isValid) isValid = CheckQuadCoverage(&patch, opt);

    // If the patch is valid, add it to the valid patches
    if(isValid) $(validPatches->list, add)(patch);

    // Else, free it
    else $(&patch, destruct)();
  }

  // Return the valid patches
  return validPatches;
}

// Get the homogeneous patches in an image
// Input:
//   that: the ColorChart
//    img: the image
//    opt: the detection options
// Output:
//   Return the set of homogeneous patches
static CapyColorPatches* GetHomogeneousPatches(
                  CapyColorChart* const that,
                   CapyImg const* const img,
   CapyColorChartDetectOpt const* const opt) {

  // Create the homogeneous image (black pixels are homogeneous patch,
  // other pixels are white) by using Canny edge detection
  CapyImg* edgeMap = NULL;
  if(opt->flagKmeansClustering) {
    CapyImg* edgeMapSrc = CapyImgClone(img);
    CapyPointCloud* pcd = $(edgeMapSrc, getColorsAsPointCloud)();
    CapyKMeans* kmeans = CapyKMeansAlloc();
    $(kmeans, run)(pcd, img->dims.width * img->dims.height / 1000);
    $(kmeans, applyToImgColor)(edgeMapSrc);
    CapyPointCloudFree(&pcd);
    CapyKMeansFree(&kmeans);
    edgeMap = $(edgeMapSrc, getCannyEdgeMap3Chan)(
      opt->gaussKernelSize, opt->stdDev,
      opt->hysteresis[0],
      opt->hysteresis[1]);
    CapyImgFree(&edgeMapSrc);
  } else {
    edgeMap = $(img, getCannyEdgeMap3Chan)(
      opt->gaussKernelSize, opt->stdDev,
      opt->hysteresis[0],
      opt->hysteresis[1]);
  }

  // Segment the homogeneous image with flood fill algorithm
  CapyFloodFill flood = CapyFloodFillCreate();
  CapyColorPatches* patches =
    $(&flood, segmentImg)(img, edgeMap, FloodConditionFun);
  $(&flood, destruct)();

  // For each patch
  forEach(patch, patches->list->iter) {

    // Update the average color, quadrilateral and center of mass
    $(patchPtr, updateAvgColor)();
    $(patchPtr, updatePosCenterOfMass)();
    $(patchPtr, updateApproxQuadrilateral)();
  }

  // Filter the homogeneous patches to keep only those who are possibly
  // color patch
  CapyColorPatches* validPatches =
    FilterPatches(that, edgeMap, patches, opt);

  // Free memory
  CapyImgFree(&edgeMap);
  CapyColorPatchesFree(&patches);

  // Return the valid patches
  return validPatches;
}

// Structures to memorise the data during color patch matching
typedef struct {

  // Number of patches
  size_t nbPatch;

  // Neighbour relations based on location and approximating quadrilateral
  // links[i].from is i-th CapyColorPatch in the list of valid color
  // patches in the image
  struct {
    CapyColorPatch const* from;
    CapyColorPatch const* to[4];
    size_t toIdx[4];
    bool paired;
    CapyPad(bool, 0);
  }* links;

  // Matching score for a pair (color patch in image, color patch in chart)
  // Higher is better
  // scores[i*nbPatch+j] is the score for the pair (j-th color patch in
  // image, i-th color patch in chart)
  double* scores;

  // Pairs of matching patches in the image and chart
  // pairs[i] is the pair for the i-th patch in the color chart
  // If pairs[i].score is negative, there is no pairing for the i-th patch
  // in the color chart
  struct {
    size_t iPatchImg;
    double score;
  }* pairs;

  // Number of successfully paired patches in the color chart
  size_t nbPairedPatch;

  // Destructor
  void (*destruct)(void);
} CapyPatchMatching;

static void PatchMatchingDestruct(void) {
  methodOf(CapyPatchMatching);
  free(that->links);
  free(that->scores);
  free(that->pairs);
}

static CapyPatchMatching CapyPatchMatchingCreate(
  size_t const nbPatch,
  size_t const sizeChart) {
  CapyPatchMatching matching = {
    .nbPatch = nbPatch,
    .nbPairedPatch = 0,
    .destruct = PatchMatchingDestruct,
  };
  safeMalloc(matching.links, nbPatch);
  safeMalloc(matching.scores, nbPatch * sizeChart);
  safeMalloc(matching.pairs, sizeChart);
  if(matching.pairs) loop(iPatchChart, sizeChart) {
    matching.pairs[iPatchChart].score = -1.0;
  }
  return matching;
}

// Create the neighbouring relations between color patches in the image based
// on location and approximating quadrilateral. Two patches are considered
// neighbour if their relative position is around the size and direction of
// one side of the quadrilateral.
// Input:
//    patches: the valid patches
//   matching: the struct updated with the neighbour relations
//        img: the image
static void CreateNeighbourRelations(
  CapyColorPatches const* const patches,
       CapyPatchMatching* const matching) {

  // Loop on the patches
  forEach(patch, patches->list->iter) {

    // Initialise the link for this patch and direction
    matching->links[patches->list->iter.idx].from = patchPtr;

    // Loop on the four directions
    loop(iDir, 4) {

      // Initialise the link for this patch and direction
      size_t iPatch = patches->list->iter.idx;
      matching->links[iPatch].to[iDir] = NULL;

      // Get the expected location of the neighbour in that direction.
      // It is the center of mass of the patch translated by the side
      // of the bounding box in that direction
      // Increase a bit the edge length to account for the space between the
      // color patches
      double edge[2];
      edge[0] =
        patch.approxQuad.corners[(iDir + 1) % 4].x -
        patch.approxQuad.corners[iDir].x;
      edge[1] =
        patch.approxQuad.corners[(iDir + 1) % 4].y -
        patch.approxQuad.corners[iDir].y;
      double expectedPos[2] = {
        (double)patch.centerOfMass.x + edge[0],
        (double)patch.centerOfMass.y + edge[1]
      };

      // Variables to search the other patch nearest to the expected location
      double minDist = 0.0;
      CapyColorPatch const* nearest = NULL;
      size_t iNearest = 0;

      // Loop on the other patches
      CapyListColorPatchIterator iterOther = patches->list->iter;
      forEach(otherPatch, iterOther) if(otherPatch.idx != patch.idx) {

        // Search the other patch with center of mass nearest to the expected
        // location
        double dist = 0.0;
        dist += pow(expectedPos[0] - (double)otherPatch.centerOfMass.x, 2.0);
        dist += pow(expectedPos[1] - (double)otherPatch.centerOfMass.y, 2.0);
        dist = sqrt(dist);
        if(nearest == NULL || dist < minDist) {
          nearest = otherPatchPtr;
          iNearest = iterOther.idx;
          minDist = dist;
        }
      }

      // If the nearest patch is in a realistic range from the expected
      // location (nearest!=NULL for the case there is only one patch),
      // memorise it as the neighbour in that direction
      double distMax = sqrt(edge[0] * edge[0] + edge[1] * edge[1]);
      if(nearest != NULL && minDist < distMax) {
        matching->links[iPatch].to[iDir] = nearest;
        matching->links[iPatch].toIdx[iDir] = iNearest;
      }
    }
  }
}

// Calculate the matching score for a given pair (color patch in image,
// color patch in chart) based on the difference of color gradient in chart
// and image
// Input:
//         that: the color chart
//          img: the image
//     matching: the struct updated with the neighbour relations
//    iPatchImgOrig: the index of the image patch in the list of valid
//                    patches
//  iPatchChartOrig: the index of the color chart patch matching->links
// Output:
//   Return the score.
static double CalculateMatchingScoreGradient(
  CapyColorChart const* const that,
         CapyImg const* const img,
     CapyPatchMatching* const matching,
                 size_t const iPatchImgOrig,
                 size_t const iPatchChartOrig) {

  // Variable to memorise the best score over rotation
  double bestScoreRot = 0.0;

  // Variable to memorise the row and column of the original patch in chart
  struct {size_t iRow, iCol;} coordPatchChartOrig = {
    .iRow = iPatchChartOrig / that->nbCol,
    .iCol = iPatchChartOrig % that->nbCol
  };

  // Get the patch color in image and reference
  CapyColorPatch const* imgPatchOrig = matching->links[iPatchImgOrig].from;
  CapyColorData const* chartPatchColorOrig = that->colors + iPatchChartOrig;

  // Loop over the rotations
  loop(iRot, 4) {

    // Variable to memorise the score for this rotation
    double scoreRot = 0.0;

    // Calculate the vectors to calculate the expected position of the
    // neighbours for this rotation
    double vecRow[2] = {0, 0};
    double vecCol[2] = {0, 0};
    bool flagRow = false;
    bool flagCol = false;
    if(matching->links[iPatchImgOrig].to[iRot] != NULL) {
      flagRow = true;
      CapyColorPatch const* imgPatchDest =
        matching->links[iPatchImgOrig].to[iRot];
      loop(i, 2) {
        vecRow[i] =
          (double)imgPatchDest->centerOfMass.coords[i] -
          (double)imgPatchOrig->centerOfMass.coords[i];
      }
    } else if(matching->links[iPatchImgOrig].to[(iRot + 2) % 4] != NULL) {
      flagRow = true;
      CapyColorPatch const* imgPatchDest =
        matching->links[iPatchImgOrig].to[(iRot + 2) % 4];
      loop(i, 2) {
        vecRow[i] =
          (double)imgPatchOrig->centerOfMass.coords[i] -
          (double)imgPatchDest->centerOfMass.coords[i];
      }
    }
    if(matching->links[iPatchImgOrig].to[(iRot + 1) % 4] != NULL) {
      flagCol = true;
      CapyColorPatch const* imgPatchDest =
        matching->links[iPatchImgOrig].to[(iRot + 1) % 4];
      loop(i, 2) {
        vecCol[i] =
          (double)imgPatchDest->centerOfMass.coords[i] -
          (double)imgPatchOrig->centerOfMass.coords[i];
      }
    } else if(matching->links[iPatchImgOrig].to[(iRot + 3) % 4] != NULL) {
      flagCol = true;
      CapyColorPatch const* imgPatchDest =
        matching->links[iPatchImgOrig].to[(iRot + 3) % 4];
      loop(i, 2) {
        vecCol[i] =
          (double)imgPatchOrig->centerOfMass.coords[i] -
          (double)imgPatchDest->centerOfMass.coords[i];
      }
    }

    // If the vectors necessary to calculate the expected position are
    // known
    if(flagRow && flagCol) {

      // Loop on the color chart patches
      loop(iRow, that->nbRow) loop(iCol, that->nbCol) {
        size_t iPatchChart = iCol + iRow * that->nbCol;
        struct {size_t iRow, iCol;} coordPatchChart = {
          .iRow = iRow,
          .iCol = iCol
        };

        // If it's not the original patch in chart
        if(
          coordPatchChart.iRow != coordPatchChartOrig.iRow ||
          coordPatchChart.iCol != coordPatchChartOrig.iCol
        ) {

          // Get the patch color in reference
          CapyColorData const* chartPatchColor = that->colors + iPatchChart;

          // Calculate the relative location of that patch in chart
          double relPosChart[2] = {
            (double)coordPatchChart.iRow - (double)coordPatchChartOrig.iRow,
            (double)coordPatchChart.iCol - (double)coordPatchChartOrig.iCol,
          };

          // Calculate the expected location in the image of that chart patch
          CapyImgPos expectedPos = imgPatchOrig->centerOfMass;
          loop(i, 2) {
            expectedPos.coords[i] += (CapyImgPos_t)(
              vecRow[i] * relPosChart[1] + vecCol[i] * relPosChart[0]);
          }

          // Calculate the gradient of color in the chart
          double gradientChart[3] = {0., 0., 0.};
          loop(i, 3) {
            gradientChart[i] =
              chartPatchColorOrig->vals[i] - chartPatchColor->vals[i];
          }

          // Calculate the average color at the expected location
          CapyColorValue_t avgColor[3] = {0.0, 0.0, 0.0};
          int8_t kernelSize = 2;
          CapyRangeInt8 rangeKernel = {.min = -kernelSize, .max = kernelSize};
          size_t nbPixel = 0;
          loopRange(i, rangeKernel) loopRange(j, rangeKernel) {
            CapyImgPos posKernel = expectedPos;
            posKernel.x += i;
            posKernel.y += j;
            bool isPosInImg = $(img, isValidCoord)(&posKernel);
            if(isPosInImg) {
              CapyColorData const* color = $(img, getColor)(&posKernel);
              loop(k, 3) avgColor[k] += color->vals[k];
              ++nbPixel;
            }
          }
          if(nbPixel > 0) loop(i, 3) avgColor[i] /= (double)nbPixel;

          // Calculate the gradient of color in the image
          double gradientImg[3] = {0., 0., 0.};
          loop(i, 3) {
            gradientImg[i] = imgPatchOrig->avgColor.vals[i] - avgColor[i];
          }

          // Update the score based on difference of gradient
          double diff[3] = {0., 0., 0.};
          loop(i, 3) diff[i] = gradientChart[i] - gradientImg[i];
          double n = 0.0;
          loop(i, 3) n += pow(diff[i], 2.0);
          scoreRot += 1.0 / (1.0 + sqrt(n));
        }
      }
    }

    // Update the best score over rotation
    if(bestScoreRot < scoreRot) bestScoreRot = scoreRot;
  }

  // Return the score
  return bestScoreRot;
}

// Calculate the matching score for each pair (color patch in image,
// color patch in chart)
// Input:
//       that: the color chart
//        img: the image
//   matching: the struct updated with the neighbour relations
//        opt: detection options
static void CalculateMatchingScore(
              CapyColorChart const* const that,
                     CapyImg const* const img,
                 CapyPatchMatching* const matching,
     CapyColorChartDetectOpt const* const opt) {

  // Loop on the pairs
  size_t nbPatchChart = that->nbCol * that->nbRow;
  loop(iPatchImg, matching->nbPatch) loop(iPatchChart, nbPatchChart) {

    // Variable to memorise the score
    double score = 0.0;

    // Get the patch color in image and reference
    CapyColorData const* imgPatchColor =
      &(matching->links[iPatchImg].from->avgColor);
    CapyColorData const* chartPatchColor = that->colors + iPatchChart;

    // Update the score based on the difference in patches' color
    double n = 0.0;
    loop(i, 3) {
      n += pow(chartPatchColor->vals[i] - imgPatchColor->vals[i], 2.0);
    }
    score += 1.0 / (1.0 + sqrt(n));

    // Update the score based on the difference of gradient relative to
    // all other patches
    if(opt->flagScoreGradient) {
      score +=
        CalculateMatchingScoreGradient(
          that, img, matching, iPatchImg, iPatchChart);
    }

    // Set the score
    matching->scores[iPatchChart * matching->nbPatch + iPatchImg] = score;
  }
}

// Diffuse the matching scores through the links
// Input:
//       that: the color chart
//   matching: the struct updated with the neighbour relations
//        opt: the detection options
static void DiffuseMatchingScore(
           CapyColorChart const* const that,
              CapyPatchMatching* const matching,
  CapyColorChartDetectOpt const* const opt) {

  // Variable to memorise the updated matching scores
  double* updatedScores = NULL;
  size_t nbPatchChart = that->nbCol * that->nbRow;
  safeMalloc(updatedScores, matching->nbPatch * nbPatchChart);
  if(!updatedScores) return;

  // Loop on the diffusion steps
  loop(iStep, opt->nbStepDiffusion) {

    // Maximum score, used to normalise
    double maxScore = 0.0;

    // Loop on the pairs
    loop(iPatchImg, matching->nbPatch) loop(iPatchChart, nbPatchChart) {

      // Variable to memorise the row and column of the patch in chart
      struct {size_t iRow, iCol;} coordPatchChart = {
        .iRow = iPatchChart / that->nbCol,
        .iCol = iPatchChart % that->nbCol
      };

      // Initialise the updated score
      size_t iScore = iPatchChart * matching->nbPatch + iPatchImg;
      updatedScores[iScore] = matching->scores[iScore];

      // Loop on the neighbours of the patch in the image
      loop(iDirImg, 4) if(matching->links[iPatchImg].to[iDirImg] != NULL) {

        // Get the index of the neighbour patch in the image
        size_t idxImgNeighbour = 0;
        CapyColorPatch const* toPatch = matching->links[iPatchImg].to[iDirImg];
        while(matching->links[idxImgNeighbour].from != toPatch) {
          ++idxImgNeighbour;
        }

        // Variable to memorise the best score of this neighbour
        // amongst its scores for the neighbours of the chart patch
        double bestScore = -1.0;

        // Variable to memorise the delta of indices in the color
        // chart given a direction.
        size_t coordNeighbours[4][2] = {
          {coordPatchChart.iCol + 1, coordPatchChart.iRow + 0},
          {coordPatchChart.iCol + 0, coordPatchChart.iRow + 1},
          {coordPatchChart.iCol - 1, coordPatchChart.iRow + 0},
          {coordPatchChart.iCol + 0, coordPatchChart.iRow - 1},
        };

        // Loop on the neighbours of the chart patch
        loop(iDirChart, 4) {
          if(
            coordNeighbours[iDirChart][0] < that->nbCol &&
            coordNeighbours[iDirChart][1] < that->nbRow
          ) {

            // Get the index of the neighbour chart patch
            size_t idxChartNeighbour =
              coordNeighbours[iDirChart][1] * that->nbCol +
              coordNeighbours[iDirChart][0];

            // Get the value of the neighbour in the image for this
            // neighbour in the chart
            size_t iNeighbourScore =
              idxChartNeighbour * matching->nbPatch + idxImgNeighbour;
            double scoreNeighbour = matching->scores[iNeighbourScore];

            // Update the best score for this neighbour
            if(bestScore < 0.0 || bestScore < scoreNeighbour) {
              bestScore = scoreNeighbour;
            }
          }
        }

        // Increase the score of the current patch with the best
        // score of its neighbour
        updatedScores[iScore] += bestScore;
      }

      // Update the maximum score
      if(maxScore < updatedScores[iScore]) maxScore = updatedScores[iScore];
    }

    // Update the matching scores
    memcpy(
      matching->scores,
      updatedScores,
      matching->nbPatch * nbPatchChart * sizeof(double));

    // Normalise the matching scores
    loop(iScore, matching->nbPatch * nbPatchChart) {
      matching->scores[iScore] /= maxScore;
    }
  }

  // Free memory
  free(updatedScores);
}

// Pair the patches in the image and in the chart based on the scores
// Input:
//       that: the color chart
//   matching: the struct updated with the neighbour relations
//        opt: the detection options
static void PairPatches(
           CapyColorChart const* const that,
              CapyPatchMatching* const matching,
  CapyColorChartDetectOpt const* const opt) {

  // Variable to memorise the number of patch in the chart
  size_t nbPatchChart = that->nbCol * that->nbRow;

  // Init the pairing flag
  loop(iPatchImg, matching->nbPatch) matching->links[iPatchImg].paired = false;

  // Init the number of pair
  matching->nbPairedPatch = 0;

  // Flag for the first pairing
  bool flagFirstPair = true;
  bool flagHasJump = false;

  // Loop until we couldn't find new pair
  bool keepPairing = true;
  while(flagFirstPair || !flagHasJump || keepPairing) {
    keepPairing = false;

    // Get the patch in image with highest score and linked to an already
    // paired patch
    size_t iBestPatchImg = 0;
    size_t iBestPatchChart = 0;
    double scoreBestPatchImg = -1.0;
    loop(iPatchImg, matching->nbPatch) {
      if(matching->links[iPatchImg].paired == false) {
        bool hasPairedNeighbour = false;
        loop(iDir, 4) {
          size_t iNeighbourPatchImg =
            matching->links[iPatchImg].toIdx[iDir];
          if(
            matching->links[iPatchImg].to[iDir] != NULL &&
            matching->links[iNeighbourPatchImg].paired
          ) {
            hasPairedNeighbour = true;
          }
        }
        if(flagFirstPair || hasPairedNeighbour) {
          loop(iPatchChart, nbPatchChart) {
            if(matching->pairs[iPatchChart].score < 0.0) {
              size_t iScore = iPatchChart * matching->nbPatch + iPatchImg;
              if(scoreBestPatchImg < matching->scores[iScore]) {
                scoreBestPatchImg = matching->scores[iScore];
                iBestPatchImg = iPatchImg;
                iBestPatchChart = iPatchChart;
              }
            }
          }
        }
      }
    }

    // If we could find a pair and it has a good enough score, assign it
    if(scoreBestPatchImg > opt->thresholdPairing) {
      ++(matching->nbPairedPatch);
      matching->pairs[iBestPatchChart].score = scoreBestPatchImg;
      matching->pairs[iBestPatchChart].iPatchImg = iBestPatchImg;
      matching->links[iBestPatchImg].paired = true;
      flagFirstPair = false;
      flagHasJump = false;
      keepPairing = true;
    } else if(opt->flagKeepPairing == false) {
      flagFirstPair = false;
      flagHasJump = true;
      keepPairing = false;
    } else if(flagHasJump == false) {
      flagFirstPair = true;
      flagHasJump = true;
    } else flagFirstPair = false;
  }
}

// Get the color chart corners coordinates
// Input:
//       that: the color chart
//   matching: the struct updated with the neighbour relations
// Exception:
//   May raise CapyExc_NoColorChartInImage
static CapyQuadrilateral LocateChart(
                 CapyColorChart* that,
  CapyPatchMatching const* const matching) {

  // If there is not enough pairing to calculate the location,
  // raise an exception
  if(matching->nbPairedPatch < 4) raiseExc(CapyExc_NoColorChartInImage);

  // Variable to memorise the result
  CapyQuadrilateral quad = CapyQuadrilateralCreate();

  // Variables to memorise the matrix and vector to calculate the coordinates
  CapyMat M = CapyMatCreate(4, matching->nbPairedPatch);
  CapyVec X[2] = {
    CapyVecCreate(matching->nbPairedPatch),
    CapyVecCreate(matching->nbPairedPatch)
  };

  // Variable to memorise the number of patch in the chart
  size_t nbPatchChart = that->nbCol * that->nbRow;

  // Index of the next updated row in the matrix and vectors
  size_t iRow = 0;

  // Loop on the patch in the chart
  loop(iPatch, nbPatchChart) {

    // If this patch has a pairing
    if(matching->pairs[iPatch].score > 0.0) {

      // Get the coords of the patch in the chart
      struct {size_t iRow, iCol;} coordPatch = {
        .iRow = iPatch / that->nbCol,
        .iCol = iPatch % that->nbCol
      };

      // Calculate the weights for the bilinear interpolation for this patch
      double relPos[2] = {
        (0.5 + (double)coordPatch.iCol) / (double)(that->nbCol),
        (0.5 + (double)coordPatch.iRow) / (double)(that->nbRow)
      };
      double weights[4];
      weights[0] = (1.0 - relPos[1]) * (1.0 - relPos[0]);
      weights[1] = (1.0 - relPos[1]) * relPos[0];
      weights[2] = relPos[1] * (1.0 - relPos[0]);
      weights[3] = relPos[1] * relPos[0];

      // Update the matrix to calculate the coordinates
      loop(iCol, M.nbCol) M.vals[iRow * M.nbCol + iCol] = weights[iCol];

      // Update the vector to calculate the coordinates
      size_t iPatchImg = matching->pairs[iPatch].iPatchImg;
      loop(iDim, 2) {
        X[iDim].vals[iRow] =
          matching->links[iPatchImg].from->centerOfMass.coords[iDim];
      }

      // Update the next row to set in the matrix and vectors
      ++iRow;
    }
  }

  // Inverse the matrix
  CapyMat MInv = CapyMatCreate(matching->nbPairedPatch, 4);
  try {
    CapyMatPseudoInv(&M, &MInv);
  } endCatch;
  if(CapyGetLastExcId() != 0) raiseExc(CapyExc_NoColorChartInImage);

  // Variable to memorise the result coordinates per dimension
  CapyVec coords = CapyVecCreate(4);

  // Loop on the dimensions
  loop(iDim, 2) {

    // Calculate the coordinates
    CapyMatProdVec(&MInv, X + iDim, &coords);

    // Upate the quadrilateral with the result
    size_t idxCtrl[4] = {0, 1, 3, 2};
    loop(iCoord, 4) {
      quad.corners[iCoord].coords[iDim] = coords.vals[idxCtrl[iCoord]];
    }
  }

  // Free memory
  CapyMatDestruct(&M);
  CapyMatDestruct(&MInv);
  CapyVecDestruct(X);
  CapyVecDestruct(X + 1);
  CapyVecDestruct(&coords);

  // Return the result
  return quad;
}

// Locate a color chart in an image
// Input:
//   img: the image containing the chart
//   opt: the detection options
// Output:
//   Return the location of the color chart, clockwise from the
//   top-left corner.
// Exceptions:
//   May raise CapyExc_NoColorChartInImage if no color chart were located.
static CapyQuadrilateral Locate(
                  CapyImg const* const img,
  CapyColorChartDetectOpt const* const opt) {
  methodOf(CapyColorChart);

  // Convert the image in the color space of the chart
  CapyImg* normImg = CapyImgClone(img);
  $(normImg, normalise)();
  CapyImg* convImg = CapyImgClone(normImg);
  $(convImg, convertToColorSpace)(that->colorSpace);
  CapyImgFree(&normImg);

  // Variable to memorise the result
  CapyQuadrilateral location = {
    .corners = {{{{0, 0}}}, {{{0, 0}}}, {{{0, 0}}}, {{{0, 0}}}}
  };

  // Get the valid homogeneous patches in the image
  CapyColorPatches* validPatches = GetHomogeneousPatches(that, convImg, opt);

  // If there is no valid patches, raise an exception
  if($(validPatches, getNbPatch)() == 0) {
    CapyColorPatchesFree(&validPatches);
    raiseExc(CapyExc_NoColorChartInImage);
    return location;
  }

  // Create the neighbouring relations between color patches in the image,
  // and calculate the color gradient between neighbours
  CapyPatchMatching matching = CapyPatchMatchingCreate(
    $(validPatches, getNbPatch)(),
    that->nbRow * that->nbCol);
  CreateNeighbourRelations(validPatches, &matching);

  // Calculate the matching score for each pair (color patch in image,
  // color patch in chart)
  CalculateMatchingScore(that, convImg, &matching, opt);

  // Normalise the scores
  double maxScore = 0.0;
  size_t nbPatchChart = that->nbCol * that->nbRow;
  loop(iScore, matching.nbPatch * nbPatchChart) {
    if(maxScore < matching.scores[iScore]) {
      maxScore = matching.scores[iScore];
    }
  }
  loop(iScore, matching.nbPatch * nbPatchChart) {
    matching.scores[iScore] /= maxScore;
  }

  // Diffuse the matching scores through the links
  DiffuseMatchingScore(that, &matching, opt);

  // Pair the patches in the image and in the chart based on the scores
  PairPatches(that, &matching, opt);
  try {

    // Locate the color chart in the optimised image using differential
    // evolution
    location = LocateChart(that, &matching);
  } endCatch;

  // Free memory
  $(&matching, destruct)();
  CapyColorPatchesFree(&validPatches);
  CapyImgFree(&convImg);

  // Forward eventual exception
  CapyForwardExc();

  // Return the result
  return location;
}

// Update the color values of the chart with those extracted from the
// color chart contained in an image
// Input:
//      img: the image containing the color chart
//   coords: the four coordinates of the corners of the QP203 in the
//           image, as returned by locate(). If NULL, an automatic
//           localisation of the color chart in the image is performed
//           (to be implemented).
// Output:
//   Return true if the colors could be extracted, else false.
static bool Extract(
      CapyImg const* const img,
  CapyQuadrilateral const* chartCoord) {
  methodOf(CapyColorChart);

  // If the coordinates were not given in argument automatically
  // localise the chart
  CapyQuadrilateral quad;
  if(chartCoord == NULL) {
    try {
      quad = $(that, locate)(img, &capyColorChartDetectDefaultOpt);
    } catchDefault {
      return false;
    } endCatch;
    chartCoord = &quad;
  }

  // Set the color space of the chart to match the one of the image
  that->colorSpace = img->colorSpace;

  // Flag to remember if the extraction was successfull
  bool flagSuccess = true;

  // Loop on the colors of the chart
  loop(iCol, that->nbCol) loop(iRow, that->nbRow) {

    // Get the position in image of the color patch
    double relPos[2] = {
      (0.5 + (double)iCol) / (double)(that->nbCol),
      (0.5 + (double)iRow) / (double)(that->nbRow)
    };
    double weights[4] = {0., 0., 0., 0.};
    weights[0] = (1.0 - relPos[1]) * (1.0 - relPos[0]);
    weights[1] = (1.0 - relPos[1]) * relPos[0];
    weights[2] = relPos[1] * relPos[0];
    weights[3] = relPos[1] * (1.0 - relPos[0]);
    CapyImgPos_t posPatch[2] = {0, 0};
    loop(i, 2) loop(j, 4) {
      posPatch[i] +=
        (CapyImgPos_t)(weights[j] * chartCoord->corners[j].coords[i]);
    }

    // Set the color at that position as the color of the chart as seen
    // in the image. Average over a kernel to attenuate noise impact
    size_t idx = iRow * that->nbCol + iCol;
    that->colors[idx] = capyColorRGBABlack;
    loop(i, that->kernelSize) loop(j, that->kernelSize) {
      CapyImgPos posColorImg = {
        .x = posPatch[0] - (CapyImgPos_t)(that->kernelSize / 2 + i),
        .y = posPatch[1] - (CapyImgPos_t)(that->kernelSize / 2 + j)
      };
      CapyColorData const* color = ($(img, getColor)(&posColorImg));
      loop(k, 3) that->colors[idx].RGB[k] += color->RGB[k];
    }
    loop(k, 3) {
      that->colors[idx].RGB[k] /= (double)(that->kernelSize * that->kernelSize);
    }
  }

  // Return the flag to indicates the success of extraction
  return flagSuccess;
}

// Get the position of the center of a patch in an image given the
// position of the corners of the color chart.
// Input:
//   coords: the four coordinates of the corners of the QP203 in the
//           image, as returned by locate().
//   iCol, iRow: the coordinates of the patch in the chart
// Output:
//   Return the position in the image of the center of the patch
static CapyImgPos GetPatchPos(
  CapyQuadrilateral const* const chartCoord,
                    size_t const iCol,
                    size_t const iRow) {
  methodOf(CapyColorChart);

  // Variable to memorise the position of the patch
  CapyImgPos pos = {0};

  // Calculate the position by bilinear interpolation on the corners
  double relPos[2] = {
    (0.5 + (double)iCol) / (double)(that->nbCol),
    (0.5 + (double)iRow) / (double)(that->nbRow)
  };
  double weights[4];
  weights[0] = (1.0 - relPos[1]) * (1.0 - relPos[0]);
  weights[1] = (1.0 - relPos[1]) * relPos[0];
  weights[2] = relPos[1] * relPos[0];
  weights[3] = relPos[1] * (1.0 - relPos[0]);
  loop(i, 2) loop(j, 4) {
    pos.coords[i] +=
      (CapyImgPos_t)(weights[j] * chartCoord->corners[j].coords[i]);
  }

  // Return the position of the patch
  return pos;
}

// Convert the values of the color chart from sRGB to LAB
static void FromRGBToLAB(CapyColorChart* const that) {

  // Color converter
  CapyColor cvt = CapyColorCreate(capyColorRGBABlack);
  cvt.illuminant = that->illuminant;

  // Loop on the colors
  loop(iRow, that->nbRow) loop(iCol, that->nbCol) {

    // Convert the color
    size_t idx = iRow * that->nbCol + iCol;
    loop(i, 4) cvt.vals.vals[i] = that->colors[idx].vals[i];
    cvt.vals = $(&cvt, RGB2XYZ)();
    that->colors[idx] = $(&cvt, XYZ2LAB)();
  }

  // Free memory
  $(&cvt, destruct)();
}

// Convert the values of the color chart from LAB to sRGB
static void FromLABToRGB(CapyColorChart* const that) {

  // Color converter
  CapyColor cvt = CapyColorCreate(capyColorRGBABlack);
  cvt.illuminant = that->illuminant;

  // Loop on the colors
  loop(iRow, that->nbRow) loop(iCol, that->nbCol) {

    // Convert the color
    size_t idx = iRow * that->nbCol + iCol;
    loop(i, 4) cvt.vals.vals[i] = that->colors[idx].vals[i];
    cvt.vals = $(&cvt, LAB2XYZ)();
    that->colors[idx] = $(&cvt, XYZ2RGB)();
  }

  // Free memory
  $(&cvt, destruct)();
}

// Convert the colorChart to another color space
// Input:
//   colorSpace: the target color space
static void ConvertToColorSpace(CapyColorSpace const colorSpace) {
  methodOf(CapyColorChart);
  if(that->colorSpace == capyColorSpace_sRGB) {
    if(colorSpace == capyColorSpace_LAB) {
      FromRGBToLAB(that);
    } else if(colorSpace != capyColorSpace_sRGB) {
      raiseExc(CapyExc_UndefinedExecution);
    }
  } else if(that->colorSpace == capyColorSpace_LAB) {
    if(colorSpace == capyColorSpace_sRGB) {
      FromLABToRGB(that);
    } else if(colorSpace != capyColorSpace_LAB) {
      raiseExc(CapyExc_UndefinedExecution);
    }
  } else raiseExc(CapyExc_UndefinedExecution);
  that->colorSpace = colorSpace;
}

// Correct the color chart to match the brightness of a given color
// chart (sRGB version)
// Input:
//       that: the color chart
//   refChart: the target color chart
static void MatchBrightnessRGB(
        CapyColorChart* const that,
  CapyColorChart const* const refChart) {

  // Range to memorise the range in brightness of the charts
  CapyRangeDouble range = CapyRangeDoubleCreate(1.0, 0.0);
  CapyRangeDouble rangeRef = CapyRangeDoubleCreate(1.0, 0.0);

  // Calculate the range in brightness of the charts
  loop(iRow, that->nbRow) loop(iCol, that->nbCol) loop(i, 3) {
    size_t idx = iRow * that->nbCol + iCol;
    range.min = (
      range.min > that->colors[idx].RGB[i] ?
      that->colors[idx].RGB[i] : range.min);
    range.max = (
      range.max < that->colors[idx].RGB[i] ?
      that->colors[idx].RGB[i] : range.max);
    rangeRef.min = (
      rangeRef.min > refChart->colors[idx].RGB[i] ?
      refChart->colors[idx].RGB[i] : rangeRef.min);
    rangeRef.max = (
      rangeRef.max < refChart->colors[idx].RGB[i] ?
      refChart->colors[idx].RGB[i] : rangeRef.max);
  }

  // Correct the brightness
  loop(iRow, that->nbRow) loop(iCol, that->nbCol) loop(i, 3) {
    size_t idx = iRow * that->nbCol + iCol;
    that->colors[idx].RGB[i] =
      CapyLerp(that->colors[idx].RGB[i], &range, &rangeRef);
  }
}

// Correct the color chart to match the brightness of a given color
// chart (L*a*b* version)
// Input:
//       that: the color chart
//   refChart: the target color chart
static void MatchBrightnessLAB(
        CapyColorChart* const that,
  CapyColorChart const* const refChart) {

  // Set the L* value to the one of the reference and correct the
  // a* and b* values to account for the modification of L*
  loop(iRow, that->nbRow) loop(iCol, that->nbCol) {
    size_t idx = iRow * that->nbCol + iCol;
    double corr = refChart->colors[idx].LAB[0] / that->colors[idx].LAB[0];
    that->colors[idx].LAB[0] = refChart->colors[idx].LAB[0];
    loop(i, 2) that->colors[idx].LAB[1 + i] *= corr;
  }
}

// Correct the color chart to match the brightness of a given color
// chart
// Input:
//   refChart: the target color chart
static void MatchBrightness(CapyColorChart const* const refChart) {
  methodOf(CapyColorChart);

  // Use a clone of the reference chart to convert it in the necessary
  // color space if needed
  CapyColorChart* safeRefChart = CapyColorChartClone(refChart);
  $(safeRefChart, convertToColorSpace)(that->colorSpace);
  switch(that->colorSpace) {
    case capyColorSpace_sRGB:
      MatchBrightnessRGB(that, safeRefChart);
      break;
    case capyColorSpace_LAB:
      MatchBrightnessLAB(that, safeRefChart);
      break;
    default:
      raiseExc(CapyExc_UndefinedExecution);
  }

  // Free memory for the converted color chart
  CapyColorChartFree(&safeRefChart);
}

// Save the color chart to a path.
// Format: type, nbRow, nbCol, colors[0].RGBA[0], colors[0].RGBA[1],
// colors[0].RGBA[2], colors[0].RGBA[3], colors[1].RGBA[0], ...
// Input:
//   path: the path
// Exceptions:
//   May raise CapyExc_StreamOpenError, CapyExc_StreamWriteError
static void SaveToPath(char const* const path) {
  methodOf(CapyColorChart);

  // Open the stream
  CapyStreamIo stream = CapyStreamIoCreate();
  $(&stream, open)(path, "wb");

  // Save to the stream
  $(that, saveToStream)(&stream);

  // Close the stream
  $(&stream, close)();
}

// Save the color chart to a stream (in binary mode).
// Format: type, nbRow, nbCol, colors[0].RGBA[0], colors[0].RGBA[1],
// colors[0].RGBA[2], colors[0].RGBA[3], colors[1].RGBA[0], ...
// Input:
//   stream: the stream
// Exceptions:
//   May raise CapyExc_StreamWriteError
static void SaveToStream(CapyStreamIo* const stream) {
  methodOf(CapyColorChart);

  // Write the data
  $(stream, writeBytes)(
    &(that->type),
    sizeof(CapyColorChartType));
  $(stream, writeBytes)(
    &(that->colorSpace),
    sizeof(CapyColorSpace));
  $(stream, writeBytes)(
    that->dims,
    sizeof(size_t) * 2);
  $(stream, writeBytes)(
    that->colors,
    sizeof(CapyColorData) * that->nbRow * that->nbCol);
}

// Load the color chart from a path.
// Format: type, nbRow, nbCol, colors[0].RGBA[0], colors[0].RGBA[1],
// colors[0].RGBA[2], colors[0].RGBA[3], colors[1].RGBA[0], ...
// Input:
//   path: the path
// Exceptions:
//   May raise CapyExc_StreamOpenError, CapyExc_StreamReadError
static void LoadFromPath(char const* const path) {
  methodOf(CapyColorChart);

  // Open the stream
  CapyStreamIo stream = CapyStreamIoCreate();
  $(&stream, open)(path, "rb");

  // Load from the stream
  $(that, loadFromStream)(&stream);

  // Close the stream
  $(&stream, close)();
}

// Load the color chart from a stream (in binary mode).
// Format: type, nbRow, nbCol, colors[0].RGBA[0], colors[0].RGBA[1],
// colors[0].RGBA[2], colors[0].RGBA[3], colors[1].RGBA[0], ...
// Input:
//   stream: the stream
// Exceptions:
//   May raise CapyExc_StreamWriteError, CapyExc_MallocFailed
static void LoadFromStream(CapyStreamIo* const stream) {
  methodOf(CapyColorChart);

  // Load the data
  $(stream, readBytes)(
    &(that->type),
    sizeof(CapyColorChartType));
  $(stream, readBytes)(
    &(that->colorSpace),
    sizeof(CapyColorSpace));
  $(stream, readBytes)(
    that->dims,
    sizeof(size_t) * 2);
  free(that->colors);
  safeMalloc(
    that->colors,
    that->nbCol * that->nbRow);
  if(!(that->colors)) return;
  $(stream, readBytes)(
    that->colors,
    sizeof(CapyColorData) * that->nbRow * that->nbCol);
}

// Get the degree of similarity of the color chart relative to
// another chart.
// Input:
//   chart: the chart to compare to
// Output:
//   Return 1.0 if the two charts are identical. Return 0.0 if the
//   two charts are of different types. Return 1.0 minus the average
//   of Euclidean distances between two same patches in the charts,
//   hence the lower the returned value (possibly negative) the more
//   charts' color differ. If the two charts are in different color
//   space, a copy of 'chart' is converted to the color space of
//   'that' and used instead.
static double GetSimilarity(CapyColorChart const* const chart) {
  methodOf(CapyColorChart);

  // If the chart have different types, return 0.0
  if(that->type != chart->type) return 0.0;

  // Declare a variable to use the chart in argument or a converted
  // copy of it in case it has a different color space
  CapyColorChart* safeChart = CapyColorChartClone(chart);
  if(that->colorSpace != chart->colorSpace) {
    $(safeChart, convertToColorSpace)(that->colorSpace);
  }
  double sum = 0.0;
  size_t nb = 0;
  loop(iRow, that->nbRow) loop(iCol, that->nbCol) {

    // Variable to memorise the index in the color chart
    size_t idx = iRow * that->nbCol + iCol;

    // Avoid unreliable swatches, whatever the current color space
    if(that->isInsideSRGBGamut[idx]) {

      // Update the sum of differences
      double diff = 0.0;
      loop(i, 3) {
        diff +=
          pow(safeChart->colors[idx].vals[i] - that->colors[idx].vals[i], 2);
      }
      sum += sqrt(diff);
      ++nb;
    }
  }

  // Free memory
  CapyColorChartFree(&safeChart);

  // Return the similarity
  double similarity = 1.0 - sum / (double)nb;
  return similarity;
}

static double GetSimilarityQP203(CapyColorChart const* const chart) {
  methodOf(CapyColorChart);

  // If the chart have different types, return 0.0
  if(that->type != chart->type) return 0.0;

  // Declare a variable to use the chart in argument or a converted
  // copy of it in case it has a different color space
  CapyColorChart* safeChart = CapyColorChartClone(chart);
  if(that->colorSpace != chart->colorSpace) {
    $(safeChart, convertToColorSpace)(that->colorSpace);
  }
  double sum = 0.0;
  size_t nb = 0;
  loop(iRow, that->nbRow) loop(iCol, that->nbCol) {

    // Variable to memorise the index in the color chart
    size_t idx = iRow * that->nbCol + iCol;
    if(safeChart->isInsideSRGBGamut[idx]) {

      // Update the sum of differences
      double diff = 0.0;
      loop(i, 3) {
        diff +=
          pow(safeChart->colors[idx].vals[i] - that->colors[idx].vals[i], 2);
      }
      sum += sqrt(diff);
      ++nb;
    }
  }

  // Free memory
  CapyColorChartFree(&safeChart);

  // Return the similarity
  double similarity = 1.0 - sum / (double)nb;
  return similarity;
}

static double GetSimilarityXRiteClassic(CapyColorChart const* const chart) {
  methodOf(CapyColorChart);

  // If the chart have different types, return 0.0
  if(that->type != chart->type) return 0.0;

  // Declare a variable to use the chart in argument or a converted
  // copy of it in case it has a different color space
  CapyColorChart* safeChart = CapyColorChartClone(chart);
  if(that->colorSpace != chart->colorSpace) {
    $(safeChart, convertToColorSpace)(that->colorSpace);
  }
  double sum = 0.0;
  size_t nb = 0;
  loop(iRow, that->nbRow) loop(iCol, that->nbCol) {

    // Variable to memorise the index in the color chart
    size_t idx = iRow * that->nbCol + iCol;
    if(idx != 12 && idx != 17) {

      // Update the sum of differences
      double diff = 0.0;
      loop(i, 3) {
        diff +=
          pow(safeChart->colors[idx].vals[i] - that->colors[idx].vals[i], 2);
      }
      sum += sqrt(diff);
      ++nb;
    }
  }

  // Free memory
  CapyColorChartFree(&safeChart);

  // Return the similarity
  double similarity = 1.0 - sum / (double)nb;
  return similarity;
}

// Free the memory used by a CapyColorChart
static void Destruct(void) {
  methodOf(CapyColorChart);
  free(that->colors);
  free(that->isInsideSRGBGamut);
}

// Create a CapyColorChart
// Input:
//         type: the type of the chart
//   colorSpace: the color space of the chart
// Output:
//   Return a CapyColorChart
// Exception:
//   May raise CapyExc_MallocFailed, CapyExc_UndefinedExecution.
CapyColorChart CapyColorChartCreate(
  CapyColorChartType const type,
      CapyColorSpace const colorSpace) {
  CapyColorChartDim const* dims = NULL;
  CapyColorData const* colors = NULL;
  bool const* isInsideSRGBGamut = NULL;
  CapyColorIlluminant illuminant = capyColorIlluminant_D65;
  if(type == capyColorChart_QP203) {
    dims = &qp203Dims;
    illuminant = capyColorIlluminant_D50;
    if(colorSpace == capyColorSpace_sRGB) colors = qp203sRGB;
    else if(colorSpace == capyColorSpace_LAB) colors = qp203LAB;
    else {
      raiseExc(CapyExc_UndefinedExecution);
      assert(false && "Unsupported CapyColorChartSpace");

      // Never reaches here, assignment to avoid warning by cppCheck
      colors = qp203sRGB;
    }
    isInsideSRGBGamut = qp203IsInsideSRGBGamut;
  } else if(type == capyColorChart_XRiteClassic) {
    dims = &xriteClassicDims;
    illuminant = capyColorIlluminant_D50;
    if(colorSpace == capyColorSpace_sRGB) colors = xriteClassicsRGB;
    else if(colorSpace == capyColorSpace_LAB) colors = xriteClassicLAB;
    else {
      raiseExc(CapyExc_UndefinedExecution);
      assert(false && "Unsupported CapyColorChartSpace");

      // Never reaches here, assignment to avoid warning by cppCheck
      colors = qp203sRGB;
    }
    isInsideSRGBGamut = xriteClassicIsInsideSRGBGamut;
  } else {
    raiseExc(CapyExc_UndefinedExecution);
    assert(false && "Unsupported CapyColorChartSpace");

    // Never reaches here, assignment to avoid warning by cppCheck
    dims = &qp203Dims;
    colors = qp203sRGB;
    isInsideSRGBGamut = qp203IsInsideSRGBGamut;
  }
  CapyColorChart that = {
    .nbCol = dims->nbCol,
    .nbRow = dims->nbRow,
    .colors = NULL,
    .isInsideSRGBGamut = NULL,
    .kernelSize = 3,
    .colorSpace = colorSpace,
    .illuminant = illuminant,
    .destruct = Destruct,
    .get = Get,
    .locate = Locate,
    .extract = Extract,
    .convertToColorSpace = ConvertToColorSpace,
    .matchBrightness = MatchBrightness,
    .saveToPath = SaveToPath,
    .saveToStream = SaveToStream,
    .loadFromPath = LoadFromPath,
    .loadFromStream = LoadFromStream,
    .getSimilarity = GetSimilarity,
    .getPatchPos = GetPatchPos,
  };
  safeMalloc(that.colors, dims->nbCol * dims->nbRow);
  if(that.colors) {
    memcpy(
      that.colors,
      colors,
      dims->nbCol * dims->nbRow * sizeof(CapyColorData));
  }
  safeMalloc(that.isInsideSRGBGamut, dims->nbCol * dims->nbRow);
  if(that.isInsideSRGBGamut) {
    memcpy(
      that.isInsideSRGBGamut,
      isInsideSRGBGamut,
      dims->nbCol * dims->nbRow * sizeof(bool));
  }

  // Use dedicated similarity functions to avoid the patch
  // out of the sRGB gamut
  if(type == capyColorChart_QP203) that.getSimilarity = GetSimilarityQP203;
  else if(type == capyColorChart_XRiteClassic) {
    that.getSimilarity = GetSimilarityXRiteClassic;
  }
  return that;
}

// Allocate memory for a new CapyColorChart and create it
// Input:
//         type: the type of the chart
//   colorSpace: the color space of the chart
// Output:
//   Return a CapyColorChart
// Exception:
//   May raise CapyExc_MallocFailed, CapyExc_UndefinedExecution.
CapyColorChart* CapyColorChartAlloc(
  CapyColorChartType const type,
      CapyColorSpace const colorSpace) {
  CapyColorChart* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyColorChartCreate(type, colorSpace);
  return that;
}

// Clone a CapyColorChart
// Input:
//   chart: the color chart to clone
// Output:
//   Return a CapyColorChart
// Exception:
//   May raise CapyExc_MallocFailed, CapyExc_UndefinedExecution.
CapyColorChart* CapyColorChartClone(CapyColorChart const* const chart) {
  CapyColorChart* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = *chart;
  safeMalloc(that->colors, that->nbCol * that->nbRow);
  if(that->colors) {
    memcpy(
      that->colors,
      chart->colors,
      that->nbCol * that->nbRow * sizeof(CapyColorData));
  }
  safeMalloc(that->isInsideSRGBGamut, that->nbCol * that->nbRow);
  if(that->isInsideSRGBGamut) {
    memcpy(
      that->isInsideSRGBGamut,
      chart->isInsideSRGBGamut,
      that->nbCol * that->nbRow * sizeof(bool));
  }
  return that;
}

// Free the memory used by a CapyColorChart* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyColorChart to free
void CapyColorChartFree(CapyColorChart** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
