// ------------------- colorChart.h --------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_COLORCHART_H
#define CAPY_COLORCHART_H
#include "externalHeaders.h"
#include "image.h"
#include "streamIo.h"
#include "imgKernel.h"

// Description:
// Class to manipulate color charts.

// Types of color chart
typedef enum CapyColorChartType {
  capyColorChart_QP203,
  capyColorChart_XRiteClassic,
} CapyColorChartType;

// Definition of ColorChartDim class
#define CapyColorChartDimDef union { \
  size_t dims[2];                    \
  struct __attribute__((packed)) {   \
    size_t nbRow;                    \
    size_t nbCol;                    \
  };                                 \
}

// ColorChartDim class
typedef CapyColorChartDimDef CapyColorChartDim;

// Structure to memorise the options of the color chart detection
typedef struct CapyColorChartDetectOpt {

  // Standard deviation and size of the Gauss kernel
  double stdDev;
  CapyImgDims_t gaussKernelSize;
  CapyPad(CapyImgDims_t, gaussKernelSize);

  // Use the gradient of colors in the scoring function
  bool flagScoreGradient;
  CapyPad(bool, flagScoreGradient);

  // Threshold for the aspect ratio of patches
  double thresholdAspectRatio;

  // Threshold for the coverage of patches
  double thresholdCoverage;

  // Threshold for the parallelism of patches
  double thresholdParallelism;

  // Threshold for the pairing of patches
  double thresholdPairing;

  // Number of step for score diffusion
  size_t nbStepDiffusion;

  // Thresholds for the hysteresis step in Canny edge detection
  double hysteresis[2];

  // Flag to memorise if we keep pairing after an island of patches has
  // been completed
  bool flagKeepPairing;
  CapyPad(bool, 2);

  // Flag to run k-means clustering prior to canny edge detection
  bool flagKmeansClustering;
  CapyPad(bool, 3);
} CapyColorChartDetectOpt;

// Default options for the color chart detection
extern CapyColorChartDetectOpt const capyColorChartDetectDefaultOpt;

// Predeclaration of ColorChart class
typedef struct CapyColorChart CapyColorChart;

// ColorChart class
typedef struct CapyColorChart {

  // Type of the chart
  CapyColorChartType type;
  CapyPad(CapyColorChartType, type);

  // Dimensions of the chart
  CapyColorChartDimDef;

  // Colors of the chart (ordered as [iRow * nbCol + iCol])
  CapyColorData* colors;

  // Array of flags to memorise which colors are inside the sRGB gamut.
  bool* isInsideSRGBGamut;

  // Kernel size in pixel to blur the image when extracting the color
  // values (default: 3)
  size_t kernelSize;

  // Color space of the chart
  CapyColorSpace colorSpace;

  // Illuminant to interpret the color chart values (default D65)
  CapyColorIlluminant illuminant;

  // Destructor
  void (*destruct)(void);

  // Get the color at a given position in the chart
  CapyColorData (*get)(
    size_t const iRow,
    size_t const iCol);

  // Locate a color chart in an image
  // Input:
  //   img: the image containing the chart
  //   opt: the detection options
  // Output:
  //   Return the location of the color chart, clockwise from the
  //   top-left corner.
  // Exceptions:
  //   May raise CapyExc_NoColorChartInImage if no color chart were located.
  CapyQuadrilateral (*locate)(
                    CapyImg const* const img,
    CapyColorChartDetectOpt const* const opt);

  // Update the color values of the chart with those extracted from the
  // color chart contained in an image
  // Input:
  //      img: the image containing the color chart
  //   coords: the four coordinates of the corners of the QP203 in the
  //           image, as returned by locate(). If NULL, an automatic
  //           localisation of the color chart in the image is performed
  //           (to be implemented).
  // Output:
  //   Return true if the colors could be extracted, else false.
  bool (*extract)(
        CapyImg const* const img,
    CapyQuadrilateral const* chartCoord);

  // Convert the color chart to another color space
  // Input:
  //   colorSpace: the target color space
  void (*convertToColorSpace)(CapyColorSpace const colorSpace);

  // Correct the color chart to match the brightness of a given color
  // chart
  // Input:
  //   refChart: the target color chart
  void (*matchBrightness)(CapyColorChart const* const refChart);

  // Save the color chart to a path.
  // Format: type, nbRow, nbCol, colors[0].RGBA[0], colors[0].RGBA[1],
  // colors[0].RGBA[2], colors[0].RGBA[3], colors[1].RGBA[0], ...
  // Input:
  //   path: the path
  // Exceptions:
  //   May raise CapyExc_StreamOpenError, CapyExc_StreamWriteError
  void (*saveToPath)(char const* const path);

  // Save the color chart to a stream (in binary mode).
  // Format: type, nbRow, nbCol, colors[0].RGBA[0], colors[0].RGBA[1],
  // colors[0].RGBA[2], colors[0].RGBA[3], colors[1].RGBA[0], ...
  // Input:
  //   stream: the stream
  // Exceptions:
  //   May raise CapyExc_StreamWriteError
  void (*saveToStream)(CapyStreamIo* const stream);

  // Load the color chart from a path.
  // Format: type, nbRow, nbCol, colors[0].RGBA[0], colors[0].RGBA[1],
  // colors[0].RGBA[2], colors[0].RGBA[3], colors[1].RGBA[0], ...
  // Input:
  //   path: the path
  // Exceptions:
  //   May raise CapyExc_StreamOpenError, CapyExc_StreamReadError
  void (*loadFromPath)(char const* const path);

  // Load the color chart from a stream (in binary mode).
  // Format: type, nbRow, nbCol, colors[0].RGBA[0], colors[0].RGBA[1],
  // colors[0].RGBA[2], colors[0].RGBA[3], colors[1].RGBA[0], ...
  // Input:
  //   stream: the stream
  // Exceptions:
  //   May raise CapyExc_StreamWriteError, CapyExc_MallocFailed
  void (*loadFromStream)(CapyStreamIo* const stream);

  // Get the degree of similarity of the color chart relative to
  // another chart.
  // Input:
  //   chart: the chart to compare to
  // Output:
  //   Return 1.0 if the two charts are identical. Return 0.0 if the
  //   two charts are of different types. Return 1.0 minus the average
  //   of Euclidean distances between two same patches in the charts,
  //   hence the lower the returned value (possibly negative) the more
  //   charts' color differ. If the two charts are in different color
  //   space, a copy of 'chart' is converted to the color space of
  //   'that' and used instead.
  double (*getSimilarity)(CapyColorChart const* const chart);

  // Get the position of the center of a patch in an image given the
  // position of the corners of the color chart.
  // Input:
  //   coords: the four coordinates of the corners of the QP203 in the
  //           image, as returned by locate().
  //   iCol, iRow: the coordinates of the patch in the chart
  // Output:
  //   Return the position in the image of the center of the patch
  CapyImgPos (*getPatchPos)(
    CapyQuadrilateral const* const chartCoord,
                      size_t const iCol,
                      size_t const iRow);
} CapyColorChart;

// Create a CapyColorChart
// Input:
//         type: the type of the chart
//   colorSpace: the color space of the chart
// Output:
//   Return a CapyColorChart
// Exception:
//   May raise CapyExc_MallocFailed, CapyExc_UndefinedExecution.
CapyColorChart CapyColorChartCreate(
  CapyColorChartType const type,
      CapyColorSpace const colorSpace);

// Allocate memory for a new CapyColorChart and create it
// Input:
//         type: the type of the chart
//   colorSpace: the color space of the chart
// Output:
//   Return a CapyColorChart
// Exception:
//   May raise CapyExc_MallocFailed, CapyExc_UndefinedExecution.
CapyColorChart* CapyColorChartAlloc(
  CapyColorChartType const type,
      CapyColorSpace const colorSpace);

// Clone a CapyColorChart
// Input:
//   chart: the color chart to clone
// Output:
//   Return a CapyColorChart
// Exception:
//   May raise CapyExc_MallocFailed, CapyExc_UndefinedExecution.
CapyColorChart* CapyColorChartClone(CapyColorChart const* const chart);

// Free the memory used by a CapyColorChart* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyColorChart to free
void CapyColorChartFree(CapyColorChart** const that);
#endif
