// ------------------- colorCorrection.h --------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_COLORCORR_H
#define CAPY_COLORCORR_H
#include "externalHeaders.h"
#include "cext.h"
#include "colorChart.h"

// Description:
// Virtual parent class for color correction classes.

// CapyColorCorr object definition macro.
// Initial and final fitness to evaluate the performance of the correction.
// Updated by the match() method.
// double initialFitness, finalFitness;
// The color space in which the color correction is operating
// (default: sRGB)
// CapyColorSpace colorSpace;
// Find the correction to convert from a given color chart to
// a reference color chart
// Input:
//      chart: the original color chart
//   refChart: the reference color chart
// Output:
//   Update the initialFitness and finalFitness properties.
// void (*match)(
//   CapyColorChart const* const chart,
//   CapyColorChart const* const refChart);
// Apply the color correction to an image.
// Input:
//   img: the image to be corrected
// Output:
//   The image is corrected.
// void (*apply)(CapyImg* const img);
// Save the corrector to a path.
// Input:
//   path: the path
// Exceptions:
//   May raise CapyExc_StreamOpenError, CapyExc_StreamWriteError
// void (*saveToPath)(char const* const path);
// Save the corrector to a stream (in binary mode).
// Input:
//   stream: the stream
// Exceptions:
//   May raise CapyExc_StreamWriteError
// void (*saveToStream)(CapyStreamIo* const stream);
// Load the corrector from a path.
// Input:
//   path: the path
// Exceptions:
//   May raise CapyExc_StreamOpenError, CapyExc_StreamReadError
// void (*loadFromPath)(char const* const path);
// Load the corrector from a stream (in binary mode).
// Input:
//   stream: the stream
// Exceptions:
//   May raise CapyExc_StreamWriteError, CapyExc_MallocFailed
//void (*loadFromStream)(CapyStreamIo* const stream);
#define CapyColorCorrDef struct {                      \
  double initialFitness, finalFitness;                 \
  CapyColorSpace colorSpace;                           \
  CapyPad(CapyColorSpace, 0);                          \
  void (*destruct)(void);                              \
  void (*match)(                                       \
    CapyColorChart const* const chart,                 \
    CapyColorChart const* const refChart);             \
  void (*apply)(CapyImg* const img);                   \
  void (*saveToPath)(char const* const path);          \
  void (*saveToStream)(CapyStreamIo* const stream);    \
  void (*loadFromPath)(char const* const path);        \
  void (*loadFromStream)(CapyStreamIo* const stream);  \
}

// CapyColorCorr object
typedef CapyColorCorrDef CapyColorCorr;

// Create a CapyColorCorr
// Output:
//   Return a CapyColorCorr
CapyColorCorr CapyColorCorrCreate(void);
#endif
