// ------------------- colorCorrectionMatrix.c --------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "colorCorrectionMatrix.h"
#include "diffevo.h"

// Apply the color correction matrix to an image.
// Input:
//   img: the image to be corrected
static void Apply(CapyImg* const img) {
  methodOf(CapyColorCorrMat);

  // Memorise the original color space of the image
  CapyColorSpace origColorSpace = img->colorSpace;

  // Convert the image to the operationg color space of the color
  // correction matrix
  $(img, convertToColorSpace)(that->colorSpace);

  // Loop on the pixels
  size_t nbPixels = $(img, getNbPixels)();
  loop(iPixel, nbPixels) {

    // Calculate the corrected color, it is the product of the
    // correction matrix and the pixel color
    CapyColorData color = capyColorRGBABlack;
    loop(i, 3) loop(j, 3) {
      color.vals[i] += that->mat[i*3+j] * img->pixels[iPixel].vals[j];
    }

    // Set the corrected color in the image
    img->pixels[iPixel] = color;
  }

  // Convert the image back to its original color space
  $(img, convertToColorSpace)(origColorSpace);
}

// Fitness class for the optimiser
typedef struct CapyColorFitness {
  struct CapyMathFunDef;
  CapyColorChart const* const chart;
  CapyColorChart const* const refChart;
  void (*destructCapyMathFun)(void);
} CapyColorFitness;

// Default fitness function for the optimiser
static void FitnessFun(
  double const* const mat,
        double* const val) {
  CapyColorFitness* that = (CapyColorFitness*)capyThat;

  // Reset the fitness value
  *val = 0.0;

  // Loop on the colors of the chart
  loop(iRow, that->chart->nbRow) loop(iCol, that->chart->nbCol) {

    // Variable to memorise the index in the color chart
    size_t idx = iRow * that->refChart->nbCol + iCol;

    // Calculate the converted color
    CapyColorData convColor = capyColorRGBABlack;
    loop(i, 3) loop(j, 3) {
      convColor.vals[i] +=
        (mat[i * 3 + j] - 0.5) * 100.0 * that->chart->colors[idx].vals[j];
    }

    // Update the fitness value with the squared difference to the
    // reference
    loop(i, 3) {
      *val -=
        pow(convColor.vals[i] - that->refChart->colors[idx].vals[i], 2);
    }
  }
}

// Fitness function for QP203 for the optimiser (take into account that
// some of the color patches are out of the sRGB gamut, hence ignore
// them)
static void FitnessFunQP203(
  double const* const mat,
        double* const val) {
  CapyColorFitness* that = (CapyColorFitness*)capyThat;

  // Reset the fitness value
  *val = 0.0;

  // Loop on the colors of the chart
  loop(iRow, that->chart->nbRow) loop(iCol, that->chart->nbCol) {

    // Variable to memorise the index in the color chart
    size_t idx = iRow * that->refChart->nbCol + iCol;
    if(
      idx == 0 || idx == 3 || idx == 4 ||
      idx == 5 || idx == 9 ||
      idx == 10 || idx == 11 || idx == 12 || idx == 13 || idx == 14 ||
      idx == 15 || idx == 17 || idx == 18 || idx == 19 ||
      idx == 23 || idx == 24 ||
      idx == 25 || idx == 26 || idx == 27 || idx == 28 || idx == 29 ||
      idx == 30 || idx == 31 || idx == 32 || idx == 33 || idx == 34
    ) {

      // Calculate the converted color
      CapyColorData convColor = capyColorRGBABlack;
      loop(i, 3) loop(j, 3) {
        convColor.vals[i] +=
          (mat[i * 3 + j] - 0.5) * 100.0 * that->chart->colors[idx].vals[j];
      }

      // Update the fitness value with the squared difference to the
      // reference
      loop(i, 3) {
        *val -=
          pow(convColor.vals[i] - that->refChart->colors[idx].vals[i], 2);
      }
    }
  }
}

// Destruct an instance of the Fitness object
static void CapyColorFitnessDestruct(void) {
  methodOf(CapyColorFitness);
  $(that, destructCapyMathFun)();
}

// Create an instance of the Fitness object
static CapyColorFitness CapyColorFitnessCreate(
  CapyColorChart const* const chart,
  CapyColorChart const* const refChart) {
  CapyColorFitness that = { .chart = chart, .refChart = refChart };
  CapyInherits(that, CapyMathFun, (9, 1));
  that.eval = FitnessFun;
  that.destruct = CapyColorFitnessDestruct;
  return that;
}

// Find the correction matrix to convert from a given color chart to
// a reference color chart
// Input:
//      chart: the original color chart
//   refChart: the reference color chart
// Output:
//   Update the initialFitness and finalFitness properties.
static void Match(
  CapyColorChart const* const chart,
  CapyColorChart const* const refChart) {
  methodOf(CapyColorCorrMat);

  // Reset the correction matrix
  double identity[9] = {1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 1.0};
  loop(i, 9) that->mat[i] = identity[i];

  // Create copy of the charts in the operating color space of the
  // color correction matrix
  CapyColorChart* safeChart = CapyColorChartClone(chart);
  CapyColorChart* safeRefChart = CapyColorChartClone(refChart);
  $(safeChart, convertToColorSpace)(that->colorSpace);
  $(safeRefChart, convertToColorSpace)(that->colorSpace);

  // Create an optimiser to calculate the correction matrix. Set the
  // optimiser's seed for reproducibility.
  CapyDiffEvo optimiser = CapyDiffEvoCreate();
  optimiser.nbAgent = 100;
  double const seed[9] = {0.51, 0.5, 0.5, 0.5, 0.51, 0.5, 0.5, 0.5, 0.51};
  optimiser.seedDna = seed;
  optimiser.seed = 1;
  optimiser.nbIterReset = 0; //10;

  // Create a fitness function for the optimiser
  CapyColorFitness fitness =
    CapyColorFitnessCreate(safeChart, safeRefChart);

  // Set the fitness function according to the color charts type
  if(safeChart->type == capyColorChart_QP203) fitness.eval = FitnessFunQP203;
  else fitness.eval = FitnessFun;

  // Calculate the optimum correction matrix
  $(&optimiser, run)((CapyMathFun*)&fitness, NULL);

  // Get the intial fitness
  double initialFitness = optimiser.initialFitness;

  // Variable to memorise the best fitness and constraint
  double fitnessVal = 0.0;
  double constraintVal = 0.0;

  // Wait for the optimisation to end
  while(optimiser.running) {

    // Check for the termination of the optimisation process
    $(&optimiser, getBestDna)(that->mat, &fitnessVal, &constraintVal);
    size_t nbIter = optimiser.nbIter;
    if(
      fitnessVal >= that->thresholdFitness ||
      nbIter >= that->thresholdIter
    ) {
      $(&optimiser, stop)();
    }
    CapySleepMs(1);
  }
  loop(i, 9) that->mat[i] = (that->mat[i] - 0.5) * 100.0;

  // Free memory
  $(&fitness, destruct)();
  $(&optimiser, destruct)();
  CapyColorChartFree(&safeChart);
  CapyColorChartFree(&safeRefChart);

  // Set the result fitness
  that->initialFitness = initialFitness;
  that->finalFitness = fitnessVal;
}

// Save the correction matrix to a path.
// Input:
//   path: the path
// Exceptions:
//   May raise CapyExc_StreamOpenError, CapyExc_StreamWriteError
static void SaveToPath(char const* const path) {
  methodOf(CapyColorCorrMat);

  // Open the stream
  CapyStreamIo stream = CapyStreamIoCreate();
  $(&stream, open)(path, "wb");

  // Save to the stream
  $(that, saveToStream)(&stream);

  // Close the stream
  $(&stream, close)();
}

// Save the correction matrix to a stream (in binary mode).
// Input:
//   stream: the stream
// Exceptions:
//   May raise CapyExc_StreamWriteError
static void SaveToStream(CapyStreamIo* const stream) {
  methodOf(CapyColorCorrMat);

  // Write the data
  $(stream, writeBytes)(&(that->colorSpace), sizeof(CapyColorSpace));
  $(stream, writeBytes)(&(that->mat), sizeof(double) * 9);
}

// Load the correction matrix from a path.
// Input:
//   path: the path
// Exceptions:
//   May raise CapyExc_StreamOpenError, CapyExc_StreamReadError, 
//   CapyExc_MallocFailed.
static void LoadFromPath(char const* const path) {
  methodOf(CapyColorCorrMat);

  // Open the stream
  CapyStreamIo stream = CapyStreamIoCreate();
  $(&stream, open)(path, "rb");

  // Load from the stream
  $(that, loadFromStream)(&stream);

  // Close the stream
  $(&stream, close)();
}

// Load the correction matrix from a stream (in binary mode).
// Input:
//   stream: the stream
// Exceptions:
//   May raise CapyExc_StreamReadError, CapyExc_MallocFailed.
static void LoadFromStream(CapyStreamIo* const stream) {
  methodOf(CapyColorCorrMat);

  // Write the data
  $(stream, readBytes)(&(that->colorSpace), sizeof(CapyColorSpace));
  $(stream, readBytes)(&(that->mat), sizeof(double) * 9);
}

// Free the memory used by a CapyColorCorrMat
static void Destruct(void) {
  methodOf(CapyColorCorrMat);
  $(that, destructCapyColorCorr)();
}

// Create a CapyColorCorrMat
// Output:
//   Return a CapyColorCorrMat
CapyColorCorrMat CapyColorCorrMatCreate(void) {
  CapyColorCorrMat that = {
    .mat = {1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 1.0},
    .thresholdFitness = -0.001,
    .thresholdIter = 1000,
  };
  CapyInherits(that, CapyColorCorr, ());
  that.destruct = Destruct;
  that.match = Match;
  that.apply = Apply;
  that.saveToPath = SaveToPath;
  that.saveToStream = SaveToStream;
  that.loadFromPath = LoadFromPath;
  that.loadFromStream = LoadFromStream;
  return that;
}

// Allocate memory for a new CapyColorCorrMat and create it
// Output:
//   Return a CapyColorCorrMat
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColorCorrMat* CapyColorCorrMatAlloc(void) {
  CapyColorCorrMat* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyColorCorrMatCreate();
  return that;
}

// Free the memory used by a CapyColorCorrMat* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyColorCorrMat to free
void CapyColorCorrMatFree(CapyColorCorrMat** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
