// ------------------------------ colorHisto.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "colorHisto.h"

// Extract the color histograms from an image
// Input:
//   img: the image
static void Extract(CapyImg const* const img) {
  methodOf(CapyColorHisto);

  // Reset the values
  loop(i, capyColorHistoType_last) loop(j, CAPY_COLORHISTO_NBBUCKETS) {
    that->vals[i][j] = 0.0;
    that->cumulVals[i][j] = 0.0;
  }

  // Coefficient to calculate the index of intensity
  double coeffIdx = (double)(CAPY_COLORHISTO_NBBUCKETS - 1) / sqrt(3.0);

  // Loop on the pixels of the image
  size_t nbPixels = $(img, getNbPixels)();
  loop(iPixel, nbPixels) {

    // Update the red, green, blue values and calculate the intensity
    double intensity = 0.0;
    loop(i, 3) {
      intensity += pow(img->pixels[iPixel].RGB[i], 2.0);
      size_t idx =
        (size_t)lround(
          img->pixels[iPixel].RGB[i] *
          (double)(CAPY_COLORHISTO_NBBUCKETS - 1));
      that->vals[i][idx] += 1.0;
    }

    // Update the intensity values
    size_t idx = (size_t)lround(sqrt(intensity) * coeffIdx);
    that->vals[capyColorHistoType_intensity][idx] += 1.0;
  }

  // Update the cumulative values
  loop(i, capyColorHistoType_last) {
    that->cumulVals[i][0] = that->vals[i][0];
    for(size_t j = 1; j < CAPY_COLORHISTO_NBBUCKETS; ++j) {
      that->cumulVals[i][j] =
        that->cumulVals[i][j - 1] + that->vals[i][j];
    }
  }

  // Normalise the values
  double invNbPixels = 1.0 / (double)nbPixels;
  loop(i, capyColorHistoType_last) loop(j, CAPY_COLORHISTO_NBBUCKETS) {
    that->vals[i][j] *= invNbPixels;
    that->cumulVals[i][j] *= invNbPixels;
  }
}

// Correct an image to match the histogram for a given type
// Input:
//    img: the image
//   type: the type of histogram to be matched
static void Apply(
            CapyImg* const img,
  CapyColorHistoType const type) {
  methodOf(CapyColorHisto);

  // Get teh histogram of the image
  CapyColorHisto imgHisto = CapyColorHistoCreate();
  $(&imgHisto, extract)(img);

  // Look up table for the correction
  size_t lut[CAPY_COLORHISTO_NBBUCKETS];

  // Create the look up table
  size_t iRef = 0;
  size_t iImg = 0;
  while(iImg < CAPY_COLORHISTO_NBBUCKETS) {
    while(imgHisto.cumulVals[type][iImg] < that->cumulVals[type][iRef]) {
      lut[iImg] = iRef;
      ++iImg;
    }
    while(imgHisto.cumulVals[type][iImg] > that->cumulVals[type][iRef]) ++iRef;
    lut[iImg] = iRef;
    ++iImg;
  }

  // Loop on the pixels
  size_t nbPixels = $(img, getNbPixels)();
  loop(iPixel, nbPixels) {

    // Convert the color of the pixel
    CapyColorData* pixel = img->pixels + iPixel;
    if(
      type == capyColorHistoType_red ||
      type == capyColorHistoType_green ||
      type == capyColorHistoType_blue
    ) {
      size_t idx =
        (size_t)lround(
          pixel->RGB[type] * (double)(CAPY_COLORHISTO_NBBUCKETS - 1));
      size_t idxCorrected = lut[idx];
      pixel->RGB[type] =
        ((double)idxCorrected) / (double)(CAPY_COLORHISTO_NBBUCKETS - 1);
    } else if(type == capyColorHistoType_intensity) {
      double intensity = 0.0;
      loop(i, 3) intensity += pow(pixel->RGB[i], 2.0);
      intensity = sqrt(intensity) / sqrt(3.0);
      size_t idx =
        (size_t)lround(
          intensity * (double)(CAPY_COLORHISTO_NBBUCKETS - 1));
      size_t idxCorrected = lut[idx];
      double intensityCorrected =
        ((double)idxCorrected) / (double)(CAPY_COLORHISTO_NBBUCKETS - 1);
      double corrCoeff = intensityCorrected / intensity;
      loop(i, 3) {
        pixel->RGB[i] *= corrCoeff;
        if(pixel->RGB[i] > 1.0) pixel->RGB[i] = 1.0;
      }
    } else raiseExc(CapyExc_UndefinedExecution);
  }

  // Free memory
  $(&imgHisto, destruct)();
}

// Free the memory used by a CapyColorHisto
static void Destruct(void) {
  return;
}

// Create a CapyColorHisto
// Output:
//   Return a CapyColorHisto
CapyColorHisto CapyColorHistoCreate(void) {
  return (CapyColorHisto){
    .destruct = Destruct,
    .extract = Extract,
    .apply = Apply,
  };
}

// Allocate memory for a new CapyColorHisto and create it
// Output:
//   Return a CapyColorHisto
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColorHisto* CapyColorHistoAlloc(void) {
  CapyColorHisto* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyColorHistoCreate();
  return that;
}

// Create a CapyColorHisto with uniform distribution
// Output:
//   Return a CapyColorHisto
CapyColorHisto CapyColorHistoCreateUniform(void) {
  CapyColorHisto histo = {
    .extract = Extract,
    .apply = Apply,
  };
  double density = 1.0 / (double)CAPY_COLORHISTO_NBBUCKETS;
  loop(i, capyColorHistoType_last) loop(j, CAPY_COLORHISTO_NBBUCKETS) {
    histo.vals[i][j] = density;
    histo.cumulVals[i][j] =
      ((double)j) / (double)(CAPY_COLORHISTO_NBBUCKETS - 1);
  }
  return histo;
}

// Allocate memory for a new CapyColorHisto wit uniform distribution and
// create it
// Output:
//   Return a CapyColorHisto
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColorHisto* CapyColorHistoAllocUniform(void) {
  CapyColorHisto* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyColorHistoCreateUniform();
  return that;
}

// Free the memory used by a CapyColorHisto* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyColorHisto to free
void CapyColorHistoFree(CapyColorHisto** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
