// ------------------------------- colorHisto.h ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_COLORHISTO_H
#define CAPY_COLORHISTO_H
#include "externalHeaders.h"
#include "image.h"

// Description:
// Class to manipulate image's color histogram.

// Number of buckets in the histogram
#define CAPY_COLORHISTO_NBBUCKETS 256

// Histogram types
typedef enum CapyColorHistoType {
  capyColorHistoType_red = 0,
  capyColorHistoType_green = 1,
  capyColorHistoType_blue = 2,
  capyColorHistoType_intensity,
  capyColorHistoType_last
} CapyColorHistoType;

// ColorHisto object
typedef struct CapyColorHisto {

  // Histogram values
  double vals[capyColorHistoType_last][CAPY_COLORHISTO_NBBUCKETS];

  // Histogram cumulative values
  double cumulVals[capyColorHistoType_last][CAPY_COLORHISTO_NBBUCKETS];

  // Destructor
  void (*destruct)(void);

  // Extract the color histograms from an image
  // Input:
  //   img: the image
  void (*extract)(CapyImg const* const img);

  // Correct an image to match the histogram for a given type
  // Input:
  //    img: the image
  //   type: the type of histogram to be matched
  void (*apply)(
              CapyImg* const img,
    CapyColorHistoType const type);
} CapyColorHisto;

// Create a CapyColorHisto
// Output:
//   Return a CapyColorHisto
CapyColorHisto CapyColorHistoCreate(void);

// Allocate memory for a new CapyColorHisto and create it
// Output:
//   Return a CapyColorHisto
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColorHisto* CapyColorHistoAlloc(void);

// Create a CapyColorHisto with uniform distribution
// Output:
//   Return a CapyColorHisto
CapyColorHisto CapyColorHistoCreateUniform(void);

// Allocate memory for a new CapyColorHisto wit uniform distribution and
// create it
// Output:
//   Return a CapyColorHisto
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColorHisto* CapyColorHistoAllocUniform(void);

// Free the memory used by a CapyColorHisto* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyColorHisto to free
void CapyColorHistoFree(CapyColorHisto** const that);
#endif
