// ---------------------------- colorPatch.c ----------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "colorPatch.h"

// Set of color patches
CapyDefList(CapyListColorPatch, CapyColorPatch)

// Update the center of mass of the color patch.
static void UpdatePosCenterOfMass(void) {
  methodOf(CapyColorPatch);

  // Reset the center of mass
  that->centerOfMass = (CapyImgPos){{0, 0}};

  // Calculate the center of mass has the average coordinate of the pixels in
  // the patch
  double avgPos[2] = {0.0, 0.0};
  $(&(that->pixels), initIterator)();
  forEach(pixel, that->pixels.iter) {
    loop(i, 2) avgPos[i] += (double)pixel.pos.coords[i];
  }
  loop(i, 2) avgPos[i] /= (double)$(&(that->pixels), getSize)();
  loop(i, 2) that->centerOfMass.coords[i] = (CapyImgPos_t)avgPos[i];
}

// Update the rasterised quadrilateral approximating the color patch.
// The corners of the resulting quadrilateral are guaranteed to be in
// clockwise order. The first corner is guaranted to be the one nearest
// to the origin.
static void UpdateApproxQuadrilateral(void) {
  methodOf(CapyColorPatch);

  // Reset the quadrilateral
  $(&(that->approxQuad), destruct)();
  that->approxQuad = CapyQuadrilateralCreate();

  // Upadte the center of mass of the color patch
  $(that, updatePosCenterOfMass)();

  // Get the first corner as the farthest pixel from the center of mass
  double maxDist = -10.0;
  $(&(that->pixels), initIterator)();
  forEach(pixel, that->pixels.iter) {
    double dist = 0.0;
    loop(i, 2) {
      dist += pow(
        (double)that->centerOfMass.coords[i] - (double)pixel.pos.coords[i],
        2.0);
    }
    if(dist > maxDist) {
      loop(i, 2) that->approxQuad.corners[0].coords[i] = pixel.pos.coords[i];
      maxDist = dist;
    }
  }

  // Variable to memorise the transformation matrix from image coordinates
  // to quadrilateral coordinates
  double b1 =
    that->approxQuad.corners[0].coords[0] -
    (double)that->centerOfMass.coords[0];
  double b2 =
    that->approxQuad.corners[0].coords[1] -
    (double)that->centerOfMass.coords[1];
  double mat[4] = { b1, b2, -b2, b1 };

  // Variable to commonalise code for the three other corners
  size_t axisCorner[3] = {1, 0, 1};
  double signCorner[3] = {1.0, -1.0, -1.0};

  // Loop on the three other corners
  loop(iCorner, 3) {

    // Reset the variable to memorise the farthest pixel for this corner
    maxDist = -10.0;

    // Loop on the pixels
    forEach(pixel, that->pixels.iter) {

      // Set the corner as the farthest pixel in quadrilateral coordinates
      // in the direction of that corner
      double dist = 0.0;
      loop(i, (size_t)2) {
        dist +=
          signCorner[iCorner] *
          mat[2 * axisCorner[iCorner] + i] *
          ((double)pixel.pos.coords[i] - (double)that->centerOfMass.coords[i]);
      }
      if(dist > maxDist) {
        loop(i, 2) {
          that->approxQuad.corners[1 + iCorner].coords[i] = pixel.pos.coords[i];
        }
        maxDist = dist;
      }
    }
  }

  // Get the index of the corner nearest to the origin
  int iNearest = 0;
  double distNearest =
    pow(that->approxQuad.corners[0].coords[0], 2.0) +
    pow(that->approxQuad.corners[0].coords[1], 2.0);
  loop(iCorner, 3) {
    double dist =
      pow(that->approxQuad.corners[1 + iCorner].coords[0], 2.0) +
      pow(that->approxQuad.corners[1 + iCorner].coords[1], 2.0);
    if(distNearest > dist) {
      distNearest = dist;
      iNearest = 1 + iCorner;
    }
  }

  // Rotate the corners so that the first one is the nearest to the origin
  double tmp[8] = {0, 0, 0, 0, 0, 0, 0, 0};
  loop(iCorner, 4) loop(iCoord, 2) {
    tmp[iCorner * 2 + iCoord] =
      that->approxQuad.corners[(iCorner + iNearest) % 4].coords[iCoord];
  }
  loop(iCorner, 4) loop(iCoord, 2) {
    that->approxQuad.corners[iCorner].coords[iCoord] =
      tmp[iCorner * 2 + iCoord];
  }
}

// Update the average color of the color patch
static void UpdateAvgColor(void) {
  methodOf(CapyColorPatch);

  // Reset the average color
  that->avgColor = capyColorRGBABlack;

  // Loop on the pixels in the patch and calculate the average color
  $(&(that->pixels), initIterator)();
  forEach(pixel, that->pixels.iter) loop(k, 3) {
    that->avgColor.vals[k] += pixel.color->vals[k];
  }
  loop(k, 3) that->avgColor.vals[k] /= (double)$(&(that->pixels), getSize)();
}

// Free the memory used by a CapyColorPatch
static void Destruct(void) {
  methodOf(CapyColorPatch);
  $(&(that->pixels), destruct)();
}

// Create a CapyColorPatch of given dimensions and mode
// Output:
//   Return a CapyColorPatch
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColorPatch CapyColorPatchCreate(void) {
  CapyColorPatch that = {
    .idx = 0,
    .pixels = CapyImgPixelsCreate(0),
    .avgColor = capyColorRGBABlack,
    .centerOfMass = {{0, 0}},
    .destruct = Destruct,
    .approxQuad = CapyQuadrilateralCreate(),
    .updateApproxQuadrilateral = UpdateApproxQuadrilateral,
    .updatePosCenterOfMass = UpdatePosCenterOfMass,
    .updateAvgColor = UpdateAvgColor,
  };
  $(&(that.pixels), initIterator)();
  return that;
}

// Allocate memory for a new CapyColorPatch and create it
// Output:
//   Return a CapyColorPatch
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColorPatch* CapyColorPatchAlloc(void) {

  // Allocate memory for the instance
  CapyColorPatch* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;

  // Create the instance
  *that = CapyColorPatchCreate();

  // Return the instance
  return that;
}

// Free the memory used by a CapyColorPatch* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyColorPatch to free
void CapyColorPatchFree(CapyColorPatch** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Get the color patch at a given index
// Input:
//   idx: the index
// Output:
//   Return the CapyColorPatch
static CapyColorPatch GetColorPatch(size_t const idx) {
  methodOf(CapyColorPatches);
  return $(that->list, get)(idx);
}

// Get the number of patches
// Output:
//   Return the number of patches
static size_t GetNbPatch(void){
  methodOf(CapyColorPatches);
  return $(that->list, getSize)();
}

// Add a CapyColorPatch to the patches
// Input:
//   patch: the patch to add
static void AddPatch(CapyColorPatch* const patch) {
  methodOf(CapyColorPatches);
  $(that->list, add)(*patch);
}

// Free the memory used by a CapyColorPatches
static void ColorPatchesDestruct(void) {
  methodOf(CapyColorPatches);
  while($(that->list, getSize)() > 0) {
    CapyColorPatch patch = $(that->list, pop)();
    $(&patch, destruct)();
  }
  CapyListColorPatchFree(&(that->list));
}

// Create a CapyColorPatches of given dimensions and mode
// Output:
//   Return a CapyColorPatches
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColorPatches CapyColorPatchesCreate(void) {

  // Create the CapyColorPatches and return it
  CapyColorPatches that = {
    .list = CapyListColorPatchAlloc(),
    .get = GetColorPatch,
    .getNbPatch = GetNbPatch,
    .add = AddPatch,
    .destruct = ColorPatchesDestruct,
  };
  $(that.list, initIterator)();
  return that;
}

// Allocate memory for a new CapyColorPatches and create it
// Output:
//   Return a CapyColorPatches
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColorPatches* CapyColorPatchesAlloc(void) {

  // Allocate memory for the instance
  CapyColorPatches* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;

  // Create the instance
  *that = CapyColorPatchesCreate();
  $(that->list, initIterator)();

  // Return the instance
  return that;
}

// Free the memory used by a CapyColorPatches* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyColorPatches to free
void CapyColorPatchesFree(CapyColorPatches** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
