// -------------------------------- colorPatch.h -------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_COLOR_PATCH_H
#define CAPY_COLOR_PATCH_H
#include "externalHeaders.h"
#include "image.h"

// Description:
// Class to manipulate patches of color.

// Color patch
typedef struct CapyColorPatch {

  // Set of pixels in the patch
  CapyImgPixels pixels;

  // Index of the patch (default: 0)
  size_t idx;

  // Average color of the patch, to be updated by the user with
  // getAvgColor
  CapyColorData avgColor;

  // Center of mass of the patch, to be updated by the user with
  // getPosCenterOfMass
  CapyImgPos centerOfMass;

  // Approximating quadrilateral
  CapyQuadrilateral approxQuad;

  // Destructor
  void (*destruct)(void);

  // Update the rasterised quadrilateral approximating the color patch.
  // The corners of the resulting quadrilateral are guaranteed to be in
  // clockwise order. The first corner is guaranted to be the one nearest
  // to the origin.
  void (*updateApproxQuadrilateral)(void);

  // Update the center of mass of the color patch.
  void (*updatePosCenterOfMass)(void);

  // Update the average color of the color patch
  void (*updateAvgColor)(void);
} CapyColorPatch;

// Create a CapyColorPatch of given dimensions and mode
// Output:
//   Return a CapyColorPatch
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColorPatch CapyColorPatchCreate(void);

// Allocate memory for a new CapyColorPatch and create it
// Output:
//   Return a CapyColorPatch
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColorPatch* CapyColorPatchAlloc(void);

// Free the memory used by a CapyColorPatch* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyColorPatch to free
void CapyColorPatchFree(CapyColorPatch** const that);

// Set of color patches
CapyDecList(CapyListColorPatch, CapyColorPatch)

// CapyColorPatches object
typedef struct CapyColorPatches {

  // The list of patches
  CapyListColorPatch* list;

  // Destructor
  void (*destruct)(void);

  // Get the color patch at a given index
  // Input:
  //   idx: the index
  // Output:
  //   Return the CapyColorPatch
  CapyColorPatch (*get)(size_t const idx);

  // Get the number of patches
  // Output:
  //   Return the number of patches
  size_t (*getNbPatch)(void);

  // Add a CapyColorPatch to the patches
  // Input:
  //   patch: the patch to add
  void (*add)(CapyColorPatch* const patch);
} CapyColorPatches;

// Create a CapyColorPatches of given dimensions and mode
// Output:
//   Return a CapyColorPatches
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColorPatches CapyColorPatchesCreate(void);

// Allocate memory for a new CapyColorPatches and create it
// Output:
//   Return a CapyColorPatches
// Exception:
//   May raise CapyExc_MallocFailed.
CapyColorPatches* CapyColorPatchesAlloc(void);

// Free the memory used by a CapyColorPatches* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyColorPatches to free
void CapyColorPatchesFree(CapyColorPatches** const that);
#endif
