// ---------------------------- compressor.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "compressor.h"
#include "burrowswheelertransform.h"
#include "list.h"
#include "tree.h"
#include "comparator.h"
#include "bitarray.h"

// Compress the data
// Input:
//   data: the data to compress
// Output:
//   Return the compressed data
static CapyCompressorData Compress(CapyCompressorData const data) {
  (void)data;
  raiseExc(CapyExc_UndefinedExecution);
  assert(false && "Compresor.compress is undefined.");
  return (CapyCompressorData){0};
}

// Decompress the data
// Input:
//   data: the data to decompress
// Output:
//   Return the decompressed data
static CapyCompressorData Decompress(CapyCompressorData const data) {
  (void)data;
  raiseExc(CapyExc_UndefinedExecution);
  assert(false && "Compresor.decompress is undefined.");
  return (CapyCompressorData){0};
}

// Free the memory used by a CapyCompressor
static void Destruct(void) {
  return;
}

// Create a CapyCompressor
// Output:
//   Return a CapyCompressor
CapyCompressor CapyCompressorCreate(void) {
  CapyCompressor that = {
    .destruct = Destruct,
    .compress = Compress,
    .decompress = Decompress,
  };
  return that;
}

// Allocate memory for a new CapyCompressor and create it
// Output:
//   Return a CapyCompressor
// Exception:
//   May raise CapyExc_MallocFailed.
CapyCompressor* CapyCompressorAlloc(void) {
  CapyCompressor* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyCompressorCreate();
  return that;
}

// Free the memory used by a CapyCompressor* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyCompressor to free
void CapyCompressorFree(CapyCompressor** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Compress the data using RLE
// Input:
//   data: the data to compress
// Output:
//   Return the compressed data
static CapyCompressorData CompressRLE(CapyCompressorData const data) {
  CapyCompressorData res = {0};
  if(data.size == 0 || data.bytes == NULL) return res;
  safeMalloc(res.bytes, data.size);
  uint8_t* ptr = data.bytes;
  uint8_t* ptrRes = res.bytes;
  size_t idx = 0;
  size_t memSize = data.size;
  while(idx < data.size) {
    uint8_t len = 1;
    while(len < 254 && idx + len < data.size && ptr[0] == ptr[len]) ++len;
    if(res.size + 2 > memSize) {
      memSize += data.size;
      safeRealloc(res.bytes, memSize);
      ptrRes = res.bytes + res.size;
    }
    ptrRes[0] = len;
    ptrRes[1] = ptr[0];
    ptrRes += 2;
    ptr += len;
    res.size += 2;
    idx += len;
  }
  safeRealloc(res.bytes, res.size);
  return res;
}

// Decompress the data using RLE
// Input:
//   data: the data to decompress
// Output:
//   Return the decompressed data
static CapyCompressorData DecompressRLE(CapyCompressorData const data) {
  CapyCompressorData res = {0};
  if(data.size == 0 || data.bytes == NULL) return res;
  safeMalloc(res.bytes, data.size);
  uint8_t* ptr = data.bytes;
  uint8_t* ptrRes = res.bytes;
  size_t memSize = data.size;
  loop(i, data.size) {
    if(res.size + ptr[0] > memSize) {
      memSize += data.size;
      safeRealloc(res.bytes, memSize);
      ptrRes = res.bytes + res.size;
    }
    loop(j, ptr[0]) ptrRes[j] = ptr[1];
    ptrRes += ptr[0];
    res.size += ptr[0];
    ptr += 2;
    ++i;
  }
  safeRealloc(res.bytes, res.size);
  return res;
}

// Free the memory used by a CapyCompressor
static void DestructRLE(void) {
  methodOf(CapyRLECompressor);
  $(that, destructCapyCompressor)();
}

// Create a CapyRLECompressor
// Output:
//   Return a CapyRLECompressor
CapyRLECompressor CapyRLECompressorCreate(void) {
  CapyRLECompressor that;
  CapyInherits(that, CapyCompressor, ());
  that.compress = CompressRLE;
  that.decompress = DecompressRLE;
  that.destruct = DestructRLE;
  return that;
}

// Allocate memory for a new CapyRLECompressor and create it
// Output:
//   Return a CapyRLECompressor
// Exception:
//   May raise CapyExc_MallocFailed.
CapyRLECompressor* CapyRLECompressorAlloc(void) {
  CapyRLECompressor* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyRLECompressorCreate();
  return that;
}

// Free the memory used by a CapyRLECompressor* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyRLECompressor to free
void CapyRLECompressorFree(CapyRLECompressor** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Compress the data using BWTRLE
// Input:
//   data: the data to compress
// Output:
//   Return the compressed data
static CapyCompressorData CompressBWTRLE(CapyCompressorData const data) {
  CapyCompressorData res = {0};
  if(data.size == 0 || data.bytes == NULL) return res;
  CapyBurrowsWheelerTransform bwt = CapyBurrowsWheelerTransformCreate();
  CapyBurrowsWheelerTransformData bwtData =
    {.bytes = data.bytes, .size = data.size};
  CapyBurrowsWheelerTransformData bwtRes = $(&bwt, transform)(bwtData);
  CapyRLECompressor rle = CapyRLECompressorCreate();
  CapyCompressorData rleData = {.bytes = bwtRes.bytes, .size = bwtRes.size};
  CapyCompressorData rleRes = $(&rle, compress)(rleData);
  size_t sizeRes = rleRes.size + sizeof(size_t);
  safeMalloc(res.bytes, sizeRes);
  if(res.bytes == NULL) {
    free(rleRes.bytes);
    free(bwtRes.bytes);
    return res;
  }
  res.size = sizeRes;
  memcpy(res.bytes, &(bwtRes.idxFirstByte), sizeof(size_t));
  memcpy(res.bytes + sizeof(size_t), rleRes.bytes, rleRes.size);
  free(rleRes.bytes);
  free(bwtRes.bytes);
  $(&rle, destruct)();
  $(&bwt, destruct)();
  return res;
}

// Decompress the data using BWTRLE
// Input:
//   data: the data to decompress
// Output:
//   Return the decompressed data
static CapyCompressorData DecompressBWTRLE(CapyCompressorData const data) {
  CapyCompressorData res = {0};
  if(data.size == 0 || data.bytes == NULL) return res;
  CapyRLECompressor rle = CapyRLECompressorCreate();
  CapyCompressorData rleData =
    {.bytes = data.bytes + sizeof(size_t), .size = data.size - sizeof(size_t)};
  CapyCompressorData rleRes = $(&rle, decompress)(rleData);
  CapyBurrowsWheelerTransform bwt = CapyBurrowsWheelerTransformCreate();
  CapyBurrowsWheelerTransformData bwtData = {
    .bytes = rleRes.bytes, .size = rleRes.size,
    .idxFirstByte = *(size_t*)(data.bytes)
  };
  CapyBurrowsWheelerTransformData bwtRes = $(&bwt, untransform)(bwtData);
  safeMalloc(res.bytes, bwtRes.size);
  if(res.bytes == NULL) {
    free(bwtRes.bytes);
    free(rleRes.bytes);
    return res;
  }
  res.size = bwtRes.size;
  memcpy(res.bytes, bwtRes.bytes, bwtRes.size);
  free(bwtRes.bytes);
  free(rleRes.bytes);
  $(&rle, destruct)();
  $(&bwt, destruct)();
  return res;
}

// Free the memory used by a CapyCompressor
static void DestructBWTRLE(void) {
  methodOf(CapyBWTRLECompressor);
  $(that, destructCapyCompressor)();
}

// Create a CapyBWTRLECompressor
// Output:
//   Return a CapyBWTRLECompressor
CapyBWTRLECompressor CapyBWTRLECompressorCreate(void) {
  CapyBWTRLECompressor that;
  CapyInherits(that, CapyCompressor, ());
  that.compress = CompressBWTRLE;
  that.decompress = DecompressBWTRLE;
  that.destruct = DestructBWTRLE;
  return that;
}

// Allocate memory for a new CapyBWTRLECompressor and create it
// Output:
//   Return a CapyBWTRLECompressor
// Exception:
//   May raise CapyExc_MallocFailed.
CapyBWTRLECompressor* CapyBWTRLECompressorAlloc(void) {
  CapyBWTRLECompressor* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyBWTRLECompressorCreate();
  return that;
}

// Free the memory used by a CapyBWTRLECompressor* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyBWTRLECompressor to free
void CapyBWTRLECompressorFree(CapyBWTRLECompressor** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}

// Structure used during Huffman compression to create the encoding tree
typedef struct CapyHuffmanSymbol CapyHuffmanSymbol;
struct CapyHuffmanSymbol {
  uint8_t symbol;
  CapyPad(uint8_t, symbol);
  size_t nbOcc;
  CapyHuffmanSymbol* childs[2];
};

// Structure used during Huffman compression to create the encoding
typedef struct CapyHuffmanEncoding {
  bool bits[256];
  size_t nbBit;
} CapyHuffmanEncoding;

// Compare two CapyHuffmanSymbol.
// Input:
//   a: the first input
//   b: the second input
// Output:
//   Return an int, negative if a is 'before' b, 0 if a is 'same' as b,
//   positive if a is 'after' b.
static int CmpHuffmanSymbol(
  void const* a,
  void const* b) {
  CapyHuffmanSymbol const* const symbolA = *(CapyHuffmanSymbol**)a;
  CapyHuffmanSymbol const* const symbolB = *(CapyHuffmanSymbol**)b;
  if(symbolA->nbOcc < symbolB->nbOcc) return -1;
  else if(symbolA->nbOcc > symbolB->nbOcc) return 1;
  else return 0;
}

// List of CapyHuffmanSymbol
CapyDecList(CapyListHuffmanSymbol, CapyHuffmanSymbol*)
CapyDefList(CapyListHuffmanSymbol, CapyHuffmanSymbol*)

// Test for the Huffman compression loop 
static bool AreQueueEmpty(CapyListHuffmanSymbol const* const queues) {
  int n =
    (queues[0].head != NULL) +
    (queues[0].head != NULL && queues[0].head->next != NULL) +
    (queues[1].head != NULL) +
    (queues[1].head != NULL && queues[1].head->next != NULL);
  return (n <= 1);
}

// Create the encoding tree for Huffman compression
static CapyHuffmanSymbol* CreateHuffmanEncodingTree(
  CapyCompressorData const data) {

  // Create the table of number of occurence per symbol (one symbol = one byte)
  size_t nbOccSymbols[256];
  loop(i, 256) nbOccSymbols[i] = 0;
  loop(iByte, data.size) nbOccSymbols[data.bytes[iByte]] += 1;

  // Create the two queues of symbols
  CapyListHuffmanSymbol queues[2] = {
    CapyListHuffmanSymbolCreate(),
    CapyListHuffmanSymbolCreate(),
  };
  CapyListHuffmanSymbol* const initialQueue = queues;
  CapyListHuffmanSymbol* const combinedQueue = queues + 1;

  // Insert symbols in the first queue, sorted by increasing number of occurence
  // (least likely item at the head of the queue)
  CapyComparator cmpSymbol = { .eval = CmpHuffmanSymbol };
  loop(iSymbol, 256) if(nbOccSymbols[iSymbol] > 0) {
    CapyHuffmanSymbol* symbol = NULL;
    safeMalloc(symbol, 1);
    *symbol = (CapyHuffmanSymbol){
      .symbol = (uint8_t)iSymbol,
      .nbOcc = nbOccSymbols[iSymbol],
      .childs = {NULL, NULL},
    };
    $(initialQueue, insertSort)(&cmpSymbol, symbol);
  }

  // If the queues are not empty
  if(AreQueueEmpty(queues) == false) {

    // While there is more than one node in the queues
    do {

      // Get the two symbols with lower number of occurence
      CapyHuffmanSymbol* symbols[2] = {0};
      int iSymbol = 0;
      while(iSymbol < 2) loop(iQueue, 2) {
        CapyListHuffmanSymbol* const queueA = queues + iQueue;
        CapyListHuffmanSymbol* const queueB = queues + 1 - iQueue;
        while(
          iSymbol < 2 &&
          queueA->head != NULL &&
          (
            queueB->head == NULL ||
            queueB->head->data->nbOcc >= queueA->head->data->nbOcc
          )
        ) {
          symbols[iSymbol] = $(queueA, pop)();
          iSymbol += 1;
        }
      }

      // Create a new symbol with number of occurence equal to the sum of
      // numbers of occurence of the selected two symbols and these symbols
      // as childs
      CapyHuffmanSymbol* symbol = NULL;
      safeMalloc(symbol, 1);
      *symbol = (CapyHuffmanSymbol){
        .symbol = 0,
        .nbOcc = symbols[0]->nbOcc + symbols[1]->nbOcc,
        .childs = {symbols[0], symbols[1]},
      };

      // Add the new symbol at the end of the second queue
      $(combinedQueue, add)(symbol);
    } while(AreQueueEmpty(queues) == false);

  // Else, the queues are empty from the beginning. It means there is only one
  // symbol in the data. Compressing could still be done but doesn't make much
  // sense, so I raise an exception instead
  } else {
    raiseExc(CapyExc_InvalidParameters);
  }

  // The remaining symbol (necessarily in the second queue) is the encoding
  // tree
  CapyHuffmanSymbol* encodingTree = $(combinedQueue, pop)();

  // Free memory
  loop(iQueue, 2) {
    while(queues[iQueue].head != NULL) {
      CapyHuffmanSymbol* symbol = $(queues + iQueue, pop)();
      free(symbol);
    }
    $(queues + iQueue, destruct)();
  }

  // Return the encoding tree
  return encodingTree;
}

// Calculate the number of bits necessary to convert the encoding tree
static size_t GetSizeBitHuffmanEncodingTree(
  CapyHuffmanSymbol const* const encodingTree) {
  size_t nb = 0;
  if(encodingTree->childs[0] == NULL && encodingTree->childs[1] == NULL) {
    nb = 9;
  } else {
    loop(iChild, 2) {
      nb += 1 + GetSizeBitHuffmanEncodingTree(encodingTree->childs[iChild]);
    }
  }
  return nb;
}

static void ConvertHuffmanEncodingTreeToBitArray(
  CapyHuffmanSymbol const* const encodingTree,
             CapyBitArray* const bitArray,
                   size_t* const iBit) {
  if(encodingTree->childs[0] == NULL && encodingTree->childs[1] == NULL) {
    $(bitArray, setBit)(*iBit, true);
    *iBit += 1;
    loop(jBit, 8) {
      bool bit = ((encodingTree->symbol & (1 << (7 - jBit))) != 0);
      $(bitArray, setBit)(*iBit, bit);
      *iBit += 1;
    }
  } else {
    $(bitArray, setBit)(*iBit, false);
    *iBit += 1;
    loop(iChild, 2) {
      ConvertHuffmanEncodingTreeToBitArray(
        encodingTree->childs[iChild], bitArray, iBit);
    }
  }
}

// Create the Huffman encoding table
// Input:
//   encodingTree: the encoding tree
//   encodingSymbols: the result encoding table
//   encoding bits: current encoding pattern in the recurrence
//   iBit: current bit in the recurrence
// Output:
//   the encoding table is update according to the encoding tree.
static void CreateHuffmanEncoding(
  CapyHuffmanSymbol const* const encodingTree,
      CapyHuffmanEncoding* const encodingSymbols,
                     bool* const encodingBits,
                    size_t const iBit) {
  if(encodingTree->childs[0] == NULL && encodingTree->childs[1] == NULL) {
    encodingSymbols[encodingTree->symbol].nbBit = iBit;
    loop(i, iBit) {
      encodingSymbols[encodingTree->symbol].bits[i] = encodingBits[i];
    }
  } else {
    loop(iChild, 2) {
      encodingBits[iBit] = iChild;
      CreateHuffmanEncoding(
        encodingTree->childs[iChild], encodingSymbols, encodingBits, iBit + 1);
    }
  }
}

// Free a CapyHuffmanSymbol
// Input:
//   that: the tree to free
// Output:
//   The tree is freed
static void FreeHuffmanEncodingTree(CapyHuffmanSymbol** const that) {
  loop(iChild, 2) if((*that)->childs[iChild] != NULL) {
    FreeHuffmanEncodingTree((*that)->childs + iChild);
  }
  free(*that);
  *that = NULL;
}

// Compress the data using Huffman
// Input:
//   data: the data to compress
// Output:
//   Return the compressed data
static CapyCompressorData CompressHuffman(CapyCompressorData const data) {
  CapyCompressorData res = {0};
  if(data.size == 0 || data.bytes == NULL) return res;

  // Get the encoding tree
  CapyHuffmanSymbol* encodingTree = CreateHuffmanEncodingTree(data);

  // Calculate the number of bits necessary for the encoding tree
  size_t nbBitTree = GetSizeBitHuffmanEncodingTree(encodingTree);

  // Create the conversion table for encoding
  CapyHuffmanEncoding encodingSymbols[256];
  bool encodingBits[256] = {0};
  CreateHuffmanEncoding(encodingTree, encodingSymbols, encodingBits, 0);

  // Calculate the number of bits necessary for the data
  size_t nbBitData = 0;
  loop(iByte, data.size) nbBitData += encodingSymbols[data.bytes[iByte]].nbBit;

  // Create a bit array to convert the encoding tree and encoded data
  CapyBitArray bitArray = CapyBitArrayCreate();
  $(&bitArray, resize)(nbBitTree + nbBitData + 3 * sizeof(size_t) * 8);

  // Add the uncompressed data size
  size_t iBit = 0;
  size_t sizeSizeTBit = sizeof(size_t) * 8;
  loop(iShift, sizeSizeTBit) {
    bool const bit =
      ((data.size & (((size_t)1) << (sizeSizeTBit - 1 - iShift))) != 0);
    $(&bitArray, setBit)(iBit, bit);
    iBit += 1;
  }

  // Add the encoding tree size
  loop(iShift, sizeSizeTBit) {
    bool const bit =
      ((nbBitTree & (((size_t)1) << (sizeSizeTBit - 1 - iShift))) != 0);
    $(&bitArray, setBit)(iBit, bit);
    iBit += 1;
  }

  // Convert the encoding tree
  ConvertHuffmanEncodingTreeToBitArray(encodingTree, &bitArray, &iBit);

  // Add the compressed data size
  loop(iShift, sizeSizeTBit) {
    bool const bit =
      ((nbBitData & (((size_t)1) << (sizeSizeTBit - 1 - iShift))) != 0);
    $(&bitArray, setBit)(iBit, bit);
    iBit += 1;
  }

  // Encode the data
  loop(iByte, data.size) {
    loop(jBit, encodingSymbols[data.bytes[iByte]].nbBit) {
      bool const bit = encodingSymbols[data.bytes[iByte]].bits[jBit];
      $(&bitArray, setBit)(iBit, bit);
      iBit += 1;
    }
  }

  // Copy the bit array in the compressed data
  res.size = bitArray.size / 8;
  res.size += ((bitArray.size % 8) != 0);
  safeMalloc(res.bytes, res.size);
  memcpy(res.bytes, bitArray.data, res.size);

  // Free memory
  FreeHuffmanEncodingTree(&encodingTree);
  $(&bitArray, destruct)();

  // Return the result of compression
  return res;
}

// Read the encoding tree for Huffman compression
// Input:
//   bitArray: the bit array containing the encoding tree
//   iBit: the index in the bit array
// Output:
//   Return the encoding tree and update the index in the bit array
static CapyHuffmanSymbol* ReadHuffmanEncodingTree(
  CapyBitArray const* const bitArray,
              size_t* const iBit) {
  CapyHuffmanSymbol* encodingTree = NULL;
  safeMalloc(encodingTree, 1);
  if(encodingTree == NULL) return NULL;
  *encodingTree = (CapyHuffmanSymbol){
    .symbol = 0,
    .nbOcc = 0,
    .childs = {NULL, NULL},
  };
  bool bit = $(bitArray, getBit)(*iBit);
  *iBit += 1;
  if(bit) {
    loop(jBit, 8) {
      bit = $(bitArray, getBit)(*iBit);
      *iBit += 1;
      encodingTree->symbol |= (((uint8_t)bit) << (7 - jBit));
    }
  } else {
    loop(iChild, 2) {
      encodingTree->childs[iChild] = ReadHuffmanEncodingTree(bitArray, iBit);
    }
  }
  return encodingTree;
}

// Decompress the data using Huffman
// Input:
//   data: the data to decompress
// Output:
//   Return the decompressed data
static CapyCompressorData DecompressHuffman(CapyCompressorData const data) {
  CapyCompressorData res = {0};
  if(data.size == 0 || data.bytes == NULL) return res;

  // Create a bit array with the data
  CapyBitArray bitArray = CapyBitArrayCreate();
  bitArray.data = data.bytes;
  bitArray.size = data.size * 8;

  // Read the uncompressed data size
  size_t iBit = 0;
  size_t sizeSizeTBit = sizeof(size_t) * 8;
  res.size = 0;
  loop(iShift, sizeSizeTBit) {
    bool const bit = $(&bitArray, getBit)(iBit);
    res.size |= ((size_t)bit) << (sizeSizeTBit - 1 - iShift);
    iBit += 1;
  }

  // Allocate memory for the uncompressed data
  safeMalloc(res.bytes, res.size);

  // Read the encoding tree size
  size_t nbBitTree = 0;
  loop(iShift, sizeSizeTBit) {
    bool const bit = $(&bitArray, getBit)(iBit);
    nbBitTree |= ((size_t)bit) << (sizeSizeTBit - 1 - iShift);
    iBit += 1;
  }

  // Read the encoding tree
  CapyHuffmanSymbol* encodingTree = ReadHuffmanEncodingTree(&bitArray, &iBit);

  // Read the compressed data size
  size_t nbBitData = 0;
  loop(iShift, sizeSizeTBit) {
    bool const bit = $(&bitArray, getBit)(iBit);
    nbBitData |= ((size_t)bit) << (sizeSizeTBit - 1 - iShift);
    iBit += 1;
  }

  // Decompress data
  size_t iByte = 0;
  CapyHuffmanSymbol const* ptrSymbol = encodingTree;
  loop(jBit, nbBitData) {
    bool const bit = $(&bitArray, getBit)(iBit);
    iBit += 1;
    ptrSymbol = ptrSymbol->childs[bit];
    if(ptrSymbol->childs[0] == NULL && ptrSymbol->childs[1] == NULL) {
      res.bytes[iByte] = ptrSymbol->symbol;
      iByte += 1;
      ptrSymbol = encodingTree;
    }
  }

  // Free memory
  FreeHuffmanEncodingTree(&encodingTree);

  // Return the result of decompression
  return res;
}

// Free the memory used by a CapyCompressor
static void DestructHuffman(void) {
  methodOf(CapyHuffmanCompressor);
  $(that, destructCapyCompressor)();
}

// Create a CapyHuffmanCompressor
// Output:
//   Return a CapyHuffmanCompressor
CapyHuffmanCompressor CapyHuffmanCompressorCreate(void) {
  CapyHuffmanCompressor that;
  CapyInherits(that, CapyCompressor, ());
  that.compress = CompressHuffman;
  that.decompress = DecompressHuffman;
  that.destruct = DestructHuffman;
  return that;
}

// Allocate memory for a new CapyHuffmanCompressor and create it
// Output:
//   Return a CapyHuffmanCompressor
// Exception:
//   May raise CapyExc_MallocFailed.
CapyHuffmanCompressor* CapyHuffmanCompressorAlloc(void) {
  CapyHuffmanCompressor* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyHuffmanCompressorCreate();
  return that;
}

// Free the memory used by a CapyHuffmanCompressor* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyHuffmanCompressor to free
void CapyHuffmanCompressorFree(CapyHuffmanCompressor** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
