// ---------------------------------- compressor.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache info@baillehachepascal.dev
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_COMPRESSOR_H
#define CAPY_COMPRESSOR_H
#include "externalHeaders.h"
#include "cext.h"

// Description:
// Compressor class.

// Structure to memorise the compressed/decompressed data and its size in byte
typedef struct CapyCompressorData {

  // Bytes of data
  uint8_t* bytes;

  // Size in byte
  size_t size;
} CapyCompressorData;

// CapyCompressor object definition macro
//
// Compress the data
// Input:
//   data: the data to compress
// Output:
//   Return the compressed data
// CapyCompressorData (*compress)(CapyCompressorData const data);
//
// Decompress the data
// Input:
//   data: the data to decompress
// Output:
//   Return the decompressed data
// CapyCompressorData (*decompress)(CapyCompressorData const data);
#define CapyCompressorDef {                                        \
  void (*destruct)(void);                                          \
  CapyCompressorData (*compress)(CapyCompressorData const data);   \
  CapyCompressorData (*decompress)(CapyCompressorData const data); \
}

// CapyCompressor object
typedef struct CapyCompressor CapyCompressorDef CapyCompressor;

// Create a CapyCompressor
// Output:
//   Return a CapyCompressor
CapyCompressor CapyCompressorCreate(void);

// Allocate memory for a new CapyCompressor and create it
// Output:
//   Return a CapyCompressor
// Exception:
//   May raise CapyExc_MallocFailed.
CapyCompressor* CapyCompressorAlloc(void);

// Free the memory used by a CapyCompressor* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyCompressor to free
void CapyCompressorFree(CapyCompressor** const that);

// RLECompressor object
typedef struct CapyRLECompressor {

  // Inherit CapyCompressor
  struct CapyCompressorDef;

  // Destructor
  void (*destructCapyCompressor)(void);
} CapyRLECompressor;

// Create a CapyRLECompressor
// Output:
//   Return a CapyRLECompressor
CapyRLECompressor CapyRLECompressorCreate(void);

// Allocate memory for a new CapyRLECompressor and create it
// Output:
//   Return a CapyRLECompressor
// Exception:
//   May raise CapyExc_MallocFailed.
CapyRLECompressor* CapyRLECompressorAlloc(void);

// Free the memory used by a CapyCompressor* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyCompressor to free
void CapyRLECompressorFree(CapyRLECompressor** const that);

// BWTRLECompressor object
typedef struct CapyBWTRLECompressor {

  // Inherit CapyCompressor
  struct CapyCompressorDef;

  // Destructor
  void (*destructCapyCompressor)(void);
} CapyBWTRLECompressor;

// Create a CapyBWTRLECompressor
// Output:
//   Return a CapyBWTRLECompressor
CapyBWTRLECompressor CapyBWTRLECompressorCreate(void);

// Allocate memory for a new CapyBWTRLECompressor and create it
// Output:
//   Return a CapyBWTRLECompressor
// Exception:
//   May raise CapyExc_MallocFailed.
CapyBWTRLECompressor* CapyBWTRLECompressorAlloc(void);

// Free the memory used by a CapyCompressor* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyCompressor to free
void CapyBWTRLECompressorFree(CapyBWTRLECompressor** const that);

// HuffmanCompressor object
typedef struct CapyHuffmanCompressor {

  // Inherit CapyCompressor
  struct CapyCompressorDef;

  // Destructor
  void (*destructCapyCompressor)(void);
} CapyHuffmanCompressor;

// Create a CapyHuffmanCompressor
// Output:
//   Return a CapyHuffmanCompressor
CapyHuffmanCompressor CapyHuffmanCompressorCreate(void);

// Allocate memory for a new CapyHuffmanCompressor and create it
// Output:
//   Return a CapyHuffmanCompressor
// Exception:
//   May raise CapyExc_MallocFailed.
CapyHuffmanCompressor* CapyHuffmanCompressorAlloc(void);

// Free the memory used by a CapyCompressor* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyCompressor to free
void CapyHuffmanCompressorFree(CapyHuffmanCompressor** const that);
#endif
