// ---------------------------- date.c ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "date.h"

// Set the date values from its representation as a uint64_t (yyyymmddhhiiss).
// Input:
//   date: the date
// Output:
//   The date values are updated. 'mm' and 'dd' start at 1.
static void SetFromUInt64(uint64_t const date) {
  methodOf(CapyDate);
  uint64_t c[capyDateUnit_nb] = {
    10000000000UL,
    100000000UL,
    1000000UL,
    10000UL,
    100UL,
    1UL,
  };
  loop(i, capyDateUnit_nb) {
    if(i == 0) that->vals[i] = (CapyDate_t)(date / c[i]);
    else {
      that->vals[i] =
        (CapyDate_t)((date - (date / c[i - 1]) * c[i - 1]) / c[i]);
    }
  }
  that->vals[capyDateUnit_day] -= 1;
  that->vals[capyDateUnit_month] -= 1;
}

// Get the date as a uint64_t (yyyymmddhhiiss).
// Output:
//   Return the formated date. 'mm' and 'dd' start at 1.
static uint64_t GetAsUInt64(void) {
  methodOf(CapyDate);
  uint64_t date = 0;
  uint64_t c[capyDateUnit_nb] = {
    10000000000UL,
    100000000UL,
    1000000UL,
    10000UL,
    100UL,
    1UL,
  };
  that->vals[capyDateUnit_day] += 1;
  that->vals[capyDateUnit_month] += 1;
  loop(i, capyDateUnit_nb) {
    if(that->vals[i] < 0) raiseExc(CapyExc_InvalidParameters);
    date += (uint64_t)(that->vals[i]) * c[i];
  }
  that->vals[capyDateUnit_day] -= 1;
  that->vals[capyDateUnit_month] -= 1;
  return date;
}

// Get the delay to another date in a given unit.
// Input:
//   date: the other date
//   unit: the unit in which the result is returned
// Output:
//   Return the delay from 'that' to 'date' in the given unit.
static int64_t GetDelayTo(
  CapyDate const* const date,
     CapyDateUnit const unit) {
  methodOf(CapyDate);
  int64_t d[2] = {0};
  CapyDate_t c[capyDateUnit_nb] = { 12, 30, 24, 60, 60, 1 };
  loop(i, capyDateUnit_nb) {
    if(i == 0) {
      d[0] = that->vals[i];
      d[1] = date->vals[i];
    } else {
      d[0] = that->vals[i] + d[0] * c[i - 1];
      d[1] = date->vals[i] + d[1] * c[i - 1];
    }
  }
  int64_t delay = d[1] - d[0];
  int64_t conv = 1;
  loop(i, capyDateUnit_nb - unit) conv *= c[capyDateUnit_nb - 1 - i];
  delay /= conv;
  return delay;
}

// Get the day of the week of the date (year > 1752)
// Output:
//   Return the day of the week
static CapyWeekDay GetWeekDay(void) {
  methodOf(CapyDate);
  if(that->vals[capyDateUnit_year] <= 1752) raiseExc(CapyExc_InvalidParameters);
  int t[] = {0, 3, 2, 5, 0, 3, 5, 1, 4, 6, 2, 4};
  CapyDate_t y = that->vals[capyDateUnit_year];
  CapyDate_t m = that->vals[capyDateUnit_month] + 1;
  CapyDate_t d = that->vals[capyDateUnit_day] + 1;
  if(m < 3) y -= 1;
  return (y + y / 4 - y / 100 + y / 400 + t[m - 1] + d) % 7;
}

// Normalise a date.
// Output:
//   'that' is updated
static void Normalise(void) {
  methodOf(CapyDate);
  CapyDate_t c[capyDateUnit_nb] = { 1, 12, 30, 24, 60, 60 };
  loop(iUnit, capyDateUnit_nb - 1) {
    __typeof__(iUnit) jUnit = capyDateUnit_nb - 1 - iUnit;
    while(that->vals[jUnit] < 0) {
      that->vals[jUnit] += c[jUnit];
      that->vals[jUnit - 1] -= 1;
      if(that->vals[jUnit - 1] == 0) {
      }
    }
    while(that->vals[jUnit] >= c[jUnit]) {
      that->vals[jUnit] -= c[jUnit];
      that->vals[jUnit - 1] += 1;
    }
  }
}

// Add another date
// Input:
//   date: the date to add
// Output:
//   'that' is updated.
static void AddDate(CapyDate const* const date) {
  methodOf(CapyDate);
  loop(iUnit, capyDateUnit_nb) that->vals[iUnit] += date->vals[iUnit];
  $(that, normalise)();
}

// Print the date on a stream in 'yyyy-mm-dd hh:ii:ss' format
// Input:
//   stream: the stream
// Output:
//   'that' is printed
static void Print(FILE* const stream) {
  methodOf(CapyDate);
  fprintf(
    stream, "%04d-%02d-%02d %02d:%02d:%02d",
    that->vals[capyDateUnit_year],
    that->vals[capyDateUnit_month] + 1,
    that->vals[capyDateUnit_day] + 1,
    that->vals[capyDateUnit_hour],
    that->vals[capyDateUnit_minute],
    that->vals[capyDateUnit_second]);
}

// Convert the date to a CapyDateCosSin
// Output:
//   Return a cos/sin encoding of the month/day of the date.
//   cos/sin(1,0) represents january, 1st
//   cos/sin(0,1) represents april, 1st
//   cos/sin(-1,0) represents july, 1st
//   cos/sin(0,-1) represents october, 1st
static CapyDateCosSin GetDateAsCosSin(void) {
  methodOf(CapyDate);
  CapyDateCosSin cosSin = {0};
  double theta = that->vals[capyDateUnit_month] * 30.0;
  theta = (theta + that->vals[capyDateUnit_day]) / 360.0 * 2.0 * M_PI;
  cosSin.cos = cos(theta);
  cosSin.sin = sin(theta);
  return cosSin;
}

// Convert the time to a CapyDateCosSin
// Output:
//   Return a cos/sin encoding of the hour/minute/second of the date.
//   cos/sin(1,0) represents 00:00:00
//   cos/sin(0,1) represents 06:00:00
//   cos/sin(-1,0) represents 12:00:00
//   cos/sin(0,-1) represents 18:00:00
static CapyDateCosSin GetTimeAsCosSin(void) {
  methodOf(CapyDate);
  CapyDateCosSin cosSin = {0};
  double theta = that->vals[capyDateUnit_hour] * 60.0;
  theta = (theta + that->vals[capyDateUnit_minute]) * 60.0;
  theta = (theta + that->vals[capyDateUnit_second]) / 86400.0 * 2.0 * M_PI;
  cosSin.cos = cos(theta);
  cosSin.sin = sin(theta);
  return cosSin;
}

// Get the date as a real where month, day, ... are normalised within each
// year
// Output:
//   Return the converted date.
static double GetDateAsNormalisedDouble(void) {
  methodOf(CapyDate);
  double date = ((double)(that->vals[capyDateUnit_second])) / 60.0;
  date = (date + (double)(that->vals[capyDateUnit_minute])) / 60.0;
  date = (date + (double)(that->vals[capyDateUnit_hour])) / 24.0;
  date = (date + (double)(that->vals[capyDateUnit_day])) / 30.0;
  date = (date + (double)(that->vals[capyDateUnit_month])) / 12.0;
  date += (double)(that->vals[capyDateUnit_year]);
  return date;
}

// Get the time as a real normalised within a day
// Output:
//   Return the converted time.
static double GetTimeAsNormalisedDouble(void) {
  methodOf(CapyDate);
  double date = ((double)(that->vals[capyDateUnit_second])) / 60.0;
  date = (date + (double)(that->vals[capyDateUnit_minute])) / 60.0;
  date = (date + (double)(that->vals[capyDateUnit_hour])) / 24.0;
  return date;
}

// Set the date from a real where month, day, ... are normalised within each
// year
// Output:
//   Update the date
static void SetDateFromNormalisedDouble(double const val) {
  methodOf(CapyDate);
  double v = val;
  that->vals[capyDateUnit_year] = (CapyDate_t)floor(v);
  v -= (double)(that->vals[capyDateUnit_year]);
  that->vals[capyDateUnit_month] = (CapyDate_t)floor(v * 12.0);
  v -= ((double)(that->vals[capyDateUnit_month])) / 12.0;
  v *= 12.0;
  that->vals[capyDateUnit_day] = (CapyDate_t)floor(v * 30.0);
  v -= ((double)(that->vals[capyDateUnit_day])) / 30.0;
  v *= 30.0;
  that->vals[capyDateUnit_hour] = (CapyDate_t)floor(v * 24.0);
  v -= ((double)(that->vals[capyDateUnit_hour])) / 24.0;
  v *= 24.0;
  that->vals[capyDateUnit_minute] = (CapyDate_t)floor(v * 60.0);
  v -= ((double)(that->vals[capyDateUnit_minute])) / 60.0;
  v *= 60.0;
  that->vals[capyDateUnit_second] = (CapyDate_t)floor(v * 60.0);
}

// Set the time from a real normalised within a day
// Output:
//   Update the time.
static void SetTimeFromNormalisedDouble(double const val) {
  methodOf(CapyDate);
  double v = val;
  that->vals[capyDateUnit_hour] = (CapyDate_t)floor(val * 24.0);
  v -= ((double)(that->vals[capyDateUnit_hour])) / 24.0;
  v *= 24.0;
  that->vals[capyDateUnit_minute] = (CapyDate_t)floor(v * 60.0);
  v -= ((double)(that->vals[capyDateUnit_minute])) / 60.0;
  v *= 60.0;
  that->vals[capyDateUnit_second] = (CapyDate_t)floor(v * 60.0);
}


// Free the memory used by a CapyDate
static void Destruct(void) {
  return;
}

// Create a CapyDate
// Output:
//   Return a CapyDate
CapyDate CapyDateCreate(void) {
  CapyDate that = {
    .destruct = Destruct,
    .setFromUInt64 = SetFromUInt64,
    .getAsUInt64 = GetAsUInt64,
    .getDelayTo = GetDelayTo,
    .getWeekDay = GetWeekDay,
    .addDate = AddDate,
    .normalise = Normalise,
    .print = Print,
    .getDateAsCosSin = GetDateAsCosSin,
    .getTimeAsCosSin = GetTimeAsCosSin,
    .getDateAsNormalisedDouble = GetDateAsNormalisedDouble,
    .getTimeAsNormalisedDouble = GetTimeAsNormalisedDouble,
    .setDateFromNormalisedDouble = SetDateFromNormalisedDouble,
    .setTimeFromNormalisedDouble = SetTimeFromNormalisedDouble,
  };
  return that;
}

// Allocate memory for a new CapyDate and create it
// Output:
//   Return a CapyDate
// Exception:
//   May raise CapyExc_MallocFailed.
CapyDate* CapyDateAlloc(void) {
  CapyDate* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyDateCreate();
  return that;
}

// Free the memory used by a CapyDate* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyDate to free
void CapyDateFree(CapyDate** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
