// ---------------------------------- date.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_DATE_H
#define CAPY_DATE_H
#include "externalHeaders.h"
#include "cext.h"

// Description:
// Date class.

// Date units
typedef enum CapyDateUnit {
  capyDateUnit_year,
  capyDateUnit_month,
  capyDateUnit_day,
  capyDateUnit_hour,
  capyDateUnit_minute,
  capyDateUnit_second,
  capyDateUnit_nb,
} CapyDateUnit;

// Day of week
typedef enum CapyWeekDay {
  capyWeekDay_sunday,
  capyWeekDay_monday,
  capyWeekDay_tuesday,
  capyWeekDay_wednesday,
  capyWeekDay_thursday,
  capyWeekDay_friday,
  capyWeekDay_saturday,
  capyWeekDay_nb,
} CapyWeekDay;

// Type for the values of a CapyDate
typedef int16_t CapyDate_t;

// Structure to encode a date or time as cos/sin values
typedef struct CapyDateCosSin {
  double cos;
  double sin;
} CapyDateCosSin;

// Date object
typedef struct CapyDate CapyDate;
struct CapyDate {

  // Values (default: all 0), if not specified otherwise assumes for
  // calculation:
  // second in [0, 59]
  // minute in [0, 59]
  // hour in [0, 23]
  // day in [0, 30]
  // month in [0, 11]
  // 60 seconds in one minute
  // 60 minutes in one hour
  // 24 hours in one day
  // 30 days in one month
  // 12 months in one year
  CapyDate_t vals[capyDateUnit_nb];
  CapyPad(CapyDate_t[2], vals);

  // Destructor
  void (*destruct)(void);

  // Set the date values from its representation as a uint64_t (yyyymmddhhiiss).
  // Input:
  //   date: the date
  // Output:
  //   The date values are updated. 'mm' and 'dd' start at 1.
  void (*setFromUInt64)(uint64_t const date);

  // Get the date as a uint64_t (yyyymmddhhiiss).
  // Output:
  //   Return the formated date. 'mm' and 'dd' start at 1.
  uint64_t (*getAsUInt64)(void);

  // Get the delay to another date in a given unit.
  // Input:
  //   date: the other date
  //   unit: the unit in which the result is returned
  // Output:
  //   Return the delay from 'that' to 'date' in the given unit. Uses 30 days
  //   per month if necessary.
  int64_t (*getDelayTo)(
    CapyDate const* const date,
       CapyDateUnit const unit);

  // Get the day of the week of the date (year > 1752)
  // Output:
  //   Return the day of the week
  CapyWeekDay (*getWeekDay)(void);

  // Normalise a date
  // Output:
  //   'that' is updated
  void (*normalise)(void);

  // Add another date
  // Input:
  //   date: the date to add
  // Output:
  //   'that' is updated.
  void (*addDate)(CapyDate const* const date);

  // Print the date on a stream in 'yyyy-mm-dd hh:ii:ss' format
  // Input:
  //   stream: the stream
  // Output:
  //   'that' is printed
  void (*print)(FILE* const stream);

  // Convert the date to a CapyDateCosSin
  // Output:
  //   Return a cos/sin encoding of the month/day of the date.
  //   cos/sin(1,0) represents january, 1st
  //   cos/sin(0,1) represents april, 1st
  //   cos/sin(-1,0) represents july, 1st
  //   cos/sin(0,-1) represents october, 1st
  CapyDateCosSin (*getDateAsCosSin)(void);

  // Convert the time to a CapyDateCosSin
  // Output:
  //   Return a cos/sin encoding of the hour/minute/second of the date.
  //   cos/sin(1,0) represents 00:00:00
  //   cos/sin(0,1) represents 06:00:00
  //   cos/sin(-1,0) represents 12:00:00
  //   cos/sin(0,-1) represents 18:00:00
  CapyDateCosSin (*getTimeAsCosSin)(void);

  // Get the date as a real where month, day, ... are normalised within each
  // year
  // Output:
  //   Return the converted date.
  double (*getDateAsNormalisedDouble)(void);

  // Get the time as a real normalised within a day
  // Output:
  //   Return the converted time.
  double (*getTimeAsNormalisedDouble)(void);

  // Set the date from a real where month, day, ... are normalised within each
  // year
  // Output:
  //   Update the date
  void (*setDateFromNormalisedDouble)(double const val);

  // Set the time from a real normalised within a day
  // Output:
  //   Update the time.
  void (*setTimeFromNormalisedDouble)(double const val);
};

// Create a CapyDate
// Output:
//   Return a CapyDate
CapyDate CapyDateCreate(void);

// Allocate memory for a new CapyDate and create it
// Output:
//   Return a CapyDate
// Exception:
//   May raise CapyExc_MallocFailed.
CapyDate* CapyDateAlloc(void);

// Free the memory used by a CapyDate* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyDate to free
void CapyDateFree(CapyDate** const that);
#endif
