// --------------------------------- dict.h ----------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_DICT_H
#define CAPY_DICT_H
#include "externalHeaders.h"
#include "cext.h"
#include "hashFun.h"

// Description:
// Generic dictionary (implemented as a hash table).

// Iterator for generic dict structure. Declaration macro for an iterator
// structure named 'name' ## Iterator, associated to a dict named 'name'
#define CapyDecDictIterator(name, type)                          \
typedef struct name name;                                      \
typedef struct name ## Entry name ## Entry;                    \
struct name ## Entry {                                         \
  char* key;                                                   \
  type val;                                                    \
  CapyPad(type, 0);                                            \
  name ## Entry* next;                                         \
};                                                             \
typedef struct name ## Iterator {                                \
  size_t idx;                                                    \
  name ## Entry* entry;                                          \
  size_t idxHash;                                                \
  name* dict;                                                    \
  name ## Entry datatype;                                        \
  void (*destruct)(void);                                        \
  name ## Entry* (*reset)(void);                                 \
  name ## Entry* (*next)(void);                                  \
  bool (*isActive)(void);                                        \
  name ## Entry* (*get)(void);                                   \
} name ## Iterator;                                              \
name ## Iterator name ## Iterator ## Create(name* const dict);   \
name ## Iterator* name ## Iterator ## Alloc(name* const dict);   \
void name ## Iterator ## Destruct(void);                         \
void name ## Iterator ## Free(name ## Iterator** const that);    \
name ## Entry* name ## Iterator ## Reset(void);                  \
name ## Entry* name ## Iterator ## Next(void);                   \
bool name ## Iterator ## IsActive(void);                         \
name ## Entry* name ## Iterator ## Get(void);

// Create an iterator on a generic dictionary
// Input:
//   dict: the generic dict on which to iterate
// Output:
//   Return the iterator
#define CapyDefDictIteratorCreate(name)                         \
name ## Iterator name ## Iterator ## Create(name* const dict) { \
  name ## Iterator that = {                                     \
    .idx = 0,                                                   \
    .entry = NULL,                                              \
    .idxHash = 0,                                               \
    .dict = dict,                                               \
    .destruct = name ## Iterator ## Destruct,                   \
    .reset = name ## Iterator ## Reset,                         \
    .next = name ## Iterator ## Next,                           \
    .isActive = name ## Iterator ## IsActive,                   \
    .get = name ## Iterator ## Get,                             \
  };                                                            \
  $(&that, reset)();                                            \
  return that;                                                  \
}

// Allocate memory and create an iterator on a generic dict
// Input:
//   dict: the dictionary on which to iterate
// Output:
//   Return the iterator
#define CapyDefDictIteratorAlloc(name)                          \
name ## Iterator* name ## Iterator ## Alloc(name* const dict) { \
  name ## Iterator* that = NULL;                                \
  safeMalloc(that, sizeof(name ## Iterator));                   \
  if(!that) return NULL;                                        \
  *that = name ## Iterator ## Create(dict);                     \
  return that;                                                  \
}

// Free the memory used by an iterator.
// Input:
//   that: the iterator to free
#define CapyDefDictIteratorDestruct(name) \
void name ## Iterator ## Destruct(void) { \
  return;                                 \
}

// Free the memory used by a pointer to an iterator and reset '*that' to NULL
// Input:
//   that: a pointer to the iterator to free
#define CapyDefDictIteratorFree(name)                          \
void name ## Iterator ## Free(name ## Iterator** const that) { \
  if(*that == NULL) return;                                    \
  $(*that, destruct)();                                        \
  free(*that);                                                 \
  *that = NULL;                                                \
}

// Reset the iterator
// Output:
//   Return the first element of the iteration
#define CapyDefDictIteratorReset(name)                  \
name ## Entry* name ## Iterator ## Reset(void) {        \
  name ## Iterator* that = (name ## Iterator*)capyThat; \
  that->idx = 0;                                        \
  that->idxHash = 0;                                    \
  that->entry = NULL;                                   \
  while(                                                \
    that->idxHash < that->dict->sizeHash &&             \
    that->dict->data[that->idxHash] == NULL             \
  ) {                                                   \
    ++(that->idxHash);                                  \
  }                                                     \
  if(that->idxHash < that->dict->sizeHash)              \
    that->entry = that->dict->data[that->idxHash];      \
  if(name ## Iterator ## IsActive())                    \
    return name ## Iterator ## Get();                   \
  else                                                  \
    return NULL;                                        \
}

// Move the iterator to the next element
// Output:
//   Return the next element of the iteration
#define CapyDefDictIteratorNext(name)                   \
name ## Entry* name ## Iterator ## Next(void) {         \
  name ## Iterator* that = (name ## Iterator*)capyThat; \
  if(that->entry == NULL) return NULL;                  \
  if(that->entry->next) {                               \
    that->entry = that->entry->next;                    \
    ++(that->idx);                                      \
  } else {                                              \
    that->entry = NULL;                                 \
    ++(that->idxHash);                                  \
    while(                                              \
      that->idxHash < that->dict->sizeHash &&           \
      that->dict->data[that->idxHash] == NULL           \
    ) {                                                 \
      ++(that->idxHash);                                \
    }                                                   \
    if(that->idxHash < that->dict->sizeHash) {          \
      that->entry = that->dict->data[that->idxHash];    \
      ++(that->idx);                                    \
    }                                                   \
  }                                                     \
  return that->entry;                                   \
}

// Check if the iterator is on a valid element
// Output:
//   Return true if the iterator is on a valid element, else false
#define CapyDefDictIteratorIsActive(name)               \
bool name ## Iterator ## IsActive(void) {               \
  name ## Iterator* that = (name ## Iterator*)capyThat; \
  return (that->entry != NULL);                         \
}

// Get the current element of the iteration
// Output:
//   Return a pointer to the current element
#define CapyDefDictIteratorGet(name)                    \
name ## Entry* name ## Iterator ## Get(void) {          \
  name ## Iterator* that = (name ## Iterator*)capyThat; \
  return that->entry;                                   \
}

// Definition macro calling all the submacros at once for an iterator on an
// dictionary structure named 'name'
#define CapyDefDictIterator(name)   \
  CapyDefDictIteratorCreate(name)   \
  CapyDefDictIteratorAlloc(name)    \
  CapyDefDictIteratorDestruct(name) \
  CapyDefDictIteratorFree(name)     \
  CapyDefDictIteratorReset(name)    \
  CapyDefDictIteratorNext(name)     \
  CapyDefDictIteratorIsActive(name) \
  CapyDefDictIteratorGet(name)

// Generic dict structure. Declaration macro for a dict structure named
// 'name', containing entries with keys of type 'char*' and values of
// type 'type'
#define CapyDecDict(name, type)                                \
CapyDecDictIterator(name, type)                                \
typedef struct name {                                          \
  name ## Entry** data;                                        \
  type defaultVal;                                             \
  CapyPad(type, 0);                                            \
  size_t sizeHash;                                             \
  size_t nbEntry;                                              \
  CapyHashFun* hash;                                           \
  name ## Iterator iter;                                       \
  void (*destruct)(void);                                      \
  size_t (*getNbEntry)(void);                                  \
  size_t (*getSizeHash)(void);                                 \
  void (*set)(char const* const key, type const val);          \
  type (*get)(char const* const key);                          \
  void (*remove)(char const* const key);                       \
  type* (*getPtr)(char const* const key);                      \
  void (*initIterator)(void);                                  \
  name* (*clone)(void);                                        \
  void (*setDefaultVal)(type const val);                       \
  bool (*hasKey)(char const* const key);                       \
  void (*setHash)(CapyHashFun* const hash);                    \
} name;                                                        \
name name ## Create(size_t const size, type const defaultVal); \
name* name ## Alloc(size_t const size, type const defaultVal); \
name* name ## Clone(void);                                     \
void name ## Destruct(void);                                   \
void name ## Free(name** const that);                          \
size_t name ## GetNbEntry(void);                               \
size_t name ## GetSizeHash(void);                              \
void name ## Set(char const* const key, type const val);       \
type name ## Get(char const* const key);                       \
void name ## Remove(char const* const key);                    \
type* name ## GetPtr(char const* const key);                   \
void name ## InitIterator(void);                               \
void name ## SetDefaultVal(type const val);                    \
void name ## SetHash(CapyHashFun* const hash);                 \
bool name ## HasKey(char const* const key);

// Generic dict structure. Definition macro for a dict structure named
// 'name', containing entries with keys of type 'char*' and values of
// type 'type'
// Create a dictionary
// Input:
//   sizeHash: size of the hash table
//   defaultVal: default value returned by 'get' for an unknown key
// Output:
//   Return an empty dictionary containing elements of type 'type'
#define CapyDefDictCreate(name, type)              \
name name ## Create(                               \
  size_t const sizeHash,                           \
    type const defaultVal) {                       \
  name that = {                                    \
    .defaultVal = defaultVal,                      \
    .sizeHash = sizeHash,                          \
    .nbEntry = 0,                                  \
    .hash = (CapyHashFun*)CapyFNV1aHashFunAlloc(), \
    .destruct = name ## Destruct,                  \
    .getNbEntry = name ## GetNbEntry,              \
    .getSizeHash = name ## GetSizeHash,            \
    .set = name ## Set,                            \
    .remove = name ## Remove,                      \
    .get = name ## Get,                            \
    .getPtr = name ## GetPtr,                      \
    .initIterator = name ## InitIterator,          \
    .clone = name ## Clone,                        \
    .setDefaultVal = name ## SetDefaultVal,        \
    .hasKey = name ## HasKey,                      \
    .setHash = name ## SetHash,                    \
  };                                               \
  safeMalloc(that.data, sizeHash);                 \
  assert(that.data != NULL);                       \
  loop(iHash, sizeHash) that.data[iHash] = NULL;   \
  return that;                                     \
}

// Allocate memory for a new dictionary and create it
// Input:
//   sizeHash: size of the hash table
//   defaultVal: default value returned by 'get' for an unknown key
// Output:
//   Return a dictionary containing elements of type 'type'
// Exception:
//   May raise CapyExc_MallocFailed.
#define CapyDefDictAlloc(name, type)            \
name* name ## Alloc(                            \
  size_t const sizeHash,                        \
    type const defaultVal) {                    \
  name* that = NULL;                            \
  safeMalloc(that, 1);                          \
  if(!that) return NULL;                        \
  *that = name ## Create(sizeHash, defaultVal); \
  that->iter = name ## IteratorCreate(that);    \
  return that;                                  \
}

// Clone a dictionary
// Output:
//   Return a clone of the dictionary
// Exception:
//   May raise CapyExc_MallocFailed.
#define CapyDefDictClone(name, type)                              \
name* name ## Clone(void) {                                       \
  name* that = (name*)capyThat;                                   \
  name* clone = name ## Alloc(that->sizeHash, that->defaultVal);  \
  forEach(entry, that->iter) {                                    \
    $(clone, set)(entry.key, entry.val);                          \
  }                                                               \
  return clone;                                                   \
}

// Free the memory used by a dictionary. The memory eventually used by
// the value of the entries must be freed by the user. The key of the
// entries are automatically freed.
// Input:
//   that: the dictionary to free
#define CapyDefDictDestruct(name, type)      \
void name ## Destruct(void) {                \
  name* that = (name*)capyThat;              \
  loop(iHash, that->sizeHash) {              \
    name ## Entry* elem = that->data[iHash]; \
    while(elem) {                            \
      name ## Entry* nextElem = elem->next;  \
      free(elem->key);                       \
      free(elem);                            \
      elem = nextElem;                       \
    }                                        \
  }                                          \
  free(that->data);                          \
  CapyHashFunFree(&(that->hash));            \
  *that = (name){0};                         \
}

// Free the memory used by a pointer to a dictionary and reset '*that'
// to NULL. The memory eventually used by the value of the entries must
// be freed by the user. The key of the entries are automatically freed.
// Input:
//   that: a pointer to the dictionary to free
#define CapyDefDictFree(name, type)          \
void name ## Free(name** const that) {       \
  if(that == NULL || *that == NULL) return;  \
  $(*that, destruct)();                      \
  free(*that);                               \
  *that = NULL;                              \
}

// Get the number of entries in the dictionary
// Output:
//   Return the number of entries
#define CapyDefDictGetNbEntry(name, type) \
size_t name ## GetNbEntry(void) {         \
  return ((name*)capyThat)->nbEntry;      \
}

// Get the size of the hash table of the dictionary
// Output:
//   Return the size of the hash
#define CapyDefDictGetSizeHash(name, type) \
size_t name ## GetSizeHash(void) {         \
  return ((name*)capyThat)->sizeHash;      \
}

// Get a value of the dictionary
// Input:
//   key: the key of the element to get
// Output:
//   Return the value
// Exception:
//   May raise CapyExc_MallocFailed.
#define CapyDefDictGet(name, type)        \
type name ## Get(char const* const key) { \
  return *(name ## GetPtr(key));          \
}

// Get a pointer to a value of the dictionary
// Input:
//   key: the key of the element to get, if there is no element for this
//        key, create a new one initialised with the default value
// Output:
//   Return the pointer to the element
#define CapyDefDictGetPtr(name, type)                                    \
type* name ## GetPtr(char const* const key) {                            \
  name* that = (name*)capyThat;                                          \
  size_t iHash = $(that->hash, eval)(key, strlen(key)) % that->sizeHash; \
  name ## Entry* entry = that->data[iHash];                              \
  while(entry && strcmp(entry->key, key) != 0) entry = entry->next;      \
  if(entry == NULL) {                                                    \
    safeMalloc(entry, 1);                                                \
    if(entry == NULL) return NULL;                                       \
    *entry = (name ## Entry){0};                                         \
    safeSPrintf(&(entry->key), "%s", key);                               \
    entry->val = that->defaultVal;                                       \
    entry->next = that->data[iHash];                                     \
    that->data[iHash] = entry;                                           \
    ++(that->nbEntry);                                                   \
  }                                                                      \
  return &(entry->val);                                                  \
}

// Remove an entry from the dictionary
// Input:
//   key: the key of the element to remove
// Output:
//   The entry is removed if it exists, else nothing is done. The user is
//   responsible for freeing the memory used by the entry value if any.
#define CapyDefDictRemove(name, type)                                    \
void name ## Remove(char const* const key) {                             \
  name* that = (name*)capyThat;                                          \
  size_t iHash = $(that->hash, eval)(key, strlen(key)) % that->sizeHash; \
  name ## Entry* entry = that->data[iHash];                              \
  name ## Entry* prevEntry = NULL;                                       \
  while(entry && strcmp(entry->key, key) != 0) {                         \
    prevEntry = entry;                                                   \
    entry = entry->next;                                                 \
  }                                                                      \
  if(entry != NULL) {                                                    \
    if(prevEntry != NULL) prevEntry = entry->next;                       \
    else that->data[iHash] = entry->next;                                \
    free(entry);                                                         \
    --(that->nbEntry);                                                   \
  }                                                                      \
}

// Set a value of the dictionary
// Input:
//   key: the key of the entry to set
//   val: the value to set
// Exception:
//   May raise CapyExc_MallocFailed.
#define CapyDefDictSet(name, type)                           \
void name ## Set(char const* const key, type const val) {    \
  name* that = (name*)capyThat;                              \
  type* entryVal = $(that, getPtr)(key);                     \
  *entryVal = val;                                           \
}

// Initialise the iterator of the dictionary, must be called after the creation
// of the dictionary when it is created with Create(), Alloc() automatically
// initialise the iterator.
#define CapyDefDictInitIterator(name, type)  \
void name ## InitIterator(void) {            \
  name* that = (name*)capyThat;              \
  that->iter = name ## IteratorCreate(that); \
}

// Set the default value used when an entry is automatically added
// Input:
//   val: the default value
#define CapyDefDictSetDefaultVal(name, type) \
void name ## SetDefaultVal(type const val) { \
  name* that = (name*)capyThat;              \
  that->defaultVal = val;                    \
}

// Check if a key exists in the dictionary
// Input:
//   key: the key to check
// Output:
//   Return true if the key exists, else false
#define CapyDefDictHasKey(name, type)                                    \
bool name ## HasKey(char const* const key) {                             \
  name* that = (name*)capyThat;                                          \
  size_t iHash = $(that->hash, eval)(key, strlen(key)) % that->sizeHash; \
  name ## Entry* entry = that->data[iHash];                              \
  while(entry && strcmp(entry->key, key) != 0) entry = entry->next;      \
  return (entry != NULL);                                                \
}

// Set the hash function
// Input:
//   hash: the hash function
#define CapyDefDictSetHash(name, type)          \
void name ## SetHash(CapyHashFun* const hash) { \
  name* that = (name*)capyThat;                 \
  CapyHashFunFree(&(that->hash));               \
  that->hash = hash;                            \
}

// Definition macro calling all the submacros at once for a dictionary
// structure named 'name', containing elements of type 'type'
#define CapyDefDict(name, type)        \
  CapyDefDictCreate(name, type)        \
  CapyDefDictAlloc(name, type)         \
  CapyDefDictClone(name, type)         \
  CapyDefDictDestruct(name, type)      \
  CapyDefDictFree(name, type)          \
  CapyDefDictGetNbEntry(name, type)    \
  CapyDefDictGetSizeHash(name, type)   \
  CapyDefDictSet(name, type)           \
  CapyDefDictGet(name, type)           \
  CapyDefDictGetPtr(name, type)        \
  CapyDefDictInitIterator(name, type)  \
  CapyDefDictSetDefaultVal(name, type) \
  CapyDefDictSetHash(name, type)       \
  CapyDefDictHasKey(name, type)        \
  CapyDefDictRemove(name, type)        \
  CapyDefDictIterator(name)

// Declaration of dictionaries for some basic types
CapyDecDict(CapyDictChar, char)
CapyDecDict(CapyDictInt8, int8_t)
CapyDecDict(CapyDictUInt8, uint8_t)
CapyDecDict(CapyDictInt16, int16_t)
CapyDecDict(CapyDictUInt16, uint16_t)
CapyDecDict(CapyDictInt32, int32_t)
CapyDecDict(CapyDictUInt32, uint32_t)
CapyDecDict(CapyDictInt64, int64_t)
CapyDecDict(CapyDictUInt64, uint64_t)
CapyDecDict(CapyDictFloat, float)
CapyDecDict(CapyDictDouble, double)
CapyDecDict(CapyDictPtrChar, char*)
CapyDecDict(CapyDictPtrInt8, int8_t*)
CapyDecDict(CapyDictPtrUInt8, uint8_t*)
CapyDecDict(CapyDictPtrInt16, int16_t*)
CapyDecDict(CapyDictPtrUInt16, uint16_t*)
CapyDecDict(CapyDictPtrInt32, int32_t*)
CapyDecDict(CapyDictPtrUInt32, uint32_t*)
CapyDecDict(CapyDictPtrInt64, int64_t*)
CapyDecDict(CapyDictPtrUInt64, uint64_t*)
CapyDecDict(CapyDictPtrFloat, float*)
CapyDecDict(CapyDictPtrDouble, double*)
#endif
