// ------------------------------ diffevo.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "diffevo.h"
#include "chrono.h"

// DiffEvoAgent structure
typedef struct CapyDiffEvoAgent {

  // Agent's DNA
  double* dna;

  // Agent's fitness
  double fitness;

  // Agent's constraint satisfaction
  double constraint;
} CapyDiffEvoAgent;

// Update the best DNA, fitness and constraint value (to commonalise code
// in RunChild)
static void UpdateBest(
     CapyDiffEvo* const that,
    double const* const dna,
           double const fitnessVal,
           double const constraintVal,
           size_t const dimIn) {
  sem_wait(&(that->semaphore));
  if(that->verboseStream != NULL) {
    fprintf(
      that->verboseStream,
      "best agent, fitness %lf, constraint %lf \n",
      fitnessVal, constraintVal);
  }
  that->bestFitness = fitnessVal;
  that->bestConstraint = constraintVal;
  memcpy(that->bestDna, dna, sizeof(double) * dimIn);
  sem_post(&(that->semaphore));
}

// Function to randomise an agent
static void RandomiseAgent(
       CapyDiffEvo* const that,
  CapyDiffEvoAgent* const agents,
             size_t const iAgent,
             size_t const nbTryMax,
       CapyMathFun* const fitness,
       CapyMathFun* const constraint,
        CapyRandom* const rng) {
  size_t nbTry = 0;
  agents[iAgent].constraint = -1.0;
  while(agents[iAgent].constraint < 0.0 && nbTry < nbTryMax) {
    loop(iDim, fitness->dimIn) {
      double v = 0.0;
      switch(that->initMode) {
        case capyDiffEvoInitMode_random:
          agents[iAgent].dna[iDim] =
            $(rng, getDoubleRange)(fitness->domains + iDim);
          break;
        case capyDiffEvoInitMode_randomisedSeed:
          v = $(rng, getDoubleRange)(fitness->domains + iDim);
          agents[iAgent].dna[iDim] =
            (1.0 - that->coeffRandomisationSeed) * that->seedDna[iDim] +
            that->coeffRandomisationSeed * v;
          break;
        default:
          raiseExc(CapyExc_UndefinedExecution);
      }
    }
    if(constraint) {
      $(constraint, eval)(agents[iAgent].dna, &(agents[iAgent].constraint));
    } else agents[iAgent].constraint = 1.0;
    ++nbTry;
  }
}

// Run the differential evolution algorithm in its own process
// Input:
//   arg: the CapyDiffEvoThreadData
static void* RunChild(void* arg) {

  // Cast the arguments
  CapyDiffEvo* that = ((CapyDiffEvoThreadData*)arg)->that;
  CapyMathFun* fitness = ((CapyDiffEvoThreadData*)arg)->fitness;
  CapyMathFun* constraint = ((CapyDiffEvoThreadData*)arg)->constraint;

  // Allow the main thread to cancel the child thread immediately in case it
  // dies for example
  int oldState = 0;
  pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, &oldState);

  // Reset the exception id
  that->threadExc = 0;
  try {

    // Create the random generator
    CapyRandom rng = CapyRandomCreate(that->seed);

    // Counter to avoid infinite loops
    size_t nbTry = 0;
    size_t nbTryMax = that->nbAgent * 1000;

    // Create the agents
    CapyDiffEvoAgent* agents = NULL;
    safeMalloc(agents, that->nbAgent);
    if(agents) loop(iAgent, that->nbAgent) {
      agents[iAgent].fitness = 0.0;
      agents[iAgent].dna = NULL;
      agents[iAgent].constraint = 1.0;
      safeMalloc(agents[iAgent].dna, fitness->dimIn);
      if(!(agents[iAgent].dna)) return NULL;

      // If there is a seed, use it for the first agent's DNA
      if(that->seedDna != NULL && iAgent == 0) {
        loop(iDim, fitness->dimIn) {
          agents[iAgent].dna[iDim] = that->seedDna[iDim];
        }

      // Else, initialise the dna randomly
      } else {
        RandomiseAgent(
          that, agents, iAgent, nbTryMax, fitness, constraint, &rng);
      }
    }

    // Allocate memory for the candidate DNA
    double* dna = NULL;
    safeMalloc(dna, fitness->dimIn);
    if(!dna) return NULL;

    // Range of indices for genes
    CapyRangeSize rangeDna = { .min = 0, .max = fitness->dimIn-1 };

    // Range of indices for agents
    CapyRangeSize rangeAgent = { .min = 0, .max = that->nbAgent-1 };

    // Variable to memorise the number of iteration without improvement
    size_t nbIterSinceReset = 0;

    // Variable to memorise the index of the current best agent
    size_t iBestAgent = 0;

    // Loop until the user ask to stop
    while(that->flagStop == false) {

      // If there has been no evolution since many iterations
      if(that->nbIterReset != 0 && nbIterSinceReset >= that->nbIterReset) {

        // Kick the hive !
        if(that->verboseStream != NULL) {
          fprintf(that->verboseStream, "randomise agents \n");
        }
        loop(iAgent, that->nbAgent) if(iAgent != iBestAgent) {
          RandomiseAgent(
            that, agents, iAgent, nbTryMax, fitness, constraint, &rng);
          $(fitness, eval)(agents[iAgent].dna, &(agents[iAgent].fitness));
        }
        nbIterSinceReset = 0;
      }

      // Loop on the agents
      loop(iAgent, that->nbAgent) {

        // If the user asks to stop, break the loop immediately
        if(that->flagStop) break;

        // Select three other agents
        size_t mutAgents[3] = {0, 0, 0};
        do {
          loop(i, 3) mutAgents[i] = $(&rng, getSizeRange)(&rangeAgent);
        } while(
          mutAgents[0] == iAgent || mutAgents[1] == iAgent ||
          mutAgents[2] == iAgent || mutAgents[0] == mutAgents[1] ||
          mutAgents[0] == mutAgents[2] || mutAgents[1] == mutAgents[2]);

        // Select the mutated dimension
        size_t mutDim = $(&rng, getSizeRange)(&rangeDna);

        // Variable to memorise the constraint value of the new agent
        double constraintVal;

        // Loop until we found a valid agent or give up
        bool flagInvalidAgent = true;
        nbTry = 0;
        while(flagInvalidAgent && nbTry < nbTryMax) {

          // Loop on dimensions
          loop(iDim, fitness->dimIn) {

            // Check if that dimension is mutated.
            // In standard mode, mutated dimensions are choosen randomly and
            // at least one mutation is ensured.
            // In sequential mode, dimensions in [idxSeq, idxSeq+strideSeq[
            // are mutated
            bool isMutated = false;
            if(that->mode == capyDiffEvoMode_std) {
              isMutated =
                (iDim == mutDim || $(&rng, getDouble)() < that->probMut);
            } else if(that->mode == capyDiffEvoMode_sequential) {
              isMutated = (
                iDim >= that->idxSeq &&
                iDim < that->idxSeq + that->strideSeq
              );
            }

            // If it's a mutated dimension
            if(isMutated) {

              // Calculate the new value based on the three other agents.
              double diffDna =
                agents[mutAgents[1]].dna[iDim] - agents[mutAgents[2]].dna[iDim];
              dna[iDim] =
                agents[mutAgents[0]].dna[iDim] +
                that->ampMut * $(&rng, getDouble)() * diffDna;

              // Ensure the value stays in the function domain
              if(dna[iDim] < fitness->domains[iDim].min) {
                dna[iDim] = fitness->domains[iDim].min;
              }
              if(dna[iDim] > fitness->domains[iDim].max) {
                dna[iDim] = fitness->domains[iDim].max;
              }

            // Else, it is not a mutated dimension, reuse the previous value
            } else dna[iDim] = agents[iAgent].dna[iDim];
          }

          // If the optimisation is under constraint, check the constraint
          if(constraint) {
            $(constraint, eval)(dna, &constraintVal);
            flagInvalidAgent = (constraintVal < 0.0);
          } else flagInvalidAgent = false;
          ++nbTry;
        }

        // If we have found a valid (with respect to the constraints) agent
        if(flagInvalidAgent == false) {

          // Evaluate the candidate DNA
          double fitnessVal;
          $(fitness, eval)(dna, &fitnessVal);

          // If that's the first evaluation, memorise the intial fitness
          if(that->nbIter == 0 && iAgent == 0) {
            that->initialFitness = fitnessVal;
          }

          // If the candidate is better than its predecessor or it's the
          // first time it is evaluated
          if(
            that->nbIter == 0 ||
            fitnessVal > agents[iAgent].fitness
          ) {
            if(that->verboseStream != NULL && that->nbIter > 0) {
              fprintf(
                that->verboseStream,
                "#%lu, improve agent %lu, fitness %lf -> %lf, "
                "constraint %lf \n",
                that->nbIter, iAgent, agents[iAgent].fitness,
                fitnessVal, constraintVal);
            }

            // Replace the agent's DNA
            agents[iAgent].fitness = fitnessVal;
            memcpy(
              agents[iAgent].dna,
              dna,
              sizeof(double) * fitness->dimIn);
            agents[iAgent].constraint = constraintVal;

            // If the best candidate hasn't been set up
            if(that->nbIter == 0 && iAgent == 0) {
              UpdateBest(that, dna, fitnessVal, constraintVal, fitness->dimIn);
              iBestAgent = iAgent;

            // Else, if the current agent has a better fitness
            } else if(fitnessVal > that->bestFitness) {
              UpdateBest(that, dna, fitnessVal, constraintVal, fitness->dimIn);
              iBestAgent = iAgent;
            }
          }
        }
      }

      // Increment the number of iteration
      ++nbIterSinceReset;
      sem_wait(&(that->semaphore));
      ++(that->nbIter);
      sem_post(&(that->semaphore));

      // Increment the index for sequential evolution
      that->idxSeq = (that->idxSeq + that->strideSeq) % fitness->dimIn;
    }

    // Free memory
    loop(iAgent, that->nbAgent) free(agents[iAgent].dna);
    free(agents);
    free(dna);
    $(&rng, destruct)();
  } endCatch;

  // Update the exception id
  that->threadExc = CapyGetLastExcId();

  // Memorise that the child process has stopped running
  that->running = false;

  // Nothing to return, but needed by pthread
  return NULL;
}

// Run the differential evolution algorithm in its own thread
// Input:
//   fitness: Object performing the fitness calculation, takes the
//            agent's DNA as input and return the fitness value as
//            output. The agent's DNA is an array of double values
//            in fitness' domain. The higher the fitness the better the agent.
//   constraint: function of dimensions (fitness.dimIn, 1) encoding the
//               constraints on the intputs as follow: if the output value
//               is positive or null the constraint is satisfied, else
//               the constraint is not satisfied. The greater the value, the
//               better the constraint is satisfied. A null argument indicates
//               no constraint.
// Output:
//   that->bestFitness, bestDna and bestConstraint are updated with the best
//   agent. The best agent is either the one respecting the constraint and
//   having highest fitness, or if no agent respecting constraint could
//   be found it is the one with highest fitness regardless of the constraint.
static void Run(
  CapyMathFun* const fitness,
  CapyMathFun* const constraint) {
  methodOf(CapyDiffEvo);

  // Avoid starting a new optimisation if one is already running
  if(that->running) return;

  // Initialise the  data
  that->flagStop = false;
  that->nbIter = 0;
  that->initialFitness = 0.0;
  that->bestFitness = 0.0;
  that->bestConstraint = 0.0;
  that->running = false;
  that->sizeDna = fitness->dimIn;
  free(that->bestDna);
  safeMalloc(that->bestDna, fitness->dimIn);
  if(!(that->bestDna)) return;

  // Create the optimising thread
  that->running = true;
  CapyDiffEvoThreadData args = {that, fitness, constraint};
  int ret = pthread_create(&(that->thread), NULL, RunChild, &args);
  if(ret != 0) {
    that->running = false;
    raiseExc(CapyExc_ForkFailed);
  }

  // Wait for the child to actually starts
  while(that->nbIter == 0 && that->running) CapySleepMs(1);
}

// Get the current best DNA (no effect if no optimisation currently
// running)
// Input:
//          dna: array of double updated with the best dna
//      fitness: double updated with the fitness value of the dna
//   constraint: double updated with the constraint value of the dna
static void GetBestDna(
  double* const dna,
  double* const fitness,
  double* const constraint) {
  methodOf(CapyDiffEvo);

  // Lock the best dna if the optimisation is running
  if(that->running) sem_wait(&(that->semaphore));

  // Copy the best dna and its finess
  memcpy(
    dna,
    that->bestDna,
    sizeof(double) * that->sizeDna);
  *fitness = that->bestFitness;
  *constraint = that->bestConstraint;

  // Unlock the best dna if the optimisation is running
  if(that->running) sem_post(&(that->semaphore));
}

// Stop the current optimisation, blocking until the optimisation has
// actually stopped (no effect if there is no optimisation currently
// running)
static void Stop(void) {
  methodOf(CapyDiffEvo);

  // Set the flag to terminate the thread
  that->flagStop = true;

  // Wait for the thread to actually stop, gave up after 10s to avoid being
  // caught in an infinite loop if the child has died
  CapyChrono chrono = CapyChronoCreate();
  $(&chrono, start)();
  while(that->running) {
    CapySleepMs(1);
    $(&chrono, stop)();
    double waitedTime = $(&chrono, getElapsedTime)(capyChrono_second);
    if(waitedTime > 10.0) {
      (void)pthread_cancel(that->thread);
      that->running = false;
    }
  }
}

// Free the memory used by a CapyDiffEvo
static void Destruct(void) {
  methodOf(CapyDiffEvo);

  // Make sure the optimiser is stopped
  $(that, stop)();

  // Free memory
  free(that->bestDna);

  // Destroy the semaphore
  sem_destroy(&(that->semaphore));
}

// Create a CapyDiffEvo
// Output:
//   Return a CapyDiffEvo
CapyDiffEvo CapyDiffEvoCreate(void) {
  CapyDiffEvo that = {
    .seed = (CapyRandomSeed_t)time(NULL),
    .nbAgent = 4,
    .probMut = 0.9,
    .ampMut = 0.8,
    .threadExc = 0,
    .bestDna = NULL,
    .seedDna = NULL,
    .thread = 0,
    .nbIterReset = 0,
    .initMode = capyDiffEvoInitMode_random,
    .coeffRandomisationSeed = 0.1,
    .mode = capyDiffEvoMode_std,
    .idxSeq = 0,
    .strideSeq = 1,
    .verboseStream = NULL,
    .destruct = Destruct,
    .run = Run,
    .getBestDna = GetBestDna,
    .stop = Stop,
  };
  sem_init(&(that.semaphore), 1, 1);
  return that;
}

// Allocate memory for a new CapyDiffEvo and create it
// Output:
//   Return a CapyDiffEvo
// Exception:
//   May raise CapyExc_MallocFailed.
CapyDiffEvo* CapyDiffEvoAlloc(void) {
  CapyDiffEvo* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyDiffEvoCreate();
  return that;
}

// Free the memory used by a CapyDiffEvo* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyDiffEvo to free
void CapyDiffEvoFree(CapyDiffEvo** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
