// ------------------------------- diffevo.h ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_DIFFEVO_H
#define CAPY_DIFFEVO_H
#include "externalHeaders.h"
#include "cext.h"
#include "mathfun.h"
#include "random.h"

// Description:
// Differential evolution algorithm.

// Initialisation modes
typedef enum CapyDiffEvoInitMode {
  capyDiffEvoInitMode_random,
  capyDiffEvoInitMode_randomisedSeed,
} CapyDiffEvoInitMode;

// Evolution modes
typedef enum CapyDiffEvoMode {
  capyDiffEvoMode_std,
  capyDiffEvoMode_sequential
} CapyDiffEvoMode;

// DiffEvo object
typedef struct CapyDiffEvo {

  // Current number of iteration
  size_t nbIter;

  // Size of the dna
  size_t sizeDna;

  // Initial fitness (first fitness evaluated after calling run())
  double initialFitness;

  // Current best fitness
  double bestFitness;

  // Flag to stop the optimisation
  bool flagStop;
  CapyPad(bool, 0);

  // Flag to memorise if the child is running
  bool running;
  CapyPad(bool, 1);

  // Current best agent DNA
  double* bestDna;

  // Constraint value of the current best dna
  double bestConstraint;

  // Seed for the RNG
  // By default, the current time. Can be modified before calling run()
  CapyRandomSeed_t seed;

  // Differential evolution algorithm coefficients
  // By default, probMut = 0.9 and ampMut = 0.8
  // probMut in [0, 1], ampMut in [0, 2]. Can be modified before calling
  // run(). probMut has no effect if mode==capyDiffEvoMode_sequential
  double probMut;
  double ampMut;

  // Population size
  // By default, nbAgent = 10 * dim. Can be modified before calling
  // run()
  size_t nbAgent;

  // The child thread
  pthread_t thread;

  // Exception value returned by the rendering thread
  CapyException_t threadExc;
  CapyPad(CapyException_t, threadExc);

  // Semaphore to manage shared data access
  sem_t semaphore;

  // Seed used to initialise the first DNA of the first agent if not
  // null
  double const* seedDna;

  // Maximum number of iteration without improvement before randomising the
  // agents (never reset if equals to 0, default: 0)
  size_t nbIterReset;

  // Initialisation mode (default: capyDiffEvoInitMode_random)
  CapyDiffEvoInitMode initMode;
  CapyPad(CapyDiffEvoInitMode, initMode);

  // Coeff randomisation seed for initMode 'randomisedSeed' (in [0,1],
  // default: 0.1). The seed is randomised as (1 - c) * seed + c * r, where
  // c is coeffRandomisationSeed and r is a random value in the domain.
  double coeffRandomisationSeed;

  // Evolution mode (default: capyDiffEvoMode_std)
  CapyDiffEvoMode mode;
  CapyPad(CapyDiffEvoInitMode, mode);

  // Index and stride used when mode==capyDiffEvoMode_sequential. strideEq=1
  // by default, the stride is the number of gene evoluting per iteration
  size_t idxSeq;
  size_t strideSeq;

  // Verbose stream (default: NULL). If not null print info during search.
  FILE* verboseStream;

  // Destructor
  void (*destruct)(void);

  // Run the differential evolution algorithm in its own thread
  // Input:
  //   fitness: Object performing the fitness calculation, takes the
  //            agent's DNA as input and return the fitness value as
  //            output. The agent's DNA is an array of double values
  //            in fitness' domain. The higher the fitness the better the agent.
  //   constraint: function of dimensions (fitness.dimIn, 1) encoding the
  //               constraints on the intputs as follow: if the output value
  //               is positive or null the constraint is satisfied, else
  //               the constraint is not satisfied. The greater the value, the
  //               better the constraint is satisfied. A null argument indicates
  //               no constraint.
  // Output:
  //   that->bestFitness, bestDna and bestConstraint are updated with the best
  //   agent. The best agent is either the one respecting the constraint and
  //   having highest fitness, or if no agent respecting constraint could
  //   be found it is the one with highest fitness regardless of the constraint.
  void (*run)(
    CapyMathFun* const fitness,
    CapyMathFun* const constraint);

  // Get the current best DNA (no effect if no optimisation currently
  // running)
  // Input:
  //          dna: array of double updated with the best dna
  //      fitness: double updated with the fitness value of the dna
  //   constraint: double updated with the constraint value of the dna
  void (*getBestDna)(
    double* const dna,
    double* const fitness,
    double* const constraint);

  // Stop the current optimisation (no effect if there is no
  // optimisation currently running)
  void (*stop)(void);
} CapyDiffEvo;

// Create a CapyDiffEvo
// Output:
//   Return a CapyDiffEvo
CapyDiffEvo CapyDiffEvoCreate(void);

// Allocate memory for a new CapyDiffEvo and create it
// Output:
//   Return a CapyDiffEvo
// Exception:
//   May raise CapyExc_MallocFailed.
CapyDiffEvo* CapyDiffEvoAlloc(void);

// Free the memory used by a CapyDiffEvo* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyDiffEvo to free
void CapyDiffEvoFree(CapyDiffEvo** const that);

// Structure to pass arguments to the optimising thread
typedef struct CapyDiffEvoThreadData {

  // the CapyDiffEvo
  CapyDiffEvo* that;

  // Object performing the fitness calculation, takes the agent's DNA as
  // input and return the fitness value as output. The agent's DNA is an array
  // of double values in fitness' domain. The higher the fitness the better
  // the agent.
  CapyMathFun* fitness;

  // Constraint on the agent DNA values. If not null, takes the agent's DNA as
  // input and return the constraint satisfaction as output (positive or
  // null means satisfied).
  CapyMathFun* constraint;
} CapyDiffEvoThreadData;

#endif
