// --------------------------------- display.h --------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "display.h"

// Redraw the displayed graphics with the contents of the pixbuf and
// process/update data in the communication shared segment
// Input:
//   data: pointer to the CapyDisplay
static gboolean Tick(gpointer data) {
#if BUILD_MODE == 0
  assert(data != NULL);
#endif

  // Get the pointer to the CapyDisplay
  CapyDisplay* display = (CapyDisplay*)data;

  // Update the pixbuf with the data from the client in the shared memory
  memcpy(
    gdk_pixbuf_get_pixels(display->pixbuf),
    display->sharedPixels,
    sizeof(guchar) * display->height * display->rowstride);

  // Refresh the displayed graphics
  gtk_image_set_from_pixbuf(
    display->image,
    display->pixbuf);
  gtk_widget_queue_draw(GTK_WIDGET(display->image));

  // Update the window title
  gtk_window_set_title(
    GTK_WINDOW(display->window),
    display->com->title);

  // Process the communication with the client
  if(display->com->flagClose) gtk_window_close(GTK_WINDOW(display->window));

  // Return true to stop the callback chain
  return true;
}

// Callback function for the delete event
// Input:
//   widget: Widget receiving the event
//   event: the event
//   data: pointer to the CapyDisplay
static gboolean CbDeleteEvent(
  GtkWidget* widget,
   GdkEvent* event,
    gpointer data) {
#if BUILD_MODE == 0
  assert(data != NULL);
#endif

  // Unused arguments
  (void)widget; (void)event;

  // Get the pointer to the CapyDisplay
  CapyDisplay* display = (CapyDisplay*)data;

  // Quit the application at GTK level
  gtk_main_quit();

  // Quit the application at G level
  g_application_quit(display->gApp);

  // Return false to continue the callback chain
  return false;
}

// Callback function for the key press event
// Input:
//   widget: Widget receiving the event
//   event: the event
//   data: pointer to the CapyDisplay
static gboolean CbKeyPressEvent(
    GtkWidget* widget,
  GdkEventKey* event,
      gpointer data) {
#if BUILD_MODE == 0
  assert(data != NULL);
#endif

  // Unused arguments
  (void)widget;

  // Get the pointer to the CapyDisplay
  CapyDisplay* display = (CapyDisplay*)data;

  // If the buffer is not full
  if(display->com->idxLastKeyWrite + 1 != display->com->idxLastKeyRead) {

    // Increase the index of the last pressed key in the buffer.
    display->com->idxLastKeyWrite =
      (display->com->idxLastKeyWrite + 1) % CAPY_DISPLAY_EVT_BUFFER_SIZE;

    // Update the key pressed in the buffer
    size_t idx = display->com->idxLastKeyWrite;
    display->com->keyPressed[idx] = (CapyDisplayKeyEvt){
      .keyval = event->keyval,
      .timeMs = $(display, getTimeMs)(),
    };
  }

  // Return true to stop the callback chain
  return true;
}

// Callback function for the mouse move event
// Input:
//   widget: Widget receiving the event
//   event: the event
//   data: pointer to the CapyDisplay
static gboolean CbMouseEvent(
    GtkWidget* widget,
  GdkEventKey* event,
      gpointer data) {
#if BUILD_MODE == 0
  assert(data != NULL);
#endif

  // Unused arguments
  (void)widget;

  // Get the pointer to the CapyDisplay
  CapyDisplay* display = (CapyDisplay*)data;

  // If the event is a mouse motion
  if(event->type == GDK_MOTION_NOTIFY) {

    // Update the current mouse position
    GdkEventMotion* e = (GdkEventMotion*)event;
    display->com->mousePos.x = (CapyImgPos_t)(e->x);
    display->com->mousePos.y = (CapyImgPos_t)(e->y);
  } else if(
    event->type == GDK_BUTTON_PRESS || event->type == GDK_BUTTON_RELEASE
  ) {

    // If the buffer is not full
    if(
      (display->com->idxLastMouseWrite + 1) != display->com->idxLastMouseRead
    ) {

      // Increase the index of the last pressed key in the buffer.
      display->com->idxLastMouseWrite =
        (display->com->idxLastMouseWrite + 1) % CAPY_DISPLAY_EVT_BUFFER_SIZE;

      // Update the mouse event in the buffer
      CapyImgPos pos = $(display, getMousePos)();
      CapyImgPos posImg = $(display, getMousePosImg)();
      CapyDisplayTimeMs_t timeMs = $(display, getTimeMs)();
      size_t idx = display->com->idxLastMouseWrite;
      display->com->mouseEvt[idx] = (CapyDisplayMouseEvt){
        .type = event->type,
        .pos = pos,
        .posImg = posImg,
        .timeMs = timeMs,
      };
    }
  }

  // Return true to stop the callback chain
  return true;
}

// Callback function for the activate event
// Input:
//   gtkApp: the activated application
//   data: pointer to the CapyDisplay
static void CbActivate(
  GtkApplication* gtkApp,
         gpointer data) {
#if BUILD_MODE == 0
  assert(data != NULL);
#endif

  // Unused arguments
  (void)gtkApp;

  // Get the pointer to the CapyDisplay
  CapyDisplay* display = (CapyDisplay*)data;

  // Get the main window
  display->window = gtk_application_window_new(display->gtkApp);

  // Set the callback on the delete event
  g_signal_connect(
    display->window,
    "delete-event",
    G_CALLBACK(CbDeleteEvent),
    display);

  // Set the callback on the key press event
  g_signal_connect(
    display->window,
    "key-press-event",
    G_CALLBACK(CbKeyPressEvent),
    display);

  // Set the callback functions on the mouse events
  gtk_widget_set_events(
    display->window,
    GDK_POINTER_MOTION_MASK |
    GDK_BUTTON_PRESS_MASK |
    GDK_BUTTON_RELEASE_MASK);
  g_signal_connect(
    G_OBJECT(display->window),
    "motion-notify-event",
    G_CALLBACK(CbMouseEvent),
    display);
  g_signal_connect(
    G_OBJECT(display->window),
    "button-press-event",
    G_CALLBACK(CbMouseEvent),
    display);
  g_signal_connect(
    G_OBJECT(display->window),
    "button-release-event",
    G_CALLBACK(CbMouseEvent),
    display);

  // Create the widget to display the graphics
  display->image = GTK_IMAGE(gtk_image_new_from_pixbuf(display->pixbuf));
  gtk_container_add(
    GTK_CONTAINER(display->window),
    GTK_WIDGET(display->image));

  // Avoid window resizing
  gtk_window_set_resizable(
    GTK_WINDOW(display->window),
    false);

  // Call the redraw function every 40ms (25 times per second) to update the
  // graphics and process the key pressed
  guint timerIntervalMs = 40;
  display->timerId = g_timeout_add(
    timerIntervalMs,
    Tick,
    display);

  // Center the window on the screen
  gtk_window_set_position(
    GTK_WINDOW(display->window),
    GTK_WIN_POS_CENTER_ALWAYS);

  // CapyDisplay the window on the screen
  gtk_widget_show_all(display->window);

  // Run the application at the GTK level
  gtk_main();
}

// Show a CapyDisplay on the screen
// Exception:
//   May raise CapyExc_ForkFailed
static void Show(void) {
  methodOf(CapyDisplay);

  // Start the chronometer
  $(&(that->chrono), start)();

  // Reset the flag in the shared memory
  that->com->flagClose = false;

  // Fork the process to run the display independently
  int pid = fork();
  if(pid == -1) raiseExc(CapyExc_ForkFailed);
  else if(pid == 0) {

    // If there is a handler set, execute it
    if(that->onShow != NULL) (that->onShow)(that->onShowArg);

    // Connect the callback function on the 'activate' event
    g_signal_connect(that->gtkApp, "activate", G_CALLBACK(CbActivate), that);

    // Initialise the GTK library. Make the simplification to pass
    // manually created argc and argv.
    int argc = 1;
    char* ptrTitle = (char*)&(that->com->title);
    char** argv = &ptrTitle;
    gtk_init(&argc, &argv);

    // Update the flag to memorise the display is on
    that->com->isDisplayed = true;

    // Run the application at the G level (ignore the return status).
    // Blocking until the user closes the window or it is programmatically

    // closed with gtk_window_close()
    (void)g_application_run(that->gApp, 0, NULL);

    // Update the flag to memorise the display is off
    that->com->isDisplayed = false;

    // If there is a handler set, execute it
    if(that->onClose != NULL) (that->onClose)(that->onCloseArg);

    // Detach the shared segments
    shmdt(that->sharedPixels);
    shmdt(that->com);

    // End the child process
    exit(0);
  }
}

// Close the window of the CapyDisplay
static void Close(void) {
  methodOf(CapyDisplay);

  // Set the flag in the shared segment, this will trigger gtk_window_close()
  // at the next call of Tick()
  that->com->flagClose = true;
}

// Return the last key pressed since the last call to getKeyPressed
// Output:
//   Return the key event, or CapyDisplayKeyEvt.keyval=0 if there wasn't
//   any key pressed. See the link below for a complete list of key values.
//   https://gitlab.gnome.org/GNOME/gtk/blob/master/gdk/gdkkeysyms.h
static CapyDisplayKeyEvt GetKeyPressed(void) {
  methodOf(CapyDisplay);

  // Variable to memorise the returned value
  CapyDisplayKeyEvt ret = { .keyval = 0 };

  // If there are keys waiting in the buffer
  if(that->com->idxLastKeyRead != that->com->idxLastKeyWrite) {

    // Move the index to the next key to read.
    that->com->idxLastKeyRead =
      (that->com->idxLastKeyRead + 1) % CAPY_DISPLAY_EVT_BUFFER_SIZE;

    // Update the returned value with the key in the buffer
    ret = that->com->keyPressed[that->com->idxLastKeyRead];
  }

  // Return the last key press value
  return ret;
}

// Return the last mouse event since the last call to getMouseEvent
// Output:
//   Return the mouse event, or CapyDisplayMouseEvt.type=0 if there
//   wasn't any event.
static CapyDisplayMouseEvt GetMouseEvent(void) {
  methodOf(CapyDisplay);

  // Variable to memorise the returned value
  CapyDisplayMouseEvt ret = { .type = 0 };

  // If there are events waiting in the buffer
  if(that->com->idxLastMouseRead != that->com->idxLastMouseWrite) {

    // Move the index to the next event to read.
    that->com->idxLastMouseRead =
      (that->com->idxLastMouseRead + 1) % CAPY_DISPLAY_EVT_BUFFER_SIZE;

    // Update the returned value with the event in the buffer
    ret = that->com->mouseEvt[that->com->idxLastMouseRead];
  }

  // Return the last mouse event
  return ret;
}

// Set the RGB value of a pixel in the CapyDisplay
// Input:
//   pixel: the position of the pixel (from left to right and top to bottom)
//   color: the new color of the pixel (the alpha channel is forced to 1.0)
static void SetPixel(
     CapyImgPos const* const pixel,
  CapyColorData const* const color) {
  methodOf(CapyDisplay);
#if BUILD_MODE == 0
  assert(pixel != NULL);
  assert(color != NULL);
#endif

  // Get the index of the pixel
  size_t idx =
    (size_t)(pixel->y) * (size_t)(that->rowstride) +
    (size_t)(pixel->x) * (size_t)(that->n_channels);

  // Update the RGB values of the pixel in the shared segment. The Tick()
  // function will update the displayed widget with the new values next time
  // it is called.
  loop(i, (size_t)3) {
    that->sharedPixels[idx + i] = (guchar)lround(color->RGBA[i] * 255.0);
  }
  that->sharedPixels[idx + 3] = 255;
}

// Copy a CapyImage into a CapyDisplay
// Input:
//   img: the CapyImage to copy, (the alpha channel is forced to 1.0)
static void CopyImg(CapyImg const* const img) {
  methodOf(CapyDisplay);
#if BUILD_MODE == 0
  assert(img != NULL);
#endif

  // Loop on pixels
  #pragma omp parallel for
  loop(y, that->height) loop(x, that->width) {

    // Get the index of the pixel in the shared segment
    size_t idxPixel =
      (size_t)y * (size_t)(that->rowstride) +
      (size_t)x * (size_t)(that->n_channels);

    // Convert and copy the pixel
    CapyColorData const* color =
      $(img, getColor)(
        &(CapyImgPos){.x = (CapyImgPos_t)x, .y = (CapyImgPos_t)y});
    loop(i, (size_t)3) {
      that->sharedPixels[idxPixel + i] = (guchar)lround(color->RGBA[i] * 255.0);
    }
    that->sharedPixels[idxPixel + 3] = 255;
  }
}

// Copy a CapyImage into a CapyDisplay after scaling and translating it
// with the CapyDisplayMagnifier
// Input:
//   img: the CapyImage to copy (the alpha channel is forced to 1.0)
static void MagnifyImg(CapyImg const* const img) {
  methodOf(CapyDisplay);
#if BUILD_MODE == 0
  assert(img != NULL);
#endif

  // Loop on pixels
  #pragma omp parallel for
  loop(y, that->height) loop(x, that->width) {

    // Get the coordinates of the pixel in the image from the magnified
    // coordinates in the display
    CapyImgPos coordDisplay = {.x = (CapyImgPos_t)x, .y = (CapyImgPos_t)y};
    CapyImgPos coordImg =
      $(&(that->com->magnifier), demagnifyCoord)(&coordDisplay);

    // Variable to memorise the result color of the pixel in the display,
    // by default set to black
    CapyColorData const* color = &capyColorRGBABlack;

    // If the pixel is in the image, get the color from the image
    if($(img, isValidCoord)(&coordImg)) color = $(img, getColor)(&coordImg);

    // Set the result color in the shared segment
    size_t idxPixel =
      (size_t)y * (size_t)(that->rowstride) +
      (size_t)x * (size_t)(that->n_channels);
    loop(i, (size_t)3) {
      that->sharedPixels[idxPixel + i] = (guchar)lround(color->RGBA[i] * 255.0);
    }
    that->sharedPixels[idxPixel + 3] = 255;
  }
}

// Set the RGB value of a pixel in the CapyDisplay
// Input:
//   pixel: the position of the pixel (from left to right and top to bottom)
//   color: the new color of the pixel (the alpha channel is forced to 1.0)
static gboolean IsDisplayed(void) {
  methodOf(CapyDisplay);
  return that->com->isDisplayed;
}

// Set the window's title
// Input:
//   title: the new title
static void SetTitle(char const* const title) {
  methodOf(CapyDisplay);
  sprintf(
    that->com->title, "%.*s", CAPY_DISPLAY_MAX_LENGTH_TITLE - 1, title);
}

// Get the time spent in millisecond since the CapyDisplay is
// displayed.
static CapyDisplayTimeMs_t GetTimeMs(void) {
  methodOf(CapyDisplay);

  // Calculate the time elapsed
  $(&(that->chrono), stop)();
  CapyDisplayTimeMs_t timeMs =
    (CapyDisplayTimeMs_t)$(&(that->chrono), getElapsedTime)(
      capyChrono_millisecond);

  // Return the time
  return timeMs;
}

// Set the magnifier scale to fit entirely the image in argument in
// the display. (The position is left unmodified)
// Input:
//   img: the image to fit
static void MagnifyToFitIn(CapyImg const* const img) {
  methodOf(CapyDisplay);
  double ratios[2] = {
    ((double)(img->dims.width)) / ((double)(that->width)),
    ((double)(img->dims.height)) / ((double)(that->height))
  };
  that->com->magnifier.invScale = max(ratios, 2);
}

// Get the current mouse position in display coordinates (from left
// to right and top to bottom).
// Output:
//   Return the position
static CapyImgPos GetMousePos(void) {
  methodOf(CapyDisplay);

  // Return the last recorded position
  return that->com->mousePos;
}

// Get the current mouse position converted to image coordinates.
// Output:
//   Return the position
static CapyImgPos GetMousePosImg(void) {
  methodOf(CapyDisplay);

  // Return the last recorded position converted through the magnifier
  return $(&(that->com->magnifier), demagnifyCoord)(&(that->com->mousePos));
}

// Free the memory used by a CapyDisplay
static void Destruct(void) {
  methodOf(CapyDisplay);

  // Ensure the display is closed before destructing it
  while(that->com->isDisplayed) $(that, close)();

  // Free the mangifier
  $(&(that->com->magnifier), destruct)();

  // Detach and deallocate the shared memory segment
  shmdt(that->sharedPixels);
  shmdt(that->com);
  shmctl(that->sharedMemId, IPC_RMID, NULL);
  shmctl(that->comId, IPC_RMID, NULL);

  // Free memory
  g_object_unref(that->pixbuf);
  g_object_unref(that->gtkApp);
}

// Create a CapyDisplay
// Input:
//   dim: the dimension of the display
//   title: title displayed in the title bar of the window of the CapyDisplay
// Output:
//   Return a CapyDisplay
// Exception:
//   May raise CapyExc_MallocFailed
CapyDisplay CapyDisplayCreate(
  CapyImgDims const* const dim,
         char const* const title) {
#if BUILD_MODE == 0
  assert(dim != NULL);
  assert(title != NULL);
#endif

  // Declare the result CapyDisplay
  CapyDisplay display = {
    .width = dim->width,
    .height = dim->height,
    .onShowArg = NULL,
    .onShow = NULL,
    .onCloseArg = NULL,
    .onClose = NULL,
    .destruct = Destruct,
    .chrono = CapyChronoCreate(),
    .show = Show,
    .close = Close,
    .getKeyPressed = GetKeyPressed,
    .getMouseEvent = GetMouseEvent,
    .setPixel = SetPixel,
    .copyImg = CopyImg,
    .magnifyImg = MagnifyImg,
    .isDisplayed = IsDisplayed,
    .setTitle = SetTitle,
    .getTimeMs = GetTimeMs,
    .magnifyToFitIn = MagnifyToFitIn,
    .getMousePos = GetMousePos,
    .getMousePosImg = GetMousePosImg,
  };

  // Create the GTK application
#ifdef G_APPLICATION_DEFAULT_FLAGS
  display.gtkApp = gtk_application_new(
    NULL,
    G_APPLICATION_DEFAULT_FLAGS);
#else

  // Uses 0 instead of G_APPLICATION_FLAGS_NONE to avoid a warning message
  // due to the deprecation of that macro
  display.gtkApp = gtk_application_new(
    NULL,
    0);
#endif
  display.gApp = G_APPLICATION(display.gtkApp);

  // Create the pixbuf needed to draw the RGB array onto the widget
  gboolean has_alpha = true;
  int bits_per_sample = 8;
  display.pixbuf = gdk_pixbuf_new(
    GDK_COLORSPACE_RGB,
    has_alpha,
    bits_per_sample,
    (int)(dim->width),
    (int)(dim->height));
  display.rowstride = (CapyImgDims_t)gdk_pixbuf_get_rowstride(display.pixbuf);
  display.n_channels = gdk_pixbuf_get_n_channels(display.pixbuf);

  // Create the shared segment for the pixels array
  display.sharedMemId = shmget(
    IPC_PRIVATE,
    sizeof(guchar) * display.height * display.rowstride,
    0777 | IPC_CREAT);
  display.sharedPixels = (guchar*)shmat(display.sharedMemId, 0, 0);

  // Create the shared segment for communication
  display.comId = shmget(
    IPC_PRIVATE,
    sizeof(CapyDisplayCom),
    0777 | IPC_CREAT);
  display.com = (CapyDisplayCom*)shmat(display.comId, 0, 0);

  // Init the communication parameters
  display.com->magnifier = CapyDisplayMagnifierCreate(),
  display.com->flagClose = false;
  display.com->isDisplayed = false;
  display.com->idxLastKeyWrite = 0;
  display.com->idxLastKeyRead = 0;
  display.com->idxLastMouseWrite = 0;
  display.com->idxLastMouseRead = 0;
  display.com->mousePos.x = 0;
  display.com->mousePos.y = 0;
  memset(display.com->title, 0, CAPY_DISPLAY_MAX_LENGTH_TITLE);
  sprintf(
    display.com->title, "%.*s", CAPY_DISPLAY_MAX_LENGTH_TITLE - 1, title);

  // Return the new display
  return display;
}

// Allocate memory for new CapyDisplay
// Input:
//   dim: the dimension of the display
//   title: title displayed in the title bar of the window of the CapyDisplay
// Output:
//   Return a newly allocated CapyDisplay
// Exception:
//   May raise CapyExc_MallocFailed
CapyDisplay* CapyDisplayAlloc(
  CapyImgDims const* const dim,
         char const* const title) {
#if BUILD_MODE == 0
  assert(dim != NULL);
  assert(title != NULL);
#endif

  // Allocate memory for the CapyDisplay
  CapyDisplay* display = NULL;
  safeMalloc(display, 1);
  assert(display != NULL);

  // Create the CapyDisplay
  CapyDisplay d = CapyDisplayCreate(dim, title);
  memcpy(display, &d, sizeof(CapyDisplay));

  // Return the new display
  return display;
}

// Free the memory used by a CapyDisplay* and reset '*that' to NULL
// Input:
//   display: the CapyDisplay to free
void CapyDisplayFree(CapyDisplay** display) {
  if(display == NULL || *display == NULL) return;

  // Destruct the CapyDisplay
  $(*display, destruct)();

  // Free memory
  free(*display);
  *display = NULL;
}
