// --------------------------------- display.h --------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_DISPLAY_H
#define CAPY_DISPLAY_H
#include "externalHeaders.h"
#include "image.h"
#include "displayMagnifier.h"
#include "chrono.h"

// Description:
// Class to display graphics on screen, using GTK.

// Max length of the window's title
#define CAPY_DISPLAY_MAX_LENGTH_TITLE 1024

// Type for the time values of a CapyDisplay
typedef uint64_t CapyDisplayTimeMs_t;

// Structure to memorise a key event
typedef struct CapyDisplayKeyEvt {

  // The key value. See the link below for a complete list of key values.
  // https://gitlab.gnome.org/GNOME/gtk/blob/master/gdk/gdkkeysyms.h
  guint keyval;
  CapyPad(guint, keyval);

  // Time of the event
  CapyDisplayTimeMs_t timeMs;
} CapyDisplayKeyEvt;

// Structure to memorise a mouse event
typedef struct CapyDisplayMouseEvt {

  // The coordinates in the display of the mouse at the time of the
  // event
  CapyImgPos pos;

  // The coordinates in the image of the mouse at the time of the
  // event
  CapyImgPos posImg;

  // The type of event (GDK_BUTTON_PRESS or GDK_BUTTON_RELEASE)
  GdkEventType type;
  CapyPad(GdkEventType, type);

  // Time of the event
  CapyDisplayTimeMs_t timeMs;
} CapyDisplayMouseEvt;

// Size of the buffer for events
#define CAPY_DISPLAY_EVT_BUFFER_SIZE 256

// Structure of the shared memory segment for communication between the client
// and the display server
typedef struct CapyDisplayCom {

  // Flag to memorise the request from the client to close the CapyDisplay
  gboolean flagClose;

  // Flag to memorise if the windows is displayed
  gboolean isDisplayed;

  // Buffer to memorise the key pressed waiting to be processed by the
  // user process
  CapyDisplayKeyEvt keyPressed[CAPY_DISPLAY_EVT_BUFFER_SIZE];
  size_t idxLastKeyWrite;
  size_t idxLastKeyRead;

  // Buffer to memorise the mouse event waiting to be processed by the
  // user process
  CapyDisplayMouseEvt mouseEvt[CAPY_DISPLAY_EVT_BUFFER_SIZE];
  size_t idxLastMouseWrite;
  size_t idxLastMouseRead;

  // Window's title
  char title[CAPY_DISPLAY_MAX_LENGTH_TITLE];

  // Current mouse position in pixel coordinates of the display.
  // If the mouse moves out of the display, it is the last recorded
  // position.
  CapyImgPos mousePos;

  // Display magnifier
  CapyDisplayMagnifier magnifier;
} CapyDisplayCom;

// Display object
typedef struct CapyDisplay {

  // Dimensions of the CapyDisplay
  CapyImgDims_;

  // GTK variables to manage the window
  GtkApplication* gtkApp;
  GApplication* gApp;
  GtkWidget* window;
  GdkPixbuf* pixbuf;
  CapyImgDims_t rowstride;
  int n_channels;
  GtkImage* image;
  unsigned int timerId;

  // Variables to manage the two shared segments (one for the pixels value
  // and for the communication between the CapyDisplay and the user process).
  int sharedMemId;
  int comId;
  CapyPad(int, comId);
  guchar* sharedPixels;
  CapyDisplayCom* com;

  // Chronometer to measure time since the Display is running
  CapyChrono chrono;

  // Destructor
  void (*destruct)(void);

  // Handler called in the child process when the display is shown
  // By default NULL, onShowArg is the argument passed to onShow
  void* onShowArg;
  void (*onShow)(void*);

  // Handler called in the child process when the display is closed
  // By default NULL, onCloseArg is the argument passed to onClose
  void* onCloseArg;
  void (*onClose)(void*);

  // Show a CapyDisplay on the screen. The display is centered on the screen.
  // Exception:
  //   May raise CapyExc_ForkFailed
  void (*show)(void);

  // Close the window of the CapyDisplay
  void (*close)(void);

  // Return the last key pressed since the last call to getKeyPressed
  // Output:
  //   Return the key event, or CapyDisplayKeyEvt.keyval=0 if there wasn't
  //   any key pressed. See the link below for a complete list of key values.
  //   https://gitlab.gnome.org/GNOME/gtk/blob/master/gdk/gdkkeysyms.h
  CapyDisplayKeyEvt (*getKeyPressed)(void);

  // Return the last mouse event since the last call to getMouseEvent
  // Output:
  //   Return the mouse event, or CapyDisplayMouseEvt.type=0 if there
  //   wasn't any event.
  CapyDisplayMouseEvt (*getMouseEvent)(void);

  // Set the RGB value of a pixel in the CapyDisplay
  // Input:
  //   pixel: the position of the pixel (from left to right and top to bottom)
  //   color: the new color of the pixel (the alpha channel is forced to 1.0)
  void (*setPixel)(
       CapyImgPos const* const pixel,
    CapyColorData const* const color);

  // Copy a CapyImage into a CapyDisplay
  // Input:
  //   img: the CapyImage to copy (the alpha channel is forced to 1.0)
  void (*copyImg)(CapyImg const* const img);

  // Copy a CapyImage into a CapyDisplay after scaling and translating it
  // with the CapyDisplayMagnifier
  // Input:
  //   img: the CapyImage to copy (the alpha channel is forced to 1.0)
  void (*magnifyImg)(CapyImg const* const img);

  // Check if a CapyDisplay is currently displayed
  // Output:
  //   Return true if the CapyDisplay is displayed
  gboolean (*isDisplayed)(void);

  // Set the window's title
  // Input:
  //   title: the new title
  void (*setTitle)(char const* const title);

  // Get the time spent in millisecond since the CapyDisplay is
  // displayed.
  CapyDisplayTimeMs_t (*getTimeMs)(void);

  // Set the magnifier scale to fit entirely the image in argument in
  // the display. (The position is left unmodified)
  // Input:
  //   img: the image to fit
  void (*magnifyToFitIn)(CapyImg const* const img);

  // Get the current mouse position in display coordinates (from left
  // to right and top to bottom).
  // Output:
  //   Return the position
  CapyImgPos (*getMousePos)(void);

  // Get the current mouse position converted to image coordinates.
  // Output:
  //   Return the position
  CapyImgPos (*getMousePosImg)(void);
} CapyDisplay;

// Create a CapyDisplay
// Input:
//   dim: the dimension of the display
//   title: title displayed in the title bar of the window of the CapyDisplay
// Output:
//   Return a CapyDisplay
// Exception:
//   May raise CapyExc_MallocFailed
CapyDisplay CapyDisplayCreate(
  CapyImgDims const* const dim,
         char const* const title);

// Allocate memory for new CapyDisplay
// Input:
//   dim: the dimension of the display
//   title: title displayed in the title bar of the window of the CapyDisplay
// Output:
//   Return a newly allocated CapyDisplay
// Exception:
//   May raise CapyExc_MallocFailed
CapyDisplay* CapyDisplayAlloc(
  CapyImgDims const* const dim,
         char const* const title);

// Free the memory used by a CapyDisplay* and reset '*that' to NULL
// Input:
//   display: the CapyDisplay to free
void CapyDisplayFree(CapyDisplay** display);
#endif
