// --------------------------------- display.h --------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "displayMagnifier.h"

// Reset the magnifier (position to origin, scale to 1.0)
static void Reset(void) {
  methodOf(CapyDisplayMagnifier);

  // Reset the magnifier
  that->pos.x = 0.0;
  that->pos.y = 0.0;
  that->invScale = 1.0;
}

// Move the magnifier to a position
// Input:
//   pos: The position where to move the magnifier
static void MoveTo(CapyVec const* const pos) {
  methodOf(CapyDisplayMagnifier);
#if BUILD_MODE == 0
  assert(pos != NULL);
#endif

  // Update the postion
  loop(i, 2) that->pos.vals[i] = pos->vals[i];
}

// Translate the magnifier by a vector
// Input:
//   v: The vector to add to the current position of the magnifier
static void Translate(CapyVec const* const v) {
  methodOf(CapyDisplayMagnifier);
#if BUILD_MODE == 0
  assert(v != NULL);
#endif

  // Update the postion
  loop(i, 2) that->pos.vals[i] += v->vals[i];
}

// Scale up the magnification at a given speed
// Input:
//   coeff: the speed by which the scale is scaled up (equivalent to
//          scale *= 1.0 + coeff)
//   fixedPos: if not null, the magnifier is translated such as the fixed
//             position appears at the same location on the display before
//             and after scaling. (fixedPos in display coordinates)
static void ScaleUp(
          double const coeff,
  CapyVec const* const fixedPos) {
  methodOf(CapyDisplayMagnifier);
#if BUILD_MODE == 0
  assert(coeff >= 0.0);
#endif

  // Update the scale
  that->invScale /= (1.0 + coeff);

  // If there is a fixed point
  if(fixedPos != NULL) {

    // Update the position of the magnifier
    double c = coeff * that->invScale;
    loop(i, 2) that->pos.vals[i] -= c * fixedPos->vals[i];
  }
}

// Scale down the magnification at a given speed
// Input:
//   coeff: the speed by which the scale is scaled down (equivalent to
//          scale *= 1.0 - coeff)
//   fixedPos: if not null, the magnifier is translated such as the fixed
//             position appears at the same location on the display before
//             and after scaling. (fixedPos in display coordinates)
static void ScaleDown(
          double const coeff,
  CapyVec const* const fixedPos) {
  methodOf(CapyDisplayMagnifier);
#if BUILD_MODE == 0
  assert(coeff >= 0.0);
#endif

  // Update the scale
  that->invScale /= (1.0 - coeff);

  // If there is a fixed point
  if(fixedPos != NULL) {

    // Update the position of the magnifier
    double c = coeff * that->invScale;
    loop(i, 2) that->pos.vals[i] += c * fixedPos->vals[i];
  }
}

// Convert coordinates from magnified to demagnified
// Input:
//   coord: the coordinates to convert
// Output:
//   Return the converted coordinates. Don't care about overflow if the
//   converted coordinates become negative, use appropriately. The conversion
//   formula is
//   coordImg = coordDisplay / scale - translate
static CapyImgPos DemagnifyCoord(CapyImgPos const* const coord) {
  methodOf(CapyDisplayMagnifier);
#if BUILD_MODE == 0
  assert(coord != NULL);
#endif
  return (CapyImgPos){
    .x = (CapyImgPos_t)((double)(coord->x) * that->invScale - that->pos.x),
    .y = (CapyImgPos_t)((double)(coord->y) * that->invScale - that->pos.y)
  };
}

// Free the memory used by a CapyDisplayMagnifier
static void Destruct(void) {
  return;
}

// Create a CapyDisplayMagnifier
// Output:
//   Return a CapyDisplayMagnifier
CapyDisplayMagnifier CapyDisplayMagnifierCreate(void) {

  // Create the instance
  CapyDisplayMagnifier that = {
    .pos = {.dim = 2, .x = 0.0, .y = 0.0},
    .invScale = 1.0,
    .destruct = Destruct,
    .reset = Reset,
    .moveTo = MoveTo,
    .translate = Translate,
    .demagnifyCoord = DemagnifyCoord,
    .scaleUp = ScaleUp,
    .scaleDown = ScaleDown,
  };

  // Return the instance
  return that;
}

// Allocate memory for new CapyDisplayMagnifier
// Output:
//   Return a newly allocated CapyDisplayMagnifier
// Exception:
//   May raise CapyExc_MallocFailed
CapyDisplayMagnifier* CapyDisplayMagnifierAlloc(void) {

  // Allocate memory for the instance
  CapyDisplayMagnifier* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;

  // Create the instance
  CapyDisplayMagnifier m = CapyDisplayMagnifierCreate();
  memcpy(that, &m, sizeof(CapyDisplayMagnifier));

  // Return the instance
  return that;
}

// Free the memory used by a CapyDisplayMagnifier* and reset '*that' to NULL
// Input:
//   display: the CapyDisplayMagnifier to free
void CapyDisplayMagnifierFree(CapyDisplayMagnifier** that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
