// ---------------------------- displayMagnifier.h ---------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_DISPLAY_MAGNIFIER_H
#define CAPY_DISPLAY_MAGNIFIER_H
#include "externalHeaders.h"
#include "image.h"

// Description:
// Image magnifier class.

// Display magnifier object
typedef struct CapyDisplayMagnifier {

  // Position
  CapyVecDef(2, x, y) pos;

  // Scale, memorise the inverse of the scale to avoid using division in the
  // conversion method (supposedly more often used than the scale update
  // methods)
  double invScale;

  // Destructor
  void (*destruct)(void);

  // Reset the magnifier (position to origin, scale to 1.0)
  void (* reset)(void);

  // Move the magnifier to a position
  // Input:
  //   pos: The position where to move the magnifier
  void (* moveTo)(CapyVec const* const pos);

  // Translate the magnifier by a vector
  // Input:
  //   v: The vector to add to the current position of the magnifier
  void (* translate)(CapyVec const* const v);

  // Scale up the magnification at a given speed
  // Input:
  //   coeff: the speed by which the scale is scaled up (equivalent to
  //          scale *= 1.0 + coeff)
  //   fixedPos: if not null, the magnifier is translated such as the fixed
  //             position appears at the same location on the display before
  //             and after scaling. (fixedPos in display coordinates)
  void (*scaleUp)(
            double const coeff,
    CapyVec const* const fixedPos);

  // Scale down the magnification at a given speed
  // Input:
  //   coeff: the speed by which the scale is scaled down (equivalent to
  //          scale *= 1.0 - coeff)
  //   fixedPos: if not null, the magnifier is translated such as the fixed
  //             position appears at the same location on the display before
  //             and after scaling. (fixedPos in display coordinates)
  void (*scaleDown)(
            double const coeff,
    CapyVec const* const fixedPos);

  // Convert coordinates from magnified to demagnified
  // Input:
  //   coord: the coordinates to convert
  // Output:
  //   Return the converted coordinates. Don't care about overflow if the
  //   converted coordinates become negative, use appropriately. The conversion
  //   formula is
  //   coordImg = coordDisplay / scale - translate
  CapyImgPos (*demagnifyCoord)(CapyImgPos const* const coord);
} CapyDisplayMagnifier;

// Create a CapyDisplayMagnifier
// Output:
//   Return a CapyDisplayMagnifier
CapyDisplayMagnifier CapyDisplayMagnifierCreate(void);

// Allocate memory for new CapyDisplayMagnifier
// Output:
//   Return a newly allocated CapyDisplayMagnifier
// Exception:
//   May raise CapyExc_MallocFailed
CapyDisplayMagnifier* CapyDisplayMagnifierAlloc(void);

// Free the memory used by a CapyDisplayMagnifier* and reset '*that' to NULL
// Input:
//   display: the CapyDisplayMagnifier to free
void CapyDisplayMagnifierFree(CapyDisplayMagnifier** that);
#endif
