// ----------------------------- distribution.h ----------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_DISTRIBUTION_H
#define CAPY_DISTRIBUTION_H
#include "externalHeaders.h"
#include "cext.h"
#include "array.h"

// Description:
// Statistic distribution classes.

// Definition of an event for a distribution (to abstract from continuous
// and discrete)
// Vector to identify a continuous event
// CapyVec vals;
//
// Pointer toward a user data structure to identify a discrete event
// void* ptr;
//
// Id to identify a discrete event
// size_t id;
#define CapyDistEvtDef struct { \
  CapyVec vec;                  \
  void* ptr;                    \
  size_t id;                    \
}

// Event for a distribution
typedef CapyDistEvtDef CapyDistEvt;

// Create a CapyDistEvt
// Output:
//   Return a CapyDistEvt.
CapyDistEvt CapyDistEvtCreate(void);

// Type of distributions
typedef enum CapyDistType {
  capyDistributionType_continuous,
  capyDistributionType_normal,
  capyDistributionType_discrete,
} CapyDistType;

// Definition of a CapyDist class (virtual parent class for all
// the distribution)
//
// Type of the distribution
// CapyDistType type;
//
// Get the probability of a given event.
// Input:
//    evt: the event
// Output:
//   Return the probability of the event
// double (*getProbability)(CapyDistEvt const* const evt);
//
// Get the surprise of a given event.
// Input:
//    evt: the event
// Output:
//   Return the surprise of the event (h(e) = log(1/p(e)). The higher the
//   surprise the less probable is the event.
// double (*getSurprise)(CapyDistEvt const* const evt);
//
// Get the entropy of the distribution.
// Output:
//   Return the entropy (average of the surprise). The higher the entropy the
//   more uncertain the outcome of drawing a random sample.
// double (*getEntropy)(void);
//
// Check if an event is within the most probable up to a given threshold
// Input:
//         evt: the event to check
//   threshold: the threshold
// Output:
//   Return true if the event is in the most probable events up to the
//   threshold.
// bool (*isEvtInMostProbable)(
//   CapyDistEvt const* const evt,
//               double const threshold);
#define CapyDistDef struct {                               \
  CapyDistType type;                                       \
  CapyPad(CapyDistType, 0);                                \
  void (*destruct)(void);                                  \
  double (*getProbability)(CapyDistEvt const* const evt);  \
  double (*getSurprise)(CapyDistEvt const* const evt);     \
  double (*getEntropy)(void);                              \
  bool (*isEvtInMostProbable)(                             \
    CapyDistEvt const* const evt,                          \
                double const threshold);                   \
}

// CapyDist class
typedef CapyDistDef CapyDist;

// Create a CapyDist
// Input:
//   type: type of ditribution
// Output:
//   Return a CapyDist
CapyDist CapyDistCreate(CapyDistType const type);

// Definition of a CapyDistContinuous class (probability
// distribution for continuous random variables)
//
// dimEvt: the number of dimension of an event
// range: the range of possible values for each dimension of an event
#define CapyDistContinuousDef struct { \
  CapyDistDef;                         \
  void (*destructCapyDist)(void);      \
  size_t dimEvt;                       \
  CapyRangeDouble* range;              \
}

// CapyDistContinuous class
typedef CapyDistContinuousDef CapyDistContinuous;

// Create a CapyDistContinuous
// Input:
//   dimEvt: the dimension of the random variable describing an event
// Output:
//   Return a CapyDistContinuous
CapyDistContinuous CapyDistContinuousCreate(size_t const dimEvt);

// CapyDistNormal
typedef struct CapyDistNormal {
  CapyDistContinuousDef;

  // Mean and standard deviation (aka sigma) for each dimension
  // The event variable is considered isotropic (dimensions are
  // uncorrelated)
  double* mean;
  double* stdDev;

  // Destructor
  void (*destructCapyDistContinuous)(void);

  // Get the derivative of the probability of a given event along a given
  // axis.
  // Input:
  //      evt: the event
  //    iAxis: the derivative axis
  // Output:
  //   Return the derivative of the probability of the event
  double (*getProbabilityDerivative)(
    CapyDistEvt const* const evt,
                size_t const iAxis);
} CapyDistNormal;

// Create a CapyDistNormal
// Input:
//   dimEvent: the dimension of the random variable describing an event
//       mean: the means in each dimension of the random variable
//     stdDev: the standard deviations in each dimension of the random
//             variable
// Output:
//   Return a CapyDistNormal
CapyDistNormal CapyDistNormalCreate(
         size_t const dimEvent,
  double const* const mean,
  double const* const stdDev);

// Allocate memory for a new CapyDistNormal and create it
// Input:
//   dimEvent: the dimension of the random variable describing an event
//       mean: the means in each dimension of the random variable
//     stdDev: the standard deviations in each dimension of the random
//             variable
// Output:
//   Return a CapyDistNormal
CapyDistNormal* CapyDistNormalAlloc(
         size_t const dimEvent,
  double const* const mean,
  double const* const stdDev);

// Free the memory used by a CapyDistNormal
// Input:
//   that: the CapyDistNormal to free
void CapyDistNormalFree(CapyDistNormal** const that);

// Event for a discrete distribution
typedef struct CapyDistDiscreteOccurence {

  // Probability of the event
  double prob;

  // Event definition
  CapyDistEvt evt;
} CapyDistDiscreteOccurence;

// Array of CapyDistDiscreteOccurence
CapyDecArray(CapyDistDiscreteOccs, CapyDistDiscreteOccurence)

// Discrete distribution (collection of events with each a given probability)
typedef struct CapyDistDiscrete {

  // Parent class
  CapyDistDef;

  // Occurences defining the distribution
  CapyDistDiscreteOccs* occ;

  // Destructor
  void (*destructCapyDist)(void);
} CapyDistDiscrete;

// Create a CapyDistDiscrete
// Input:
//   nbEvt: the number of events in the distribution
// Output:
//   Return a CapyDistDiscrete
CapyDistDiscrete CapyDistDiscreteCreate(size_t const nbEvt);

// Allocate memory for a CapyDistDiscrete and create a CapyDistDiscrete
// Input:
//   nbEvt: the number of events in the distribution
// Output:
//   Return a CapyDistDiscrete
CapyDistDiscrete* CapyDistDiscreteAlloc(size_t const nbEvt);

// Free the memory used by a CapyDistDiscrete
// Input:
//   that: the CapyDistDiscrete to free
void CapyDistDiscreteFree(CapyDistDiscrete** const that);

// Get the cross entropy of two discrete distributions
// Input:
//   distA: the first distribution
//   distB: the second distribution
// Output:
//   Return the cross entropy of distA relative to distB. It is higher or
//   equal to the entropy of distA, and increase with the discrepancy between
//   the probabilities of the two distribtions.
double CapyDistDiscreteGetCrossEntropy(
  CapyDistDiscrete const* const distA,
  CapyDistDiscrete const* const distB);

// Get the KL divergence of two discrete distributions
// Input:
//   distA: the first distribution
//   distB: the second distribution
// Output:
//   Return the KL divergence of distA relative to distB (equals to cross
//   entropy of (distA, distB) minus entropy of distA. If the distribution are
//   the same it returns 0.0. The more they diverge the higher the returned
//   value.
double CapyDistDiscreteGetKLDivergence(
  CapyDistDiscrete const* const distA,
  CapyDistDiscrete const* const distB);
#endif
