// ---------------------------------- elo.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_ELO_H
#define CAPY_ELO_H
#include "externalHeaders.h"
#include "cext.h"

// Description:
// ELO ranking class.

// Default K coefficient
#define CAPY_ELO_K 16.0

// Type of results
typedef enum CapyEloResult {
  capyElo_loose = 0,
  capyElo_tie = 1,
  capyElo_win = 2,
} CapyEloResult;

// Elo object
typedef struct CapyElo {

  // K coefficient
  double k;

  // Destructor
  void (*destruct)(void);

  // Get the Elo delta given two Elos and a result. This delta is to be
  // added to the first Elo and substracted to the second.
  // Input:
  //   eloA: first Elo
  //   result: result of eloA against eloB (e.g., if result is 'win' it
  //           means that eloA has won over eloB)
  //   eloB: second Elo
  // Output:
  //   Return the Elo delta to be added to eloA and substracted to eloB
  double (*get)(
           double const eloA,
    CapyEloResult const result,
           double const eloB);
} CapyElo;

// Create a CapyElo
// Output:
//   Return a CapyElo
CapyElo CapyEloCreate(void);

// Allocate memory for a new CapyElo and create it
// Output:
//   Return a CapyElo
// Exception:
//   May raise CapyExc_MallocFailed.
CapyElo* CapyEloAlloc(void);

// Free the memory used by a CapyElo* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyElo to free
void CapyEloFree(CapyElo** const that);
#endif
