// ------------------------------ strDecorator.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "feistelCipher.h"

// Dummy round function for the parent class CapyFeistelRoundFun.
// Simply XOR the output with key (repeated as necessary)
// Input:
//        key: a null terminated string, the key to be used by the round
//             function
//      block: the block of data to which apply the round function
//   sizeData: the size in byte of the block of data
//     output: the adress in memory where to write the output of the round
//             function (must be already allocated and have same size as
//             'block')
static void DummyRoundFun(
    CapyArrChar const* const key,
    CapyArrChar const* const block,
          CapyArrChar* const output) {
#if BUILD_MODE == 0
  assert(key != NULL);
  assert(block != NULL);
  assert(output != NULL);
  assert(key->size > 0);
  assert(block->size > 0);
  assert(block->size == output->size);
#endif

  // Set the output
  loop(i, block->size) {
    output->data[i] = block->data[i] ^ key->data[i % key->size];
  }
}

// Free the memory used by a CapyFeistelRoundFun
static void FeistelRoundFunDestruct(void) {
  return;
}

// Create a CapyFeistelRoundFun
// Output:
//   Return a CapyFeistelRoundFun
CapyFeistelRoundFun CapyFeistelRoundFunCreate(void) {
  return (CapyFeistelRoundFun){
    .destruct = FeistelRoundFunDestruct,
    .run = DummyRoundFun
  };
}

// Set the initialisation vector for CBC and CTR mode
// Input:
//   initVector, the initialisation vector
// Exceptions:
//   May raise CapyExc_MallocFailed
static void SetInitVector(CipherData_t const* const initVector) {
  methodOf(CapyFeistelCipher);
#if BUILD_MODE == 0
  assert(initVector != NULL);
#endif

  // If the initialisation vector already exists, free it
  CapyArrCharFree(&(that->initVector));

  // Clone the initialisation vector
  that->initVector = $(initVector, clone)();
}

// Common function to cipher/decipher
// Input:
//   roundFun: the round function used to cipher
//   iterKeys: iterator on the keys used to cipher
//       data: the data to cipher
//   sizeData: the size in byte of the data to cipher
// Output:
//   Return a newly allocated array of 'sizeData' bytes containing the
//   ciphered data.
// Exceptions:
//   May raise CapyExc_MallocFailed
static CipherData_t* Run(
      CapyFeistelRoundFun* const roundFun,
  CapyListArrCharIterator* const iterKeys,
       CipherData_t const* const data) {
  struct { CipherData_t* result; CipherData_t* output; } volatiles = {
    .result = NULL,
    .output = NULL
  };
  try {

    // Get the half-size of the data, taking care of odd case
    size_t const halfSizeData = data->size / 2 + (data->size & (size_t)1);

    // Allocate memory for the result, use the double of the half-size
    // to ensure its size is even, and set the last byte to 0 to avoid
    // it being uninitialised in case sizeData is odd
    volatiles.result = CapyArrCharAlloc(halfSizeData * 2);
    volatiles.result->data[halfSizeData * 2 - 1] = 0;

    // Copy the initial data in the result
    memcpy(volatiles.result->data, data->data, data->size);

    // Allocate memory for the output of the round function
    volatiles.output = CapyArrCharAlloc(halfSizeData);

    // Loop on the keys
    forEach(key, *iterKeys) {

      // Apply the round function to the second half of the ciphered data
      CapyArrChar secondHalf = CapyArrCharCreate(0);
      secondHalf.data = volatiles.result->data + halfSizeData;
      secondHalf.size = halfSizeData;
      $(roundFun, run)(
        &key,
        &secondHalf,
        volatiles.output);

      // Apply the xor operation on the first half and the output of the round
      // function
      loop(i, halfSizeData) {
        volatiles.output->data[i] ^= volatiles.result->data[i];
      }

      // If we are not on the last iteration
      if($(iterKeys, isLast)() == false) {

        // Copy the second half to the first half
        memcpy(
          volatiles.result->data,
          volatiles.result->data + halfSizeData,
          halfSizeData);

        // Copy the xor-ed output to the second half
        memcpy(
          volatiles.result->data + halfSizeData,
          volatiles.output->data,
          halfSizeData);

      // Else, we are on the last iteration
      } else {

        // Copy the xor-ed output to the first half (the second half
        // stays as it is)
        memcpy(
          volatiles.result->data,
          volatiles.output->data,
          halfSizeData);
      }
    }
  } catchDefault {

    // Free memory
    CapyArrCharFree(&(volatiles.result));
  } endCatch;

  // Free memory
  CapyArrCharFree(&(volatiles.output));

  // Forward the eventual exception
  CapyForwardExc();

  // Return the result
  return volatiles.result;
}

// Cipher function for ECB mode
// Input:
//   roundFun: the round function used to cipher
//       keys: the keys used to cipher, a list of null terminated strings
//       data: the data to cipher
//   sizeData: the size in byte of the data to cipher
// Output:
//   Return a newly allocated array of 'sizeData' bytes containing the
//   ciphered data.
// Exceptions:
//   May raise CapyExc_MallocFailed
static CipherData_t* CipherECB(
    CapyFeistelRoundFun* const roundFun,
     CipherKeys_t const* const keys,
     CipherData_t const* const data) {
#if BUILD_MODE == 0
  assert(roundFun != NULL);
  assert(keys != NULL);
  assert(data != NULL);
  assert(data->size != 0);
#endif

  // Create the iterator, for ciphering it is a forward iterator on the keys
  CapyListArrCharIterator iterKeys = CapyListArrCharIteratorCreate(
    (CapyListArrChar*)keys,
    capyListIteratorType_forward);

  // Variable to memorise the result
  struct { CipherData_t* cipheredData; } volatiles = { .cipheredData = NULL };
  try {

    // Run the Feistel scheme with the forward iterator
    volatiles.cipheredData = Run(roundFun, &iterKeys, data);
  } endCatch;

  // Free memory
  $(&iterKeys, destruct)();

  // Forward the eventual exception
  CapyForwardExc();

  // Return the ciphered data
  return volatiles.cipheredData;
}

// Decipher function for ECB mode
// Input:
//   roundFun: the round function used to decipher
//       keys: the keys used to decipher, a list of null terminated strings
//       data: the data to decipher
//   sizeData: the size in byte of the data to decipher
// Output:
//   Return a newly allocated array of 'sizeData' bytes containing the
//   deciphered data.
// Exceptions:
//   May raise CapyExc_MallocFailed
static CipherData_t* DecipherECB(
    CapyFeistelRoundFun* const roundFun,
     CipherKeys_t const* const keys,
     CipherData_t const* const data) {
#if BUILD_MODE == 0
  assert(roundFun != NULL);
  assert(keys != NULL);
  assert(data != NULL);
  assert(data->size != 0);
#endif

  // Create the iterator, for deciphering it is a backward iterator on the keys
  CapyListArrCharIterator iterKeys = CapyListArrCharIteratorCreate(
    (CapyListArrChar*)keys, capyListIteratorType_backward);

  // Variable to memorise the result
  struct { CipherData_t* decipheredData; } volatiles =
    { .decipheredData = NULL };
  try {

    // Run the Feistel scheme with the forward iterator
    volatiles.decipheredData = Run(roundFun, &iterKeys, data);
  } endCatch;

  // Free memory
  $(&iterKeys, destruct)();

  // Forward the eventual exception
  CapyForwardExc();

  // Return the ciphered data
  return volatiles.decipheredData;
}

// Cipher function for CBC mode
// Input:
//   roundFun: the round function used to cipher
//       keys: the keys used to cipher, a list of null terminated strings
//       data: the data to cipher
//   sizeData: the size in byte of the data to cipher
// Output:
//   Return a newly allocated array of 'sizeData' bytes containing the
//   ciphered data.
// Exceptions:
//   May raise CapyExc_MallocFailed
static CipherData_t* CipherCBC(
    CapyFeistelRoundFun* const roundFun,
     CipherKeys_t const* const keys,
     CipherData_t const* const data) {
  methodOf(CapyFeistelCipher);
#if BUILD_MODE == 0
  assert(roundFun != NULL);
  assert(keys != NULL);
  assert(data != NULL);
  assert(data->size != 0);
  assert(that->initVector != NULL);
#endif

  // Create the iterator, for ciphering it is a forward iterator on the keys
  CapyListArrCharIterator iterKeys = CapyListArrCharIteratorCreate(
    (CapyListArrChar*)keys, capyListIteratorType_forward);

  // Variable to memorise the result
  struct {
    CipherData_t* cipheredData;
    CipherData_t xoredData;
  } volatiles = {
    .cipheredData = NULL,
    .xoredData = CapyArrCharCreate(data->size),
  };
  try {

    // XOR the input data with the initialisation vector
    loop(i, data->size) {
      volatiles.xoredData.data[i] =
        data->data[i] ^ that->initVector->data[i % that->initVector->size];
    }

    // Run the Feistel scheme with the forward iterator
    volatiles.cipheredData = Run(roundFun, &iterKeys, &(volatiles.xoredData));

    // Replace the initialisation vector with the ciphered data
    $(that, setInitVector)(volatiles.cipheredData);
  } endCatch;

  // Free memory
  $(&iterKeys, destruct)();
  $(&(volatiles.xoredData), destruct)();

  // Forward the eventual exception
  CapyForwardExc();

  // Return the ciphered data
  return volatiles.cipheredData;
}

// Decipher function for CBC mode
// Input:
//   roundFun: the round function used to decipher
//       keys: the keys used to decipher, a list of null terminated strings
//       data: the data to decipher
//   sizeData: the size in byte of the data to decipher
// Output:
//   Return a newly allocated array of 'sizeData' bytes containing the
//   deciphered data.
// Exceptions:
//   May raise CapyExc_MallocFailed
static CipherData_t* DecipherCBC(
    CapyFeistelRoundFun* const roundFun,
     CipherKeys_t const* const keys,
     CipherData_t const* const data) {
  methodOf(CapyFeistelCipher);
#if BUILD_MODE == 0
  assert(roundFun != NULL);
  assert(keys != NULL);
  assert(data != NULL);
  assert(data->size != 0);
  assert(that->initVector != NULL);
#endif

  // Create the iterator, for deciphering it is a backward iterator on the keys
  CapyListArrCharIterator iterKeys = CapyListArrCharIteratorCreate(
    (CapyListArrChar*)keys, capyListIteratorType_backward);

  // Variable to memorise the result
  struct { CipherData_t* decipheredData; } volatiles =
    { .decipheredData = NULL };
  try {

    // Run the Feistel scheme with the forward iterator
    volatiles.decipheredData = Run(roundFun, &iterKeys, data);

    // XOR the result of the deciphering with the initialisation vector
    loop(i, volatiles.decipheredData->size) {
      volatiles.decipheredData->data[i] ^=
        that->initVector->data[i % that->initVector->size];
    }

    // Update the initialisation vector with the input data
    $(that, setInitVector)(data);
  } endCatch;

  // Free memory
  $(&iterKeys, destruct)();

  // Forward the eventual exception
  CapyForwardExc();

  // Return the ciphered data
  return volatiles.decipheredData;
}

// Cipher function for CTR mode
// Input:
//   roundFun: the round function used to cipher
//       keys: the keys used to cipher, a list of null terminated strings
//       data: the data to cipher
//   sizeData: the size in byte of the data to cipher
// Output:
//   Return a newly allocated array of 'sizeData' bytes containing the
//   ciphered data.
// Exceptions:
//   May raise CapyExc_MallocFailed
static CipherData_t* CipherCTR(
    CapyFeistelRoundFun* const roundFun,
     CipherKeys_t const* const keys,
     CipherData_t const* const data) {
  methodOf(CapyFeistelCipher);
#if BUILD_MODE == 0
  assert(roundFun != NULL);
  assert(keys != NULL);
  assert(data != NULL);
  assert(data->size != 0);
  assert(that->initVector != NULL);
#endif

  // Create the iterator, for ciphering it is a forward iterator on the keys
  CapyListArrCharIterator iterKeys = CapyListArrCharIteratorCreate(
    (CapyListArrChar*)keys,
    capyListIteratorType_forward);

  // Variables to memorise the result
  struct {
    CipherData_t* cipheredNonceCounter;
    CipherData_t* cipheredData;
    CipherData_t nonceCounter;
  } volatiles = {
    .cipheredNonceCounter = NULL,
    .cipheredData = NULL,
    .nonceCounter = CapyArrCharCreate(
      that->initVector->size + sizeof(CipherCounter_t)),
  };
  try {

    // Create the concatenation of the initialisation vector and the counter
    memcpy(
      volatiles.nonceCounter.data,
      that->initVector->data,
      that->initVector->size);
    memcpy(
      volatiles.nonceCounter.data + that->initVector->size,
      &(that->counter),
      sizeof(CipherCounter_t));

    // Run the Feistel scheme with the forward iterator on the concatenation
    // of the initialisation vector and counter
    volatiles.cipheredNonceCounter =
      Run(roundFun, &iterKeys, &(volatiles.nonceCounter));

    // XOR the result of the ciphering with the input data
    volatiles.cipheredData = CapyArrCharAlloc(data->size);
    loop(i, data->size) {
      size_t j = i % volatiles.cipheredNonceCounter->size;
      volatiles.cipheredData->data[i] =
        data->data[i] ^ volatiles.cipheredNonceCounter->data[j];
    }

    // Increment the counter
    ++(that->counter);
  } endCatch;

  // Free memory
  $(&iterKeys, destruct)();
  $(&(volatiles.nonceCounter), destruct)();
  CapyArrCharFree(&(volatiles.cipheredNonceCounter));

  // Forward the eventual exception
  CapyForwardExc();

  // Return the ciphered data
  return volatiles.cipheredData;
}

// Decipher function for CTR mode
// Input:
//   roundFun: the round function used to decipher
//       keys: the keys used to decipher, a list of null terminated strings
//       data: the data to decipher
//   sizeData: the size in byte of the data to decipher
// Output:
//   Return a newly allocated array of 'sizeData' bytes containing the
//   deciphered data.
// Exceptions:
//   May raise CapyExc_MallocFailed
static CipherData_t* DecipherCTR(
    CapyFeistelRoundFun* const roundFun,
     CipherKeys_t const* const keys,
     CipherData_t const* const data) {
  methodOf(CapyFeistelCipher);
#if BUILD_MODE == 0
  assert(roundFun != NULL);
  assert(keys != NULL);
  assert(data != NULL);
  assert(data->size != 0);
  assert(that->initVector != NULL);
#endif

  // Create the iterator, for CTR deciphering it is a forward iterator
  // on the keys
  CapyListArrCharIterator iterKeys = CapyListArrCharIteratorCreate(
    (CapyListArrChar*)keys,
    capyListIteratorType_forward);

  // Variables to memorise the result
  struct {
    CipherData_t* cipheredNonceCounter;
    CipherData_t* decipheredData;
    CipherData_t* nonceCounter;
  } volatiles = {
    .cipheredNonceCounter = NULL,
    .decipheredData = NULL,
    .nonceCounter = NULL
  };
  try {

    // Create the concatenation of the initialisation vector and the counter
    volatiles.nonceCounter =
      CapyArrCharAlloc(that->initVector->size + sizeof(CipherCounter_t));
    memcpy(
      volatiles.nonceCounter->data,
      that->initVector->data,
      that->initVector->size);
    memcpy(
      volatiles.nonceCounter->data + that->initVector->size,
      &(that->counter),
      sizeof(CipherCounter_t));

    // Run the Feistel scheme with the forward iterator on the concatenation
    // of the initialisation vector and counter
    volatiles.cipheredNonceCounter =
      Run(roundFun, &iterKeys, volatiles.nonceCounter);

    // XOR the result of the ciphering with the input data
    volatiles.decipheredData = CapyArrCharAlloc(data->size);
    loop(i, data->size) {
      size_t j = i % volatiles.cipheredNonceCounter->size;
      volatiles.decipheredData->data[i] =
        data->data[i] ^ volatiles.cipheredNonceCounter->data[j];
    }

    // Increment the counter
    ++(that->counter);
  } endCatch;

  // Free memory
  $(&iterKeys, destruct)();
  CapyArrCharFree(&(volatiles.nonceCounter));
  CapyArrCharFree(&(volatiles.cipheredNonceCounter));

  // Forward the eventual exception
  CapyForwardExc();

  // Return the deciphered data
  return volatiles.decipheredData;
}

// Free the memory used by a CapyFeistelCipher
static void Destruct(void) {
  methodOf(CapyFeistelCipher);
  CapyArrCharFree(&(that->initVector));
}

// Create a CapyFeistelCipher
// Input:
//   opMode: operation mode
// Output:
//   Return a CapyFeistelCipher
// Exception:
//   May raise CapyExc_MallocFailed, CapyExc_UndefinedExecution.
CapyFeistelCipher CapyFeistelCipherCreate(CapyFeistelCipherMode const opMode) {

  // Create the CapyFeistelCipher
  CapyFeistelCipher that = {
    .counter = 0,
    .initVector = NULL,
    .opMode = opMode,
    .destruct = Destruct,
    .setInitVector = SetInitVector,
  };

  // Set the ciphering/deciphering method according to the operation mode
  if(opMode == capyFeistelCipher_ECB) {
    that.cipher = CipherECB;
    that.decipher = DecipherECB;
  } else if(opMode == capyFeistelCipher_CBC) {
    that.cipher = CipherCBC;
    that.decipher = DecipherCBC;
  } else if(opMode == capyFeistelCipher_CTR) {
    that.cipher = CipherCTR;
    that.decipher = DecipherCTR;
  } else {
    raiseExc(CapyExc_UndefinedExecution);
    return that;
  }

  // Return the new CapyFeistelCipher
  return that;
}

// Allocate memory for a new CapyFeistelCipher and create it
// Input:
//   opMode: operation mode
// Output:
//   Return a CapyFeistelCipher
// Exception:
//   May raise CapyExc_MallocFailed, CapyExc_UndefinedExecution.
CapyFeistelCipher* CapyFeistelCipherAlloc(CapyFeistelCipherMode const opMode) {
  CapyFeistelCipher* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;

  // Create the CapyFeistelCipher
  *that = CapyFeistelCipherCreate(opMode);

  // Return the CapyFeistelCipher
  return that;
}

// Free the memory used by a CapyFeistelCipher* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyFeistelCipher to free
void CapyFeistelCipherFree(CapyFeistelCipher** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
