// ------------------------------- feistelCipher.h ----------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_FEISTEL_CIPHER_H
#define CAPY_FEISTEL_CIPHER_H
#include "externalHeaders.h"
#include "list.h"

// Description:
// Feistel ciphering class.

// Typedef for the keys
typedef CapyArrChar CipherData_t;

// Typedef for the list of keys
typedef CapyListArrChar CipherKeys_t;

// Definition of the round function parent class. Round functions passed to
// CapyFeistelCipher methods must inherits this class. It has a dummy 'run'
// method for test purpose (which simply xor its input with the key). The
// inheriting class should replace it with its own round function in its
// constructor. The 'run' method is called by CapyFeistelCipher at each round
// with arguments:
//        key: a null terminated string, the key to be used by the round
//             function
//      block: the block of data to which apply the round function
//   sizeData: the size in byte of the block of data
//     output: the adress in memory where to write the output of the round
//             function (must be already allocated and have same size as
//             'block')
#define CapyFeistelRoundFunDef struct { \
  void (*destruct)(void);             \
  void (* run)(                         \
    CipherData_t const* const key,      \
    CipherData_t const* const block,    \
          CipherData_t* const output);  \
}

// Round function parent class
typedef CapyFeistelRoundFunDef CapyFeistelRoundFun;

// Create a CapyFeistelRoundFun
// Output:
//   Return a CapyFeistelRoundFun
CapyFeistelRoundFun CapyFeistelRoundFunCreate(void);

// Operation modes of the Feistel network
typedef enum CapyFeistelCipherMode {
  capyFeistelCipher_ECB,
  capyFeistelCipher_CBC,
  capyFeistelCipher_CTR,
} CapyFeistelCipherMode;

// Type of the counter for CTR mode
typedef uint64_t CipherCounter_t;

// FeistelCipher object
typedef struct CapyFeistelCipher {

  // Operation mode
  CapyFeistelCipherMode opMode;
  CapyPad(CapyFeistelCipherMode, opMode);

  // Initialisation vector
  CipherData_t* initVector;

  // Counter for CTR mode
  CipherCounter_t counter;

  // Destructor
  void (*destruct)(void);

  // Cipher function
  // Input:
  //   roundFun: the round function used to cipher
  //       keys: the keys used to cipher, a list of null terminated strings
  //       data: the data to cipher
  // Output:
  //   Return a newly allocated array of 'sizeData' bytes containing the
  //   ciphered data.
  // Exceptions:
  //   May raise CapyExc_MallocFailed
  CipherData_t* (*cipher)(
    CapyFeistelRoundFun* const roundFun,
     CipherKeys_t const* const keys,
     CipherData_t const* const data);

  // Decipher function
  // Input:
  //   roundFun: the round function used to decipher
  //       keys: the keys used to decipher, a list of null terminated strings
  //       data: the data to decipher
  // Output:
  //   Return a newly allocated array of 'sizeData' bytes containing the
  //   deciphered data.
  // Exceptions:
  //   May raise CapyExc_MallocFailed
  CipherData_t* (*decipher)(
    CapyFeistelRoundFun* const roundFun,
     CipherKeys_t const* const keys,
     CipherData_t const* const data);

  // Set the initialisation vector for CBC and CTR mode
  // Input:
  //   initVector, the initialisation vector
  // Exceptions:
  //   May raise CapyExc_MallocFailed
  void (*setInitVector)(CipherData_t const* const initVector);
} CapyFeistelCipher;

// Create a CapyFeistelCipher
// Input:
//   opMode: operation mode
// Output:
//   Return a CapyFeistelCipher
// Exception:
//   May raise CapyExc_MallocFailed, CapyExc_UndefinedExecution.
CapyFeistelCipher CapyFeistelCipherCreate(CapyFeistelCipherMode const opMode);

// Allocate memory for a new CapyFeistelCipher and create it
// Input:
//   opMode: operation mode
// Output:
//   Return a CapyFeistelCipher
// Exception:
//   May raise CapyExc_MallocFailed, CapyExc_UndefinedExecution.
CapyFeistelCipher* CapyFeistelCipherAlloc(CapyFeistelCipherMode const opMode);

// Free the memory used by a CapyFeistelCipher* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyFeistelCipher to free
void CapyFeistelCipherFree(CapyFeistelCipher** const that);
#endif
