// ---------------------------------- fft.h ---------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_FFT_H
#define CAPY_FFT_H
#include "externalHeaders.h"
#include "cext.h"
#include "polynomial.h"
#include "capymath.h"
#include "image.h"

// Description:
// Class and structure implementing DFT and FFT.

// DFT coefficients
typedef struct CapyDFTCoeffs {

  // Inherits CapyMathFun
  struct CapyMathFunDef;

  // Number of coefficients
  size_t nb;

  // Original input range of the transformed function (default: [0,1])
  CapyRangeDouble range;

  // Values of the Fourier series coefficients, these are the amplitudes of the
  // cosine (real part) and sine (imaginary part) for each "frequency bin". The
  // k-th frequency bin corresponds to the sine and cosine of frequency k. The
  // original function is then approximated by
  // f(t)=a_0/2+sum_{k=0..(N-1)}(a_k*cos(2pi*kt)+b_k*sin(2pi*kt))
  double complex* vals;

  // Destructor
  void (*destructCapyMathFun)(void);

  // Get the vector of amplitude (magnitude of the complex number) per
  // frequency bin (aka spectrum plot).
  // Inputs:
  //   fold: if true, fold the symetric frequency and return only half of
  //         the bins.
  // Output:
  //   Return the amplitude per frequency bins as a vector. It is symetric
  //   relative to that->nb/2. One can get the single sided Fourier coeficients
  //   by taking only the first half values multiplied by two.
  CapyVec (*getAmpFreqBins)(bool const fold);
} CapyDFTCoeffs;

// Create a CapyDFTCoeffs
// Input:
//   nb: the number of coefficients
// Output:
//   Return a CapyDFTCoeffs
CapyDFTCoeffs CapyDFTCoeffsCreate(size_t const nb);

// Allocate memory for a new CapyDFTCoeffs and create it
// Input:
//   nb: the number of coefficients
// Output:
//   Return a CapyDFTCoeffs
CapyDFTCoeffs* CapyDFTCoeffsAlloc(size_t const nb);

// Free the memory used by a CapyDFTCoeffs* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyDFTCoeffs to free
void CapyDFTCoeffsFree(CapyDFTCoeffs** const that);

// DFT class
typedef struct CapyDFT {

  // Destructor
  void (*destruct)(void);

  // Transform a 1D polynomial from coefficients representation to values
  // representation. The polynomial must have a power-of-2 number of
  // coefficients.
  // Input:
  //     poly: the polynomial
  // Output:
  //   Return the value representation of the polynomial as a CapyDFTCoeffs
  CapyDFTCoeffs (*fftPolyFromCoeffToVal)(CapyPolynomial1D const* const poly);

  // Transform a 1D polynomial from values representation to coefficients
  // representation. The number of values must be a power-of-2.
  // Input:
  //   values: the values representation of the polynomial
  // Output:
  //   Return the polynomial corresponding to the value representation
  CapyPolynomial1D* (*fftPolyFromValToCoeff)(CapyDFTCoeffs const* const values);

  // Calculate the Discrete Fourier Transform for an input range of a given
  // function
  // Input:
  //   fun: the function (must have one input and one output)
  //   range: range of the input of the function for which the DFT is calculated
  //   nbSample: the number of samples taken from the function, equally spaced
  //             in 'range' (bounds included)
  // Output:
  //   Return the DFT coefficients as a CapyDFTCoeffs
  CapyDFTCoeffs (*fftFun)(
              CapyMathFun* const fun,
    CapyRangeDouble const* const range,
                    size_t const nbSample);

  // Calculate the Discrete Fourier Transform for a given set of samples
  // Input:
  //   samples: the samples value
  // Output:
  //   Return the DFT coefficients as a CapyDFTCoeffs
  CapyDFTCoeffs (*fftSamples)(CapyVec const* const samples);
} CapyDFT;

// Create a CapyDFT
// Output:
//   Return a CapyDFT
CapyDFT CapyDFTCreate(void);

// Allocate memory for a new CapyDFT and create it
// Output:
//   Return a CapyDFT
// Exception:
//   May raise CapyExc_MallocFailed.
CapyDFT* CapyDFTAlloc(void);

// Free the memory used by a CapyDFT* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyDFT to free
void CapyDFTFree(CapyDFT** const that);

// DFT2D coefficients
typedef struct CapyDFT2DCoeffs {

  // Inherits CapyMathFun
  struct CapyMathFunDef;

  // Number of coefficients
  size_t nb[2];

  // Original input range of the transformed function (default: [0,1])
  CapyRangeDouble range[2];

  // Values of the Fourier series coefficients, stored by rows
  double complex* vals;

  // Destructor
  void (*destructCapyMathFun)(void);

  // Convert the coefficients into a visualisation of the amplitude
  // Inputs:
  //   center: if true, center the null frequency
  //   logScale: if true, apply logarithmic scaling
  // Output:
  //   Return a normalised greyscale image.
  CapyImg* (*toAmplitudeImg)(
    bool const center,
    bool const logScale);

  // Convert the coefficients into a visualisation of the phase
  // Inputs:
  //   center: if true, center the null frequency
  // Output:
  //   Return a normalised greyscale image.
  CapyImg* (*toPhaseImg)(bool const center);

  // Convert the coefficients into a periodogram
  // Inputs:
  //   center: if true, center the null frequency
  //   logScale: if true, apply logarithmic scaling
  // Output:
  //   Return a normalised greyscale image.
  CapyImg* (*toPeriodogramImg)(
    bool const center,
    bool const logScale);
} CapyDFT2DCoeffs;

// Create a CapyDFT2DCoeffs
// Input:
//   nb: the number of coefficients
// Output:
//   Return a CapyDFT2DCoeffs
CapyDFT2DCoeffs CapyDFT2DCoeffsCreate(size_t const nb[2]);

// Allocate memory for a new CapyDFT2DCoeffs and create it
// Input:
//   nb: the number of coefficients
// Output:
//   Return a CapyDFT2DCoeffs
CapyDFT2DCoeffs* CapyDFT2DCoeffsAlloc(size_t const nb[2]);

// Free the memory used by a CapyDFT2DCoeffs* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyDFT2DCoeffs to free
void CapyDFT2DCoeffsFree(CapyDFT2DCoeffs** const that);

// DFT2D class
typedef struct CapyDFT2D {

  // Destructor
  void (*destruct)(void);

  // Calculate the 2D Discrete Fourier Transform for a given image
  // Input:
  //   img: the img to transform
  // Output:
  //   Return the DFT coefficients as a CapyDFT2DCoeffs
  CapyDFT2DCoeffs (*fftImage)(CapyImg const* const img);
} CapyDFT2D;

// Create a CapyDFT2D
// Output:
//   Return a CapyDFT2D
CapyDFT2D CapyDFT2DCreate(void);

// Allocate memory for a new CapyDFT2D and create it
// Output:
//   Return a CapyDFT2D
// Exception:
//   May raise CapyExc_MallocFailed.
CapyDFT2D* CapyDFT2DAlloc(void);

// Free the memory used by a CapyDFT2D* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyDFT2D to free
void CapyDFT2DFree(CapyDFT2D** const that);
#endif
