// ------------------- fileFormat.h --------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_FILEFORMAT_H
#define CAPY_FILEFORMAT_H
#include "externalHeaders.h"
#include "cext.h"
#include "streamIo.h"
#include "image.h"
#include "pointCloud.h"

// Description:
// Virtual parent class for all file format support.

// Supported file formats
typedef enum CapyFileFormatType {
  capyFileFormat_png,
  capyFileFormat_ply,
  capyFileFormat_nb,
} CapyFileFormatType;

// Labels for supported file format
extern char const* capyFileFormatLbl[capyFileFormat_nb];

// CapyFileFormat class definition macro.
// Type of the format
// CapyFileFormatType type;
// Human readable type
// char const* lbl;
// Given an object of type <T>,
// Load an instance of the object from a file
// Input:
//   file: the file (as an opened CapyStreamIo)
// Output:
//   Return the instance of the object.
// Exceptions:
//   May raise CapyExc_StreamReadError, CapyExc_UnsupportedFormat,
//   CapyExc_InvalidStream
// <T>* (*load<T>)(CapyStreamIo* const file);
// Save an instance of the object to a file
// Input:
//    obj: the instance to save
//   file: the file (as an opened CapyStreamIo)
// Exceptions:
//   May raise CapyExc_StreamWriteError, CapyExc_UnsupportedFormat
// void (*save<T>)(
//      <T> const* const obj,
//   CapyStreamIo* const file);
#define CapyFileFormatDef struct {                             \
  CapyFileFormatType type;                                     \
  CapyPad(CapyFileFormatType, 0);                              \
  char const* lbl;                                             \
  void (*destruct)(void);                                      \
  CapyImg* (*loadImg)(CapyStreamIo* const file);               \
  void (*saveImg)(                                             \
    CapyImg const* const img,                                  \
     CapyStreamIo* const file);                                \
  CapyPointCloud* (*loadPointCloud)(CapyStreamIo* const file); \
  void (*savePointCloud)(                                      \
    CapyPointCloud const* const pointCloud,                    \
     CapyStreamIo* const file);                                \
}

// CapyFileFormat object
typedef CapyFileFormatDef CapyFileFormat;

// Create a CapyFileFormat
// Input:
//   type: the type of the format
// Output:
//   Return a CapyFileFormat
CapyFileFormat CapyFileFormatCreate(CapyFileFormatType const type);
#endif
