// ------------------------------ floodFill.c ------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#include "floodFill.h"

// Segment an image by flood filling it.
// Input:
//           imgSrc: the image to segment
//          imgCond: the image one which the flooding condition is checked
//   floodCondition: the flood condition function to control flooding
// Output:
//   Return a CapyColorPatches, one CapyColorPatch per segment in the image
//   after flood filling.
static CapyColorPatches* SegmentImg(
      CapyImg const* const imgSrc,
      CapyImg const* const imgCond,
  CapyFloodConditionImgFun floodCondition) {

  // Variable to memorise the result
  CapyColorPatches* segments = CapyColorPatchesAlloc();

  // Variable to memorise the segmented pixels
  size_t* labels = NULL;
  size_t nbPixels = $(imgSrc, getNbPixels)();
  safeMalloc(labels, nbPixels);
  if(labels) loop(i, nbPixels) labels[i] = 0;

  // Variable to memorise the next segment index
  size_t iLabel = 1;

  // Queue of pixels to memorise the pixels to flood
  size_t sizeQueue = imgCond->dims.width;
  CapyImgPixel* queue;
  safeMalloc(queue, sizeQueue);
  if(!queue) return NULL;
  size_t endQueue = 0;

  // Loop on the pixels not already tagged
  forEach(pixel, ((CapyImg*)imgCond)->iter) if(labels[pixel.idx] == 0) {

    // Tag the pixel
    labels[pixel.idx] = iLabel;

    // Create a color patch
    size_t nbPixelPatch = 0;
    CapyColorPatch patch = CapyColorPatchCreate();
    size_t nbMaxPixelPatch = 100;
    $(&(patch.pixels), resize)(nbMaxPixelPatch);
    patch.idx = iLabel;

    // Add the pixel to the queue of pixels to flood
    queue[endQueue] = pixel;
    ++endQueue;

    // Loop until there is no more pixel to tag in the queue
    while(endQueue > 0) {

      // Get the next pixel to tag
      endQueue--;
      CapyImgPixel pixelToTag = queue[endQueue];

      // Add the corresponding pixel in the source image to the color patch
      CapyImgPixel imgSrcPixel = pixelToTag;
      imgSrcPixel.color = imgSrc->pixels + pixelToTag.idx;
      if(nbPixelPatch >= nbMaxPixelPatch) {
        nbMaxPixelPatch *= 2;
        $(&(patch.pixels), resize)(nbMaxPixelPatch);
      }
      $(&(patch.pixels), set)(nbPixelPatch, &imgSrcPixel);
      ++nbPixelPatch;

      // Loop on its neighbours
      loop(iDir, 4) {
        CapyImgPixel neighbour = $(imgCond, getNeighbour)(&pixelToTag, iDir);

        // If the neighbour exists, is not already tagged and respect the
        // flooding condition
        if(
          neighbour.color != NULL &&
          labels[neighbour.idx] == 0 &&
          floodCondition(imgCond, &pixelToTag, &neighbour)
        ) {

          // Tag the neighbour and add it to the queue of pixels to flood
          labels[neighbour.idx] = iLabel;
          if(endQueue == sizeQueue) {
            sizeQueue *= 2;
            safeRealloc(queue, sizeQueue);
          }
          queue[endQueue] = neighbour;
          ++endQueue;
        }
      }
    }

    // Add the patch to the segments and increase the label index
    $(&(patch.pixels), resize)(nbPixelPatch);
    $(segments->list, add)(patch);
    ++iLabel;
  }

  // Free memory
  free(queue);
  free(labels);

  // Return the segments
  return segments;
}

// Free the memory used by a CapyFloodFill
static void Destruct(void) {
  return;
}

// Create a CapyFloodFill
// Output:
//   Return a CapyFloodFill
CapyFloodFill CapyFloodFillCreate(void) {
  return (CapyFloodFill){
    .destruct = Destruct,
    .segmentImg = SegmentImg,
  };
}

// Allocate memory for a new CapyFloodFill and create it
// Output:
//   Return a CapyFloodFill
// Exception:
//   May raise CapyExc_MallocFailed.
CapyFloodFill* CapyFloodFillAlloc(void) {
  CapyFloodFill* that = NULL;
  safeMalloc(that, 1);
  if(!that) return NULL;
  *that = CapyFloodFillCreate();
  return that;
}

// Free the memory used by a CapyFloodFill* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyFloodFill to free
void CapyFloodFillFree(CapyFloodFill** const that) {
  if(that == NULL || *that == NULL) return;
  $(*that, destruct)();
  free(*that);
  *that = NULL;
}
