// -------------------------------- floodFill.h -------------------------------
/*
    LibCapy - a general purpose library of C functions and data structures
    Copyright (C) 2021-2025 Pascal Baillehache baillehache.pascal@gmail.com
    https://baillehachepascal.dev
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef CAPY_FLOODFILL_H
#define CAPY_FLOODFILL_H
#include "externalHeaders.h"
#include "image.h"
#include "colorPatch.h"

// Description:
// Flood fill algorithm.

// Function interface for a flooding condition on image
// Input:
//    img: the image under flooding
//   from: the flooded pixel
//     to: the neighbour candidate pixel
// Output:
//   Return true if the flooding propagates from the flooded pixel to the
//   neighbour candidate pixel, else false.
typedef bool (*CapyFloodConditionImgFun)(
  CapyImg const* const img,
  CapyImgPixel const* const from,
  CapyImgPixel const* const to);

// FloodFill object
typedef struct CapyFloodFill {

  // Destructor
  void (*destruct)(void);

  // Segment an image by flood filling it.
  // Input:
  //           imgSrc: the image to segment
  //          imgCond: the image one which the flooding condition is checked
  //   floodCondition: the flood condition function to control flooding
  // Output:
  //   Return a CapyColorPatches, one CapyColorPatch per segment in the image
  //   after flood filling.
  CapyColorPatches* (*segmentImg)(
        CapyImg const* const imgSrc,
        CapyImg const* const imgCond,
    CapyFloodConditionImgFun floodCondition);
} CapyFloodFill;

// Create a CapyFloodFill
// Output:
//   Return a CapyFloodFill
CapyFloodFill CapyFloodFillCreate(void);

// Allocate memory for a new CapyFloodFill and create it
// Output:
//   Return a CapyFloodFill
// Exception:
//   May raise CapyExc_MallocFailed.
CapyFloodFill* CapyFloodFillAlloc(void);

// Free the memory used by a CapyFloodFill* and reset '*that' to NULL
// Input:
//   that: a pointer to the CapyFloodFill to free
void CapyFloodFillFree(CapyFloodFill** const that);
#endif
